# This Source Code Form is subject to the terms of the Mozilla Public
# License, v. 2.0. If a copy of the MPL was not distributed with this
# file, You can obtain one at http://mozilla.org/MPL/2.0/.

import buildconfig
import mozpack.path as mozpath
import os
import subprocess
import string
import sys

SERVO_BASE = mozpath.join(buildconfig.topsrcdir, "servo")
SERVO_PROP_BASE = mozpath.join(SERVO_BASE, "components", "style", "properties")


def generate_data(output, template):
    output.write("# THIS IS AN AUTOGENERATED FILE.  DO NOT EDIT\n\n")
    output.write(subprocess.check_output([
        sys.executable,
        mozpath.join(SERVO_PROP_BASE, "build.py"),
        "gecko", "geckolib", template
    ], universal_newlines=True))

    # Add all relevant files into the dependencies of the generated file.
    DEP_EXTS = [".py", ".rs", ".zip"]
    deps = set()
    for path, dirs, files in os.walk(SERVO_PROP_BASE):
        for file in files:
            if os.path.splitext(file)[1] in DEP_EXTS:
                deps.add(mozpath.join(path, file))
    return deps


def generate_header(output, data):
    with open(data, "r") as f:
        data = eval(f.read())

    output.write("""/* THIS IS AN AUTOGENERATED FILE.  DO NOT EDIT */

/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at https://mozilla.org/MPL/2.0/. */

#define CSS_PROP_DOMPROP_PREFIXED(name_) \\
  CSS_PROP_PUBLIC_OR_PRIVATE(Moz ## name_, name_)

#ifndef CSS_PROP_LONGHAND
#define CSS_PROP_LONGHAND(name_, id_, method_, flags_, pref_) /* nothing */
#define DEFINED_CSS_PROP_LONGHAND
#endif

#ifndef CSS_PROP_SHORTHAND
#define CSS_PROP_SHORTHAND(name_, id_, method_, flags_, pref_) /* nothing */
#define DEFINED_CSS_PROP_SHORTHAND
#endif

#ifndef CSS_PROP_ALIAS
#define CSS_PROP_ALIAS(name_, aliasid_, id_, method_, pref_) /* nothing */
#define DEFINED_CSS_PROP_ALIAS
#endif

""")

    MACRO_NAMES = {
        "longhand": "CSS_PROP_LONGHAND",
        "shorthand": "CSS_PROP_SHORTHAND",
        "alias": "CSS_PROP_ALIAS",
    }
    for name, method, id, flags, pref, proptype in data:
        is_internal = "CSS_PROPERTY_INTERNAL" in flags
        pref = '"' + pref + '"'
        if proptype == "alias":
            params = [name, id[0], id[1], method, pref]
        else:
            if method == "CssFloat":
                method = "CSS_PROP_PUBLIC_OR_PRIVATE(CssFloat, Float)"
            elif method.startswith("Moz"):
                method = "CSS_PROP_DOMPROP_PREFIXED({})".format(method[3:])
            if flags:
                flags = " | ".join(flags)
            else:
                flags = "0"
            params = [name, id, method, flags, pref]

        if is_internal:
            output.write("#ifndef CSS_PROP_LIST_EXCLUDE_INTERNAL\n")
        output.write("{}({})\n".format(MACRO_NAMES[proptype], ", ".join(params)))
        if is_internal:
            output.write("#endif\n")

    output.write("""
#ifdef DEFINED_CSS_PROP_ALIAS
#undef CSS_PROP_ALIAS
#undef DEFINED_CSS_PROP_ALIAS
#endif

#ifdef DEFINED_CSS_PROP_SHORTHAND
#undef CSS_PROP_SHORTHAND
#undef DEFINED_CSS_PROP_SHORTHAND
#endif

#ifdef DEFINED_CSS_PROP_LONGHAND
#undef CSS_PROP_LONGHAND
#undef DEFINED_CSS_PROP_LONGHAND
#endif

#undef CSS_PROP_DOMPROP_PREFIXED
""")
