'use strict';

/* global PromiseStorage, LayoutDictionaryList, LayoutItem */

(function(exports) {

var LayoutItemList = function(app) {
  this.closeLockManager = app.closeLockManager;
  this.dictionaryList = null;
  this.dbStore = null;
  this._layoutConfigQueue = null;

  // This set stores the ids of enabled layouts.
  this._installedLayoutListSet = null;

  this.layoutItems = null;
};

LayoutItemList.prototype.DATABASE_NAME = 'imEngineData';
LayoutItemList.prototype.ENABLED_LAYOUT_KEY = '__installedLayouts';

// JSON file lists included layouts, generated by build script.
LayoutItemList.prototype.CONFIG_FILE_PATH = './js/settings/layouts.json';

LayoutItemList.prototype.onready = null;

LayoutItemList.prototype.start = function() {
  this.dbStore = new PromiseStorage(this.DATABASE_NAME);
  this.dbStore.start();

  this.dictionaryList = new LayoutDictionaryList(this);
  this.dictionaryList.start();

  this.layoutItems = new Map();

  var p = this._getConfig()
    .then(this._createLayoutItemsFromLayouts.bind(this))
    .then(function() {
      if (typeof this.onready === 'function') {
        this.onready();
      }
    }.bind(this));

  this._layoutConfigQueue = p
    .catch(function(e) {
      e && console.error(e);
    });

  // Return this promise in the start() function so unit test could catch the
  // errors. Note that we do not expect the promise to reject in production
  // so it's ok if the user of the class don't handle it.
  return p;
};

LayoutItemList.prototype.stop = function() {
  this.layoutItems.forEach(function(layoutItem) {
    layoutItem.stop();
  });

  this.dbStore.stop();
  this.dbStore = null;

  this.dictionaryList.stop();
  this.dictionaryList = null;

  this._layoutConfigQueue = null;
  this._installedLayoutListSet = null;
};

LayoutItemList.prototype.setLayoutAsInstalled = function(layoutId) {
  var p = this._layoutConfigQueue
    .then(function() {
      this._installedLayoutListSet.add(layoutId);

      return this.dbStore.setItem(
        this.ENABLED_LAYOUT_KEY, Array.from(this._installedLayoutListSet));
    }.bind(this))
    .catch(function(e) {
      this._installedLayoutListSet.delete(layoutId);

      throw e;
    });

  this._layoutConfigQueue = p
    .catch(function(e) {
      e && console.error(e);
    });

  return p;
};

LayoutItemList.prototype.setLayoutAsUninstalled = function(layoutId) {
  var p = this._layoutConfigQueue
    .then(function() {
      this._installedLayoutListSet.delete(layoutId);

      return this.dbStore.setItem(
        this.ENABLED_LAYOUT_KEY, Array.from(this._installedLayoutListSet));
    }.bind(this))
    .catch(function(e) {
      this._installedLayoutListSet.add(layoutId);

      throw e;
    });

  this._layoutConfigQueue = p
    .catch(function(e) {
      e && console.error(e);
    });

  return p;
};

LayoutItemList.prototype._getConfig = function() {
  var xhrPromise = new Promise(function(resolve, reject) {
    var xhr = new XMLHttpRequest();
    xhr.open('GET', this.CONFIG_FILE_PATH);
    xhr.responseType = 'json';
    xhr.onload = function() {
      if (xhr.response) {
        resolve(xhr.response);
      } else {
        reject();
      }
    };
    xhr.onerror = function() {
      reject();
    };
    xhr.send();

  }.bind(this));

  var installedLayoutsPromise = this.dbStore.getItem(this.ENABLED_LAYOUT_KEY);

  var p = Promise.all([xhrPromise, installedLayoutsPromise])
    .then(function(values) {
      var layouts = values[0];
      var installedLayoutListSet =
        this._installedLayoutListSet = new Set(values[1] || []);

      layouts.forEach(function(layout) {
        layout.installed =
          layout.preloaded || installedLayoutListSet.has(layout.id);
      });

      return layouts;
    }.bind(this));

  return p;
};

LayoutItemList.prototype._createLayoutItemsFromLayouts = function(layouts) {
  var needDownload = layouts.some(function(layout) {
    return (layout.preloaded === false);
  });

  // If all the layouts are preloaded,
  // we don't really need to show the list and enable the feature.
  if (!needDownload) {
    return;
  }

  this.dictionaryList.createDictionariesFromLayouts(layouts);

  layouts.forEach(function(layout) {
    var layoutItem = new LayoutItem(this, layout);
    layoutItem.start();

    this.layoutItems.set(layout.id, layoutItem);
  }, this);
};

exports.LayoutItemList = LayoutItemList;

}(window));
