
/**
 * AppStorage is a singleton that caches app storage values for
 * app storage and root panel fast access
 */
define('modules/app_storage',['require','modules/mvvm/observable'],function(require) {
  

  var Observable = require('modules/mvvm/observable');

  var AppStorage = function() {
    this._enabled = false;
    this._appStorage = navigator.getDeviceStorage('apps');

    this.storage = Observable({
      usedPercentage: 0,
      totalSize: 0,
      usedSize: 0,
      freeSize: 0
    });
  };

  AppStorage.prototype = {
    /**
     * The value indicates whether the module is responding. If it is false, the
     * UI stops reflecting the updates from the app storage.
     *
     * @access public
     * @memberOf AppStorage.prototype
     * @type {Boolean}
     */
    get enabled() {
      return this._enabled;
    },

    set enabled(value) {
      // early return if the value is not changed
      if (this._enabled === value) {
        return;
      } else {
        this._enabled = value;
      }
      if (value) {
        this._attachListeners();
        this._getSpaceInfo();
      } else {
        this._detachListeners();
      }
    },

    _attachListeners: function as_attachListeners() {
      this._appStorage.addEventListener('change', this);
    },

    _detachListeners: function as_detachListeners() {
      this._appStorage.removeEventListener('change', this);
    },

    handleEvent: function as_handler(evt) {
      switch (evt.type) {
        case 'change':
          this._getSpaceInfo();
          break;
      }
    },

    _getSpaceInfo: function as_getSpaceInfo() {
      var deviceStorage = this._appStorage;

      if (!deviceStorage) {
        console.error('Cannot get DeviceStorage for: app');
        return;
      }
      deviceStorage.freeSpace().onsuccess = function(e) {
        this.storage.freeSize = e.target.result;
        deviceStorage.usedSpace().onsuccess = function(e) {
          this.storage.usedSize = e.target.result;
          // calculate the percentage to show a space usage bar
          this.storage.totalSize =
            this.storage.usedSize + this.storage.freeSize;
          var usedPercentage = (this.storage.totalSize === 0) ?
            0 : (this.storage.usedSize * 100 / this.storage.totalSize);
          if (usedPercentage > 100) {
            usedPercentage = 100;
          }
          this.storage.usedPercentage = usedPercentage;
        }.bind(this);
      }.bind(this);
    }
  };

  // return singleton
  var instance = new AppStorage();
  instance.enabled = true;
  return instance;
});


/* global DeviceStorageHelper */
/**
 * Used to show Storage/App Storage panel
 */
define('panels/app_storage/panel',['require','modules/settings_panel','modules/app_storage'],function(require) {
  

  var SettingsPanel = require('modules/settings_panel');
  var AppStorage = require('modules/app_storage');

  return function ctor_app_storage_panel() {
    var _spaceBarElement = null;
    var _totalSpaceText = null;
    var _usedSpaceText = null;
    var _freeSpaceText = null;

    var _updateUsePercentage = function() {
      _spaceBarElement.value = AppStorage.storage.usedPercentage;
    };

    var _refreshText = function(element, size) {
      DeviceStorageHelper.showFormatedSize(element,
        'storageSize', size);
    };
    var _updateTotalSize = function() {
      _refreshText(_totalSpaceText, AppStorage.storage.totalSize);
    };
    var _updateUsedSize = function() {
      _refreshText(_usedSpaceText, AppStorage.storage.usedSize);
    };
    var _updateFreeSize = function() {
      _refreshText(_freeSpaceText, AppStorage.storage.freeSize);
    };

    var _updateElements = function() {
      _updateUsePercentage();
      _updateTotalSize();
      _updateUsedSize();
      _updateFreeSize();
    };

    return SettingsPanel({
      onInit: function(panel) {
        _spaceBarElement = panel.querySelector('.apps-space-bar');
        _totalSpaceText = panel.querySelector('.apps-total-space');
        _usedSpaceText = panel.querySelector('.apps-used-space');
        _freeSpaceText = panel.querySelector('.apps-free-space');
      },

      onBeforeShow: function() {
        AppStorage.storage.observe('usedPercentage', _updateUsePercentage);
        AppStorage.storage.observe('totalSize', _updateTotalSize);
        AppStorage.storage.observe('usedSize', _updateUsedSize);
        AppStorage.storage.observe('freeSize', _updateFreeSize);

        _updateElements();
      },

      onHide: function() {
        AppStorage.storage.unobserve('usedPercentage', _updateUsePercentage);
        AppStorage.storage.unobserve('totalSize', _updateTotalSize);
        AppStorage.storage.unobserve('usedSize', _updateUsedSize);
        AppStorage.storage.unobserve('freeSize', _updateFreeSize);
      }
    });
  };
});
