
/**
 * This is a factory method that returns a template function that is able to
 * render nested lists. The constructor takes two template functions. One for
 * the first level item and one for the second level item. The created inner
 * list views is exposed via the listViews property.
 *
 * @module keyboard_add_layouts/nested_template_factory
 */
define('panels/keyboard_add_layouts/nested_template_factory',['require'],function(require) {
  

  return function ctor_nestedTemplate(parentTemplate, childTemplate) {
    var listViews = [];
    var template = parentTemplate.bind({
      listViews: listViews,
      childTemplate: childTemplate
    });

    // Expose the list views.
    Object.defineProperty(template, 'listViews', {
      configurable: false,
      get: function() {
        return listViews;
      }
    });

    return template;
  };
});

/**
 * The template function for generating an UI element for a keyboard object.
 *
 * @module keyboard_add_layouts/keyboard_template
 */
define('panels/keyboard_add_layouts/keyboard_template',['require','modules/mvvm/list_view'],function(require) {
  

  var ListView = require('modules/mvvm/list_view');

  return function kal_keyboardTemplate(keyboard, recycled, helper) {
    // This function is served as a parent template, we expected to get the
    // childTemplate and all generated list views.
    // XXX: we need a better way the recycle and reuse list view objects.
    var layoutTemplate = this.childTemplate;
    var listViews = this.listViews;

    var container, header, h2, ul, listView;
    if (recycled) {
      container = recycled;
      h2 = container.querySelector('h2');
      ul = container.querySelector('ul');
    } else {
      container = document.createElement('div');
      header = document.createElement('header');
      h2 = document.createElement('h2');
      ul = document.createElement('ul');
      header.appendChild(h2);
      container.appendChild(header);
      container.appendChild(ul);
    }

    // if we find a listView for the ul, reuse it, otherwise create one
    listView = listViews.some(function eachListView(list) {
      if (list.element === ul) {
        list.set(keyboard.layouts);
        list.enabled = true;
        return true;
      }
    });

    if (!listView) {
      listView = ListView(ul, keyboard.layouts, layoutTemplate);
      listView.enabled = true;
      listViews.push(listView);
    }

    helper.observeAndCall(keyboard, {
      name: function refreshName() {
        h2.textContent = keyboard.name;
      }
    });

    return container;
  };
});

/**
 * The template function for generating an UI element for a layout object.
 *
 * @module keyboard_add_layouts/layout_template
 */
define('panels/keyboard_add_layouts/layout_template',['require'],function(require) {
  

  return function kal_layoutTemplate(layout, recycled, helper) {
    var container = null;
    var span, checkbox;
    if (recycled) {
      container = recycled;
      checkbox = container.querySelector('input');
      span = container.querySelector('span');
    } else {
      container = document.createElement('li');
      checkbox = document.createElement('input');
      var label = document.createElement('label');
      span = document.createElement('span');

      label.className = 'pack-checkbox';
      checkbox.type = 'checkbox';

      label.appendChild(checkbox);
      label.appendChild(span);

      container.appendChild(label);
    }

    checkbox.onchange = function() {
      layout.enabled = this.checked;
    };

    helper.observeAndCall(layout, {
      name: function refreshName() {
        span.textContent = layout.name;
      },
      enabled: function() {
        checkbox.checked = layout.enabled;
      }
    });

    return container;
  };
});

/**
 * The module initializes a ListView displaying all installed layouts.
 * Implementation details please refer to {@link KeyboardAddLayoutsCore}.
 *
 * @module keyboard_add_layouts/core
 */
define('panels/keyboard_add_layouts/core',['require','modules/settings_service','modules/mvvm/list_view'],function(require) {
  

  var SettingsService = require('modules/settings_service');
  var ListView = require('modules/mvvm/list_view');

  /**
   * @alias module:keyboard_add_layouts/core
   * @class KeyboardAddLayoutsCore
   * @requires module:modules/settings_service
   * @requires module:modules/mvvm/list_view
   * @param {KeyboardContext} context
                              The kyboard context providing installed keyboards.
   * @param {Function} template
                       The template function used to render an installed
                       keyboard.
   * @returns {KeyboardAddLayoutsCore}
   */
  function KeyboardAddLayoutsCore(context, template) {
    this._enabled = false;
    this._listView = null;
    this._keyboardContext = context;
    this._keyboardTemplate = template;
  }

  KeyboardAddLayoutsCore.prototype = {
    /**
     * The value indicates whether the module is responding. If it is false, the
     * UI stops reflecting the updates from the keyboard context.
     *
     * @access public
     * @memberOf KeyboardAddLayoutsCore.prototype
     * @type {Boolean}
     */
    get enabled() {
      return this._enabled;
    },

    set enabled(value) {
      this._enabled = value;
      if (this._listView) {
        this._listView.enabled = this._enabled;
      }
      // Disable all inner list views
      this._keyboardTemplate.listViews.forEach(function(listView) {
        listView.enabled = this._enabled;
      }.bind(this));
    },

    /**
     * @access private
     * @memberOf KeyboardAddLayoutsCore.prototype
     * @param {HTMLElement} listViewRoot
     * @param {ObservableArray} keyboards
     * @param {Function} keyboardTemplate
     */
    _initInstalledLayoutListView:
      function kal_initListView(listViewRoot, keyboards, keyboardTemplate) {
        this._listView = ListView(listViewRoot, keyboards, keyboardTemplate);
    },

    /**
     * The handler is invoked when users disable the must-have input type. In
     * the handler we navigate to the dialog.
     *
     * @access private
     * @memberOf KeyboardAddLayoutsCore.prototype
     * @param {Object} layout
     * @param {String} missingType
     */
    _showEnabledDefaultDialog: function kal_showDialog(layout, missingType) {
      SettingsService.navigate('keyboard-enabled-default', {
        layout: layout,
        origin: 'keyboard-selection-addMore',
        missingType: missingType
      });
    },

    /**
     * @access public
     * @memberOf KeyboardAddLayoutsCore.prototype
     * @param {Array} elements
     *                Elements needed by this module.
     * @param {HTMLElement} elements.listViewRoot
     *                      The root element for the list view displaying the
     *                      installed keyboards.
     */
    init: function kal_onInit(elements) {
      var that = this;
      this._keyboardContext.init(function() {
        that._keyboardContext.keyboards(function(keyboards) {
          that._initInstalledLayoutListView(
            elements.listViewRoot, keyboards, that._keyboardTemplate);
          that.enabled = true;
        });
      });

      this._keyboardContext.defaultKeyboardEnabled(
        this._showEnabledDefaultDialog);
    }
  };

  return function ctor_kalCore(context, template) {
    return new KeyboardAddLayoutsCore(context, template);
  };
});

define('panels/keyboard_add_layouts/panel',['require','modules/settings_panel','modules/keyboard_context','shared/keyboard_helper','panels/keyboard_add_layouts/nested_template_factory','panels/keyboard_add_layouts/keyboard_template','panels/keyboard_add_layouts/layout_template','panels/keyboard_add_layouts/core'],function(require) {
  

  var SettingsPanel = require('modules/settings_panel');
  var KeyboardContext = require('modules/keyboard_context');
  var KeyboardHelper = require('shared/keyboard_helper');
  var NestedTemplateFactory =
    require('panels/keyboard_add_layouts/nested_template_factory');
  var keyboardTemplate =
    require('panels/keyboard_add_layouts/keyboard_template');
  var layoutTemplate = require('panels/keyboard_add_layouts/layout_template');
  var Core = require('panels/keyboard_add_layouts/core');

  return function ctor_addLayoutsPanel() {
    var nestedTemplate =
      NestedTemplateFactory(keyboardTemplate, layoutTemplate);
    var core = Core(KeyboardContext, nestedTemplate);

    return SettingsPanel({
      onInit: function kalp_onInit(rootElement) {
        core.init({
          listViewRoot: rootElement.querySelector('.keyboardAppContainer')
        });
      },
      onBeforeShow: function kalp_onBeforeShow() {
        core.enabled = true;
      },
      onBeforeHide: function kalp_onBeforeHide() {
        // save changes to settings
        KeyboardHelper.saveToSettings();
      },
      onHide: function kalp_onHide() {
        core.enabled = false;
      }
    });
  };
});
