
/**
 * WifiWps is a module that stores some functions that would be called
 * when doing wps related operations
 *
 * @module wifi_wps/wifi_wps
 */
define('panels/wifi_wps/wifi_wps',['require'],function(require) {
  

  var WifiWps = function() {
    return {
      /**
       * To make sure wps pin is valid or not
       * @param {String} pin - value of pin
       * @returns {Boolean}
       */
      _isValidWpsPin: function(pin) {
        if (pin.match(/[^0-9]+/)) {
          return false;
        }
        if (pin.length === 4) {
          return true;
        }
        if (pin.length !== 8) {
          return false;
        }
        var num = pin - 0;
        return this._pinChecksum(Math.floor(num / 10)) === (num % 10);
      },
      /**
       * This is an inner function that we can use it to get
       * pin's checksum
       *
       * @param {Number} pin - value of pin
       * @returns {Number}
       */
      _pinChecksum: function(pin) {
        var accum = 0;
        while (pin > 0) {
          accum += 3 * (pin % 10);
          pin = Math.floor(pin / 10);
          accum += pin % 10;
          pin = Math.floor(pin / 10);
        }
        return (10 - accum % 10) % 10;
      }
    };
  };

  return WifiWps;
});

define('panels/wifi_wps/panel',['require','modules/settings_panel','modules/wifi_context','panels/wifi_wps/wifi_wps'],function(require) {
  

  var SettingsPanel = require('modules/settings_panel');
  var WifiContext = require('modules/wifi_context');
  var WifiWps = require('panels/wifi_wps/wifi_wps');

  return function ctor_wpsWifi() {
    var wifiWps = WifiWps();
    var elements = {};

    return SettingsPanel({
      onInit: function(panel) {
        elements.panel = panel;
        elements.submitWpsButton = panel.querySelector('button[type=submit]');
        elements.apSelectionArea = panel.querySelector('.wifi-wps-pin-aps');
        elements.apSelect = elements.apSelectionArea.querySelector('select');
        elements.pinItem = panel.querySelector('.wifi-wps-pin-area');
        elements.pinDesc = elements.pinItem.querySelector('p');
        elements.pinInput = elements.pinItem.querySelector('input');
        elements.wpsMethodRadios = panel.querySelectorAll('input[type=radio]');

        // Check validWpsPin each time when typing
        elements.pinInput.oninput = function() {
          elements.submitWpsButton.disabled =
            !wifiWps._isValidWpsPin(elements.pinInput.value);
        };

        for (var i = 0; i < elements.wpsMethodRadios.length; i++) {
          elements.wpsMethodRadios[i].onchange = this._onWpsMethodChange;
        }
        this._onWpsMethodChange();
      },
      onBeforeShow: function(panel, options) {
        this._cleanupApList();
        this._updateApList(options.wpsAvailableNetworks);
      },
      onBeforeHide: function() {
        // Store information on the context to make them accessible from
        // other panels.
        WifiContext.wpsOptions.selectedAp = elements.apSelect.options[
          elements.apSelect.selectedIndex].value;

        WifiContext.wpsOptions.selectedMethod = elements.panel.querySelector(
          'input[type=\'radio\']:checked').value;

        WifiContext.wpsOptions.pin = elements.pinInput.value;
      },
      _cleanupApList: function() {
        var apSelect = elements.apSelect;
        while (apSelect.hasChildNodes()) {
          apSelect.removeChild(apSelect.firstChild);
        }
      },
      _updateApList: function(wpsAvailableNetworks) {
        // Add the first option
        var option = document.createElement('option');
        option.setAttribute('data-l10n-id', 'wpsAnyAp');
        option.value = 'any';
        elements.apSelect.appendChild(option);

        // Add the other networks
        for (var i = 0; i < wpsAvailableNetworks.length; i++) {
          option = document.createElement('option');
          option.textContent = wpsAvailableNetworks[i].ssid;
          option.value = wpsAvailableNetworks[i].bssid;
          elements.apSelect.appendChild(option);
        }
      },
      _onWpsMethodChange: function() {
        var method = elements.panel.querySelector(
          'input[type=\'radio\']:checked').value;

        if (method === 'apPin') {
          elements.submitWpsButton.disabled =
            !wifiWps._isValidWpsPin(elements.pinInput.value);
          elements.pinItem.hidden = false;
        } else {
          elements.submitWpsButton.disabled = false;
          elements.pinItem.hidden = true;
        }
        elements.apSelectionArea.hidden = method === 'pbc';
      }
    });
  };
});
