
requirejs.config({
  // waitSeconds is set to the default here; the build step rewrites
  // it to 0 in build/require_config.jslike so that we never timeout
  // waiting for modules in production. This is important when the
  // device is under super-low-memory stress, as it may take a while
  // for the device to get around to loading things like Clock's alarm
  // ringing screen, and we absolutely do not want that to time out.
  waitSeconds: 0,
  paths: {
    shared: '../shared'
  },
  shim: {
    'shared/js/template': {
      exports: 'Template'
    },
    emitter: {
      exports: 'Emitter'
    },
    'shared/js/gesture_detector': {
      exports: 'GestureDetector'
    },
    'shared/js/async_storage': {
      exports: 'asyncStorage'
    },
    'shared/js/performance_testing_helper': {
      exports: 'PerformanceTestingHelper'
    },
    'shared/js/l10n_date': ['shared/js/l10n']
  }
});

define("require_config", function(){});

define('tabs',['require'],function(require) {

/**
 * Abstraction for handling the Tabs links at the bottom of the UI.
 * @param {HTMLElement} element The containing element for the Tabs UI.
 */
function Tabs(element) {
  this.element = element;
  this.links = Array.prototype.slice.call(element.querySelectorAll('a'));
  this.currentIndex = 0;
  this.element.addEventListener('click', this);
}

/**
 * Find the clicked link in the list of links and update selected attributes.
 * Also emit a 'selected' event with the relevant data.
 */
Tabs.prototype.handleEvent = function tabsHandleEvent(event) {
  var index = this.links.indexOf(event.target);
  if (index === -1 || index === this.currentIndex) {
    return;
  }
  this.currentIndex = index;
  this.links.forEach(function toggleLinks(link, linkIndex) {
    if (linkIndex === index) {
      link.parentNode.setAttribute('aria-selected', 'true');
    } else {
      link.parentNode.removeAttribute('aria-selected');
    }
  });
};

return Tabs;

});

// outer IIFE
(function(exports) {


// private storage for event handlers using WeakMap
var eventMap = new WeakMap();

/**
 * A simple Event Emitter prototype.  You can enhance any object to become an
 * event emitter by adding the methods to the object's prototype or object:
 *
 * MyConstructor.prototype = Object.create(Emitter.prototype)
 * -or-
 * Emitter.mixin(MyConstructor.prototype);
 * -or-
 * Emitter.mixin(MyObject);
 */
function Emitter() {}

/**
 * add Emitter prototype methods to an arbitrary object
 *
 * @param {object} target The object to decorate with the Emitter methods.
 */
Emitter.mixin = function(target) {
  // generate a non-enumerable property for each method
  methods.forEach(function extendObject(method) {
    Object.defineProperty(target, method, {
      configurable: true,
      value: Emitter.prototype[method]
    });
  });
  return target;
};

// Internal helper: generate or return the handler array for `type` on `object`
function getEvents(object, type) {
  var map = eventMap.get(object);
  if (!map) {
    map = {};
    eventMap.set(object, map);
  }
  if (typeof type !== 'string') {
    throw new Error('Event type must be a string');
  }
  if (!map[type]) {
    map[type] = {
      handlers: [],
      once: []
    };
  }
  return map[type];
}

/**
 * Bind Event Handlers
 * @param {string} type The event type to bind.
 * @param {function} handler The function to be called when event type emits.
 */
Emitter.prototype.on = function(type, handler) {
  var events = getEvents(this, type).handlers;
  if (typeof handler !== 'function') {
    throw new Error('handler must be a function');
  }
  events.push(handler);
  return this;
};

/**
 * Bind Event Handler to only be called once
 * @param {string} type The event type to bind.
 * @param {function} handler The function to be called when event type emits.
 */
Emitter.prototype.once = function(type, handler) {
  var events = getEvents(this, type).once;
  if (typeof handler !== 'function') {
    throw new Error('handler must be a function');
  }
  events.push(handler);
  return this;
};

/**
 * Unbind Event Handlers
 * @param {string} [type] The event type to unbind.  Removes all event
 *                        types if omitted.
 * @param {function} [handler] The function to unbind. Removes all handlers
 *                        of the specified 'type' if omitted.
 */
Emitter.prototype.off = function(type, handler) {
  // remove all events if no type
  if (!type) {
    eventMap.delete(this);
    return this;
  }
  var events = getEvents(this, type);
  // remove all events for type if no handler
  if (!handler) {
    events.handlers.length = 0;
    events.once.length = 0;
  } else {
    // remove only the handler
    var index = events.handlers.indexOf(handler);
    if (index !== -1) {
      events.handlers.splice(index, 1);
    } else {
      index = events.once.indexOf(handler);
      events.once.splice(index, 1);
    }
  }
  return this;
};

/**
 * Emit Event
 * @param {string} type The event type to emit.
 * @param {object} data The first argument passed to each handler
 *                      for the event type.
 */

Emitter.prototype.emit = function(type, data) {
  var event;
  var events = getEvents(this, type);
  var allEvents = events.handlers.concat(events.once);
  events.once.length = 0;

  if (allEvents.length) {
    while ((event = allEvents.shift())) {
      event.call(this, data);
    }
  }
  return this;
};

var methods = Object.keys(Emitter.prototype);

// export our Emitter
exports.Emitter = Emitter;

// end outer IIFE
}(this));

define("emitter", (function (global) {
    return function () {
        var ret, fn;
        return ret || global.Emitter;
    };
}(this)));

define('view',['require','emitter'],function(require) {

var Emitter = require('emitter');
var priv = new WeakMap();
var elementMap = new WeakMap();

/**
 * A View is simply a wrapper around an element.
 *
 * @constructor
 * @param {HTMLElement} element The element that will be wrapped by this view.
 */
function View(element) {
  if (!(this instanceof View)) {
    throw new Error('View must be called as a constructor');
  }
  elementMap.set(element, this);

  Object.defineProperties(this, {
    id: { value: element.id },
    element: { value: element }
  });

  priv.set(this, {
    visible: !element.classList.contains('hidden')
  });
}

/**
 * Find or create a view instance for an element.
 *
 * @param {HTMLElement} element The element that will be wrapped by the view.
 * @param {Function} ctor The constructor method for the view, defaults to View.
 */
View.instance = function(element, ctor = View) {
  if (elementMap.has(element)) {
    return elementMap.get(element);
  }
  return new ctor(element);
};

View.prototype = Object.create(Emitter.prototype);

Object.defineProperties(View.prototype, {
  /**
   * View.prototype.visible - set to true or false to toggle the "hidden" class
   * on the element.
   *
   * Also emits a 'visibilitychange' event passing either true or false to show
   * the new visible state.  The event happens before the class is changed to
   * allow time to modify the DOM before something becomes visible.
   */
  visible: {
    get: function() {
      return priv.get(this).visible;
    },
    set: function(value) {
      var state = priv.get(this);
      value = !!value;
      if (state.visible !== value) {
        state.visible = value;
        this.emit('visibilitychange', value);
        if (!value) {
          this.element.classList.add('hidden');
        } else {
          this.element.classList.remove('hidden');
        }
      }
      return value;
    }
  }
});

return View;

});

/* -*- Mode: js; js-indent-level: 2; indent-tabs-mode: nil -*- */
/* vim: set shiftwidth=2 tabstop=2 autoindent cindent expandtab: */



/**
 * This library exposes a `navigator.mozL10n' object to handle client-side
 * application localization. See: https://github.com/fabi1cazenave/webL10n
 */

(function(window) {
  var gL10nData = {};
  var gLanguage = '';
  var gMacros = {};
  var gReadyState = 'loading';

  // DOM element properties that may be localized with a key:value pair.
  var gNestedProps = ['style', 'dataset'];


  /**
   * Localization resources are declared in the HTML document with <link> nodes:
   *   <link rel="prefetch" type="application/l10n" href="locales.ini" />
   * Such *.ini files are multi-locale dictionaries where all supported locales
   * are listed / defined / imported, and where a fallback locale can easily be
   * defined.
   *
   * These *.ini files can also be compiled to locale-specific JSON dictionaries
   * with the `getDictionary()' method.  Such JSON dictionaries can be used:
   *  - either with a <link> node:
   *   <link rel="prefetch" type="application/l10n" href="{{locale}}.json" />
   *   (in which case, {{locale}} will be replaced by `navigator.language')
   *  - or with an inline <script> node:
   *   <script type="application/l10n" lang="fr"> ... </script>
   *   (in which case, the script matching `navigator.language' will be parsed)
   *
   * This is where `gDefaultLocale' comes in: if a JSON dictionary for the
   * current `navigator.language' value can't be found, use the one matching the
   * default locale.  Note that if the <html> element has a `lang' attribute,
   * its value becomes the default locale.
   */

  var gDefaultLocale = 'en-US';


  /**
   * Synchronously loading l10n resources significantly minimizes flickering
   * from displaying the app with non-localized strings and then updating the
   * strings. Although this will block all script execution on this page, we
   * expect that the l10n resources are available locally on flash-storage.
   *
   * As synchronous XHR is generally considered as a bad idea, we're still
   * loading l10n resources asynchronously -- but we keep this in a setting,
   * just in case... and applications using this library should hide their
   * content until the `localized' event happens.
   */

  var gAsyncResourceLoading = true; // read-only


  /**
   * Debug helpers
   *
   *   gDEBUG == 0: don't display any console message
   *   gDEBUG == 1: display only warnings, not logs
   *   gDEBUG == 2: display all console messages
   */

  var gDEBUG = 1;

  function consoleLog() {
    if (gDEBUG >= 2) {
      var args = [].slice.apply(arguments);
      args.unshift('[l10n] ');
      console.log(args.join(''));
    }
  };

  function consoleWarn() {
    if (gDEBUG) {
      var args = [].slice.apply(arguments);
      args.unshift('[l10n] ');
      console.warn(args.join(''));
    }
  };

  function consoleWarn_missingKeys(untranslatedElements, lang) {
    var len = untranslatedElements.length;
    if (!len || !gDEBUG) {
      return;
    }

    var missingIDs = [];
    for (var i = 0; i < len; i++) {
      var l10nId = untranslatedElements[i].getAttribute('data-l10n-id');
      if (missingIDs.indexOf(l10nId) < 0) {
        missingIDs.push(l10nId);
      }
    }
    console.warn('[l10n] ' +
        missingIDs.length + ' missing key(s) for [' + lang + ']: ' +
        missingIDs.join(', '));
  }


  /**
   * DOM helpers for the so-called "HTML API".
   *
   * These functions are written for modern browsers. For old versions of IE,
   * they're overridden in the 'startup' section at the end of this file.
   */

  function getL10nResourceLinks() {
    return document.querySelectorAll('link[type="application/l10n"]');
  }

  function getL10nDictionary(lang) {
    var getInlineDict = function(locale) {
      var sel = 'script[type="application/l10n"][lang="' + locale + '"]';
      return document.querySelector(sel);
    };
    // TODO: support multiple internal JSON dictionaries
    var script = getInlineDict(lang) || getInlineDict(gDefaultLocale);
    return script ? JSON.parse(script.innerHTML) : null;
  }

  function getTranslatableChildren(element) {
    return element ? element.querySelectorAll('*[data-l10n-id]') : [];
  }

  function getL10nAttributes(element) {
    if (!element) {
      return {};
    }

    var l10nId = element.getAttribute('data-l10n-id');
    var l10nArgs = element.getAttribute('data-l10n-args');
    var args = {};
    if (l10nArgs) {
      try {
        args = JSON.parse(l10nArgs);
      } catch (e) {
        consoleWarn('could not parse arguments for #', l10nId);
      }
    }
    return { id: l10nId, args: args };
  }

  function setTextContent(element, text) {
    // standard case: no element children
    if (!element.firstElementChild) {
      element.textContent = text;
      return;
    }

    // this element has element children: replace the content of the first
    // (non-blank) child textNode and clear other child textNodes
    var found = false;
    var reNotBlank = /\S/;
    for (var child = element.firstChild; child; child = child.nextSibling) {
      if (child.nodeType === 3 && reNotBlank.test(child.nodeValue)) {
        if (found) {
          child.nodeValue = '';
        } else {
          child.nodeValue = text;
          found = true;
        }
      }
    }
    // if no (non-empty) textNode is found, insert a textNode before the
    // element's first child.
    if (!found) {
      element.insertBefore(document.createTextNode(text), element.firstChild);
    }
  }

  function fireL10nReadyEvent() {
    var evtObject = document.createEvent('Event');
    evtObject.initEvent('localized', false, false);
    evtObject.language = gLanguage;
    window.dispatchEvent(evtObject);
  }


  /**
   * l10n resource parser:
   *  - reads (async XHR) the l10n resource matching `lang';
   *  - imports linked resources (synchronously) when specified;
   *  - parses the text data (fills `gL10nData');
   *  - triggers success/failure callbacks when done.
   *
   * @param {string} href
   *    URL of the l10n resource to parse.
   *
   * @param {string} lang
   *    locale (language) to parse.
   *
   * @param {Function} successCallback
   *    triggered when the l10n resource has been successully parsed.
   *
   * @param {Function} failureCallback
   *    triggered when the an error has occured.
   *
   * @return {void}
   *    fills gL10nData.
   */

  function parseResource(href, lang, successCallback, failureCallback) {
    var baseURL = href.replace(/\/[^\/]*$/, '/');

    // handle escaped characters (backslashes) in a string
    function evalString(text) {
      if (text.lastIndexOf('\\') < 0) {
        return text;
      }
      return text.replace(/\\\\/g, '\\')
                 .replace(/\\n/g, '\n')
                 .replace(/\\r/g, '\r')
                 .replace(/\\t/g, '\t')
                 .replace(/\\b/g, '\b')
                 .replace(/\\f/g, '\f')
                 .replace(/\\{/g, '{')
                 .replace(/\\}/g, '}')
                 .replace(/\\"/g, '"')
                 .replace(/\\'/g, "'");
    }

    // parse *.properties text data into an l10n dictionary
    function parseProperties(text) {
      var dictionary = [];

      // token expressions
      var reBlank = /^\s*|\s*$/;
      var reComment = /^\s*#|^\s*$/;
      var reSection = /^\s*\[(.*)\]\s*$/;
      var reImport = /^\s*@import\s+url\((.*)\)\s*$/i;
      var reSplit = /^([^=\s]*)\s*=\s*(.+)$/;
      var reUnicode = /\\u([0-9a-fA-F]{1,4})/g;
      var reMultiline = /[^\\]\\$/;

      // parse the *.properties file into an associative array
      function parseRawLines(rawText, extendedSyntax) {
        var entries = rawText.replace(reBlank, '').split(/[\r\n]+/);
        var currentLang = '*';
        var genericLang = lang.replace(/-[a-z]+$/i, '');
        var skipLang = false;
        var match = '';

        for (var i = 0; i < entries.length; i++) {
          var line = entries[i];

          // comment or blank line?
          if (reComment.test(line)) {
            continue;
          }

          // multi-line?
          while (reMultiline.test(line) && i < entries.length) {
            line = line.slice(0, line.length - 1) +
              entries[++i].replace(reBlank, '');
          }

          // the extended syntax supports [lang] sections and @import rules
          if (extendedSyntax) {
            if (reSection.test(line)) { // section start?
              match = reSection.exec(line);
              currentLang = match[1];
              skipLang = (currentLang !== '*') &&
                  (currentLang !== lang) && (currentLang !== genericLang);
              continue;
            } else if (skipLang) {
              continue;
            }
            if (reImport.test(line)) { // @import rule?
              match = reImport.exec(line);
              loadImport(baseURL + match[1]); // load the resource synchronously
            }
          }

          // key-value pair
          var tmp = line.match(reSplit);
          if (tmp && tmp.length == 3) {
            // unescape unicode char codes if needed (e.g. '\u00a0')
            var val = tmp[2].replace(reUnicode, function(match, token) {
              return unescape('%u' + '0000'.slice(token.length) + token);
            });
            dictionary[tmp[1]] = evalString(val);
          }
        }
      }

      // import another *.properties file
      function loadImport(url) {
        loadResource(url, function(content) {
          parseRawLines(content, false); // don't allow recursive imports
        }, null, false); // load synchronously
      }

      // fill the dictionary
      parseRawLines(text, true);
      return dictionary;
    }

    // load the specified resource file
    function loadResource(url, onSuccess, onFailure, asynchronous) {
      onSuccess = onSuccess || function _onSuccess(data) {};
      onFailure = onFailure || function _onFailure() {
        consoleWarn(url, ' not found.');
      };

      var xhr = new XMLHttpRequest();
      xhr.open('GET', url, asynchronous);
      if (xhr.overrideMimeType) {
        xhr.overrideMimeType('text/plain; charset=utf-8');
      }
      xhr.onreadystatechange = function() {
        if (xhr.readyState == 4) {
          if (xhr.status == 200 || xhr.status === 0) {
            onSuccess(xhr.responseText);
          } else {
            onFailure();
          }
        }
      };
      xhr.onerror = onFailure;
      xhr.ontimeout = onFailure;

      // in Firefox OS with the app:// protocol, trying to XHR a non-existing
      // URL will raise an exception here -- hence this ugly try...catch.
      try {
        xhr.send(null);
      } catch (e) {
        onFailure();
      }
    }

    // load and parse l10n data (warning: global variables are used here)
    loadResource(href, function(response) {
      if (/\.json$/.test(href)) {
        gL10nData = JSON.parse(response); // TODO: support multiple JSON files
      } else { // *.ini or *.properties file
        var data = parseProperties(response);
        for (var key in data) {
          var id, prop, nestedProp, index = key.lastIndexOf('.');
          if (index > 0) { // a property name has been specified
            id = key.slice(0, index);
            prop = key.slice(index + 1);
            index = id.lastIndexOf('.');
            if (index > 0) { // a nested property may have been specified
              nestedProp = id.substr(index + 1);
              if (gNestedProps.indexOf(nestedProp) > -1) {
                id = id.substr(0, index);
                prop = nestedProp + '.' + prop;
              }
            }
          } else { // no property name: assuming text content by default
            index = key.lastIndexOf('[');
            if (index > 0) { // we have a macro index
              id = key.slice(0, index);
              prop = '_' + key.slice(index);
            } else {
              id = key;
              prop = '_';
            }
          }
          if (!gL10nData[id]) {
            gL10nData[id] = {};
          }
          gL10nData[id][prop] = data[key];
        }
      }

      // trigger callback
      if (successCallback) {
        successCallback();
      }
    }, failureCallback, gAsyncResourceLoading);
  };

  // load and parse all resources for the specified locale
  function loadLocale(lang, translationRequired) {
    clear();
    gReadyState = 'loading';
    gLanguage = lang;

    var untranslatedElements = [];

    // if there is an inline / pre-compiled dictionary,
    // the current HTML document can be translated right now
    var inlineDict = getL10nDictionary(lang);
    if (inlineDict) {
      gL10nData = inlineDict;
      if (translationRequired) {
        untranslatedElements = translateFragment();
      }
    }

    // translate the document if required and fire a `localized' event
    function finish() {
      if (translationRequired) {
        if (!inlineDict) {
          // no inline dictionary has been used: translate the whole document
          untranslatedElements = translateFragment();
        } else if (untranslatedElements.length) {
          // the document should have been already translated but the inline
          // dictionary didn't include all necessary l10n keys:
          // try to translate all remaining elements now
          untranslatedElements = translateElements(untranslatedElements);
        }
      }
      // tell the rest of the world we're done
      // -- note that `gReadyState' must be set before the `localized' event is
      //    fired for `localizeElement()' to work as expected
      gReadyState = 'complete';
      fireL10nReadyEvent(lang);
      consoleWarn_missingKeys(untranslatedElements, lang);
    }

    // l10n resource loader
    function l10nResourceLink(link) {
      /**
       * l10n resource links can use the following syntax for href:
       * <link type="application/l10n" href="resources/{{locale}}.json" />
       * -- in which case, {{locale}} will be replaced by `navigator.language'.
       */
      var re = /\{\{\s*locale\s*\}\}/;

      var parse = function(locale, onload, onerror) {
        var href = unescape(link.href).replace(re, locale);
        parseResource(href, locale, onload, function notFound() {
          consoleWarn(href, ' not found.');
          onerror();
        });
      };

      this.load = function(locale, onload, onerror) {
        onerror = onerror || function() {};
        parse(locale, onload, function parseFallbackLocale() {
          /**
           * For links like <link href="resources/{{locale}}.json" />,
           * there's no way to know if the resource file matching the current
           * language has been found... before trying to fetch it with XHR
           * => if something went wrong, try the default locale as fallback.
           */
          if (re.test(unescape(link.href)) && gDefaultLocale != locale) {
            consoleLog('Trying the fallback locale: ', gDefaultLocale);
            parse(gDefaultLocale, onload, onerror);
          } else {
            onerror();
          }
        });
      };
    }

    // check all <link type="application/l10n" href="..." /> nodes
    // and load the resource files
    var resourceLinks = getL10nResourceLinks();
    var resourceCount = resourceLinks.length;
    if (!resourceCount) {
      consoleLog('no resource to load, early way out');
      translationRequired = false;
      finish();
    } else {
      var onResourceCallback = function() {
        if (--resourceCount <= 0) { // <=> all resources have been XHR'ed
          finish();
        }
      };
      for (var i = 0, l = resourceCount; i < l; i++) {
        var resource = new l10nResourceLink(resourceLinks[i]);
        resource.load(lang, onResourceCallback, onResourceCallback);
      }
    }
  }

  // clear all l10n data
  function clear() {
    gL10nData = {};
    gLanguage = '';
    // TODO: clear all non predefined macros.
    // There's no such macro /yet/ but we're planning to have some...
  }


  /**
   * Get rules for plural forms (shared with JetPack), see:
   * http://unicode.org/repos/cldr-tmp/trunk/diff/supplemental/language_plural_rules.html
   * https://github.com/mozilla/addon-sdk/blob/master/python-lib/plural-rules-generator.p
   *
   * @param {string} lang
   *    locale (language) used.
   *
   * @return {Function}
   *    returns a function that gives the plural form name for a given integer:
   *       var fun = getPluralRules('en');
   *       fun(1)    -> 'one'
   *       fun(0)    -> 'other'
   *       fun(1000) -> 'other'.
   */

  var kPluralForms = ['zero', 'one', 'two', 'few', 'many', 'other'];

  function getPluralRules(lang) {
    var locales2rules = {
      'af': 3,
      'ak': 4,
      'am': 4,
      'ar': 1,
      'asa': 3,
      'az': 0,
      'be': 11,
      'bem': 3,
      'bez': 3,
      'bg': 3,
      'bh': 4,
      'bm': 0,
      'bn': 3,
      'bo': 0,
      'br': 20,
      'brx': 3,
      'bs': 11,
      'ca': 3,
      'cgg': 3,
      'chr': 3,
      'cs': 12,
      'cy': 17,
      'da': 3,
      'de': 3,
      'dv': 3,
      'dz': 0,
      'ee': 3,
      'el': 3,
      'en': 3,
      'eo': 3,
      'es': 3,
      'et': 3,
      'eu': 3,
      'fa': 0,
      'ff': 5,
      'fi': 3,
      'fil': 4,
      'fo': 3,
      'fr': 5,
      'fur': 3,
      'fy': 3,
      'ga': 8,
      'gd': 24,
      'gl': 3,
      'gsw': 3,
      'gu': 3,
      'guw': 4,
      'gv': 23,
      'ha': 3,
      'haw': 3,
      'he': 2,
      'hi': 4,
      'hr': 11,
      'hu': 0,
      'id': 0,
      'ig': 0,
      'ii': 0,
      'is': 3,
      'it': 3,
      'iu': 7,
      'ja': 0,
      'jmc': 3,
      'jv': 0,
      'ka': 0,
      'kab': 5,
      'kaj': 3,
      'kcg': 3,
      'kde': 0,
      'kea': 0,
      'kk': 3,
      'kl': 3,
      'km': 0,
      'kn': 0,
      'ko': 0,
      'ksb': 3,
      'ksh': 21,
      'ku': 3,
      'kw': 7,
      'lag': 18,
      'lb': 3,
      'lg': 3,
      'ln': 4,
      'lo': 0,
      'lt': 10,
      'lv': 6,
      'mas': 3,
      'mg': 4,
      'mk': 16,
      'ml': 3,
      'mn': 3,
      'mo': 9,
      'mr': 3,
      'ms': 0,
      'mt': 15,
      'my': 0,
      'nah': 3,
      'naq': 7,
      'nb': 3,
      'nd': 3,
      'ne': 3,
      'nl': 3,
      'nn': 3,
      'no': 3,
      'nr': 3,
      'nso': 4,
      'ny': 3,
      'nyn': 3,
      'om': 3,
      'or': 3,
      'pa': 3,
      'pap': 3,
      'pl': 13,
      'ps': 3,
      'pt': 3,
      'rm': 3,
      'ro': 9,
      'rof': 3,
      'ru': 11,
      'rwk': 3,
      'sah': 0,
      'saq': 3,
      'se': 7,
      'seh': 3,
      'ses': 0,
      'sg': 0,
      'sh': 11,
      'shi': 19,
      'sk': 12,
      'sl': 14,
      'sma': 7,
      'smi': 7,
      'smj': 7,
      'smn': 7,
      'sms': 7,
      'sn': 3,
      'so': 3,
      'sq': 3,
      'sr': 11,
      'ss': 3,
      'ssy': 3,
      'st': 3,
      'sv': 3,
      'sw': 3,
      'syr': 3,
      'ta': 3,
      'te': 3,
      'teo': 3,
      'th': 0,
      'ti': 4,
      'tig': 3,
      'tk': 3,
      'tl': 4,
      'tn': 3,
      'to': 0,
      'tr': 0,
      'ts': 3,
      'tzm': 22,
      'uk': 11,
      'ur': 3,
      've': 3,
      'vi': 0,
      'vun': 3,
      'wa': 4,
      'wae': 3,
      'wo': 0,
      'xh': 3,
      'xog': 3,
      'yo': 0,
      'zh': 0,
      'zu': 3
    };

    // utility functions for plural rules methods
    function isIn(n, list) {
      return list.indexOf(n) !== -1;
    }
    function isBetween(n, start, end) {
      return start <= n && n <= end;
    }

    // list of all plural rules methods:
    // map an integer to the plural form name to use
    var pluralRules = {
      '0': function(n) {
        return 'other';
      },
      '1': function(n) {
        if ((isBetween((n % 100), 3, 10)))
          return 'few';
        if (n === 0)
          return 'zero';
        if ((isBetween((n % 100), 11, 99)))
          return 'many';
        if (n == 2)
          return 'two';
        if (n == 1)
          return 'one';
        return 'other';
      },
      '2': function(n) {
        if (n !== 0 && (n % 10) === 0)
          return 'many';
        if (n == 2)
          return 'two';
        if (n == 1)
          return 'one';
        return 'other';
      },
      '3': function(n) {
        if (n == 1)
          return 'one';
        return 'other';
      },
      '4': function(n) {
        if ((isBetween(n, 0, 1)))
          return 'one';
        return 'other';
      },
      '5': function(n) {
        if ((isBetween(n, 0, 2)) && n != 2)
          return 'one';
        return 'other';
      },
      '6': function(n) {
        if (n === 0)
          return 'zero';
        if ((n % 10) == 1 && (n % 100) != 11)
          return 'one';
        return 'other';
      },
      '7': function(n) {
        if (n == 2)
          return 'two';
        if (n == 1)
          return 'one';
        return 'other';
      },
      '8': function(n) {
        if ((isBetween(n, 3, 6)))
          return 'few';
        if ((isBetween(n, 7, 10)))
          return 'many';
        if (n == 2)
          return 'two';
        if (n == 1)
          return 'one';
        return 'other';
      },
      '9': function(n) {
        if (n === 0 || n != 1 && (isBetween((n % 100), 1, 19)))
          return 'few';
        if (n == 1)
          return 'one';
        return 'other';
      },
      '10': function(n) {
        if ((isBetween((n % 10), 2, 9)) && !(isBetween((n % 100), 11, 19)))
          return 'few';
        if ((n % 10) == 1 && !(isBetween((n % 100), 11, 19)))
          return 'one';
        return 'other';
      },
      '11': function(n) {
        if ((isBetween((n % 10), 2, 4)) && !(isBetween((n % 100), 12, 14)))
          return 'few';
        if ((n % 10) === 0 ||
            (isBetween((n % 10), 5, 9)) ||
            (isBetween((n % 100), 11, 14)))
          return 'many';
        if ((n % 10) == 1 && (n % 100) != 11)
          return 'one';
        return 'other';
      },
      '12': function(n) {
        if ((isBetween(n, 2, 4)))
          return 'few';
        if (n == 1)
          return 'one';
        return 'other';
      },
      '13': function(n) {
        if ((isBetween((n % 10), 2, 4)) && !(isBetween((n % 100), 12, 14)))
          return 'few';
        if (n != 1 && (isBetween((n % 10), 0, 1)) ||
            (isBetween((n % 10), 5, 9)) ||
            (isBetween((n % 100), 12, 14)))
          return 'many';
        if (n == 1)
          return 'one';
        return 'other';
      },
      '14': function(n) {
        if ((isBetween((n % 100), 3, 4)))
          return 'few';
        if ((n % 100) == 2)
          return 'two';
        if ((n % 100) == 1)
          return 'one';
        return 'other';
      },
      '15': function(n) {
        if (n === 0 || (isBetween((n % 100), 2, 10)))
          return 'few';
        if ((isBetween((n % 100), 11, 19)))
          return 'many';
        if (n == 1)
          return 'one';
        return 'other';
      },
      '16': function(n) {
        if ((n % 10) == 1 && n != 11)
          return 'one';
        return 'other';
      },
      '17': function(n) {
        if (n == 3)
          return 'few';
        if (n === 0)
          return 'zero';
        if (n == 6)
          return 'many';
        if (n == 2)
          return 'two';
        if (n == 1)
          return 'one';
        return 'other';
      },
      '18': function(n) {
        if (n === 0)
          return 'zero';
        if ((isBetween(n, 0, 2)) && n !== 0 && n != 2)
          return 'one';
        return 'other';
      },
      '19': function(n) {
        if ((isBetween(n, 2, 10)))
          return 'few';
        if ((isBetween(n, 0, 1)))
          return 'one';
        return 'other';
      },
      '20': function(n) {
        if ((isBetween((n % 10), 3, 4) || ((n % 10) == 9)) && !(
            isBetween((n % 100), 10, 19) ||
            isBetween((n % 100), 70, 79) ||
            isBetween((n % 100), 90, 99)
            ))
          return 'few';
        if ((n % 1000000) === 0 && n !== 0)
          return 'many';
        if ((n % 10) == 2 && !isIn((n % 100), [12, 72, 92]))
          return 'two';
        if ((n % 10) == 1 && !isIn((n % 100), [11, 71, 91]))
          return 'one';
        return 'other';
      },
      '21': function(n) {
        if (n === 0)
          return 'zero';
        if (n == 1)
          return 'one';
        return 'other';
      },
      '22': function(n) {
        if ((isBetween(n, 0, 1)) || (isBetween(n, 11, 99)))
          return 'one';
        return 'other';
      },
      '23': function(n) {
        if ((isBetween((n % 10), 1, 2)) || (n % 20) === 0)
          return 'one';
        return 'other';
      },
      '24': function(n) {
        if ((isBetween(n, 3, 10) || isBetween(n, 13, 19)))
          return 'few';
        if (isIn(n, [2, 12]))
          return 'two';
        if (isIn(n, [1, 11]))
          return 'one';
        return 'other';
      }
    };

    // return a function that gives the plural form name for a given integer
    var index = locales2rules[lang.replace(/-.*$/, '')];
    if (!(index in pluralRules)) {
      consoleWarn('plural form unknown for [', lang, ']');
      return function() { return 'other'; };
    }
    return pluralRules[index];
  }

  // pre-defined 'plural' macro
  gMacros.plural = function(str, param, key, prop) {
    var n = parseFloat(param);
    if (isNaN(n)) {
      return str;
    }

    var data = gL10nData[key];
    if (!data) {
      return str;
    }

    // initialize _pluralRules
    if (!gMacros._pluralRules) {
      gMacros._pluralRules = getPluralRules(gLanguage);
    }
    var index = '[' + gMacros._pluralRules(n) + ']';

    // try to find a [zero|one|two] form if it's defined
    if (n === 0 && (prop + '[zero]') in data) {
      str = data[prop + '[zero]'];
    } else if (n == 1 && (prop + '[one]') in data) {
      str = data[prop + '[one]'];
    } else if (n == 2 && (prop + '[two]') in data) {
      str = data[prop + '[two]'];
    } else if ((prop + index) in data) {
      str = data[prop + index];
    } else if ((prop + '[other]') in data) {
      str = data[prop + '[other]'];
    }

    return str;
  };


  /**
   * l10n dictionary functions
   */

  var reArgs = /\{\{\s*(.+?)\s*\}\}/;                       // arguments
  var reIndex = /\{\[\s*([a-zA-Z]+)\(([a-zA-Z]+)\)\s*\]\}/; // index macros

  // fetch an l10n object, warn if not found, apply `args' if possible
  function getL10nData(key, args) {
    var data = gL10nData[key];
    if (!data) {
      return null;
    }

    /**
     * This is where l10n expressions should be processed.
     * The plan is to support C-style expressions from the l20n project;
     * until then, only two kinds of simple expressions are supported:
     *   {[ index ]} and {{ arguments }}.
     */
    var rv = {};
    for (var prop in data) {
      var str = data[prop];
      str = substIndexes(str, args, key, prop);
      str = substArguments(str, args, key);
      rv[prop] = str;
    }
    return rv;
  }

  // return an array of all {{arguments}} found in a string
  function getL10nArgs(str) {
    var args = [];
    var match = reArgs.exec(str);
    while (match && match.length >= 2) {
      args.push({
        name: match[1], // name of the argument
        subst: match[0] // substring to replace (including braces and spaces)
      });
      str = str.substr(match.index + match[0].length);
      match = reArgs.exec(str);
    }
    return args;
  }

  // return a sub-dictionary sufficient to translate a given fragment
  function getSubDictionary(fragment) {
    if (!fragment) { // by default, return a clone of the whole dictionary
      return JSON.parse(JSON.stringify(gL10nData));
    }

    var dict = {};
    var elements = getTranslatableChildren(fragment);

    function checkGlobalArguments(str) {
      var match = getL10nArgs(str);
      for (var i = 0; i < match.length; i++) {
        var arg = match[i].name;
        if (arg in gL10nData) {
          dict[arg] = gL10nData[arg];
        }
      }
    }

    for (var i = 0, l = elements.length; i < l; i++) {
      var id = getL10nAttributes(elements[i]).id;
      var data = gL10nData[id];
      if (!id || !data) {
        continue;
      }

      dict[id] = data;
      for (var prop in data) {
        var str = data[prop];
        checkGlobalArguments(str);

        if (reIndex.test(str)) { // macro index
          for (var j = 0; j < kPluralForms.length; j++) {
            var key = id + '[' + kPluralForms[j] + ']';
            if (key in gL10nData) {
              dict[key] = gL10nData[key];
              checkGlobalArguments(gL10nData[key]);
            }
          }
        }
      }
    }

    return dict;
  }

  // replace {[macros]} with their values
  function substIndexes(str, args, key, prop) {
    var reMatch = reIndex.exec(str);
    if (!reMatch || !reMatch.length) {
      return str;
    }

    // an index/macro has been found
    // Note: at the moment, only one parameter is supported
    var macroName = reMatch[1];
    var paramName = reMatch[2];
    var param;
    if (args && paramName in args) {
      param = args[paramName];
    } else if (paramName in gL10nData) {
      param = gL10nData[paramName];
    }

    // there's no macro parser yet: it has to be defined in gMacros
    if (macroName in gMacros) {
      var macro = gMacros[macroName];
      str = macro(str, param, key, prop);
    }
    return str;
  }

  // replace {{arguments}} with their values
  function substArguments(str, args, key) {
    var match = getL10nArgs(str);
    for (var i = 0; i < match.length; i++) {
      var sub, arg = match[i].name;
      if (args && arg in args) {
        sub = args[arg];
      } else if (arg in gL10nData) {
        sub = gL10nData[arg]['_'];
      } else {
        consoleLog('argument {{', arg, '}} for #', key, ' is undefined.');
        return str;
      }
      if (typeof sub == 'string') {
        // dollar signs would be interpreted as replacement patterns
        sub = sub.replace(/\$/g, '$$$$');
      }
      str = str.replace(match[i].subst, sub);
    }
    return str;
  }

  // translate an HTML element
  // -- returns true if the element could be translated, false otherwise
  function translateElement(element) {
    var l10n = getL10nAttributes(element);
    if (!l10n.id) {
      return true;
    }

    // get the related l10n object
    var data = getL10nData(l10n.id, l10n.args);
    if (!data) {
      return false;
    }

    // translate element (TODO: security checks?)
    for (var k in data) {
      if (k === '_') {
        setTextContent(element, data._);
      } else {
        var idx = k.lastIndexOf('.');
        var nestedProp = k.substr(0, idx);
        if (gNestedProps.indexOf(nestedProp) > -1) {
          element[nestedProp][k.substr(idx + 1)] = data[k];
        } else if (k === 'ariaLabel') {
          element.setAttribute('aria-label', data[k]);
        } else {
          element[k] = data[k];
        }
      }
    }
    return true;
  }

  // translate an array of HTML elements
  // -- returns an array of elements that could not be translated
  function translateElements(elements) {
    var untranslated = [];
    for (var i = 0, l = elements.length; i < l; i++) {
      if (!translateElement(elements[i])) {
        untranslated.push(elements[i]);
      }
    }
    return untranslated;
  }

  // translate an HTML subtree
  // -- returns an array of elements that could not be translated
  function translateFragment(element) {
    element = element || document.documentElement;
    var untranslated = translateElements(getTranslatableChildren(element));
    if (!translateElement(element)) {
      untranslated.push(element);
    }
    return untranslated;
  }

  // localize an element as soon as mozL10n is ready
  function localizeElement(element, id, args) {
    if (!element) {
      return;
    }

    if (!id) {
      element.removeAttribute('data-l10n-id');
      element.removeAttribute('data-l10n-args');
      setTextContent(element, '');
      return;
    }

    // set the data-l10n-[id|args] attributes
    element.setAttribute('data-l10n-id', id);
    if (args && typeof args === 'object') {
      element.setAttribute('data-l10n-args', JSON.stringify(args));
    } else {
      element.removeAttribute('data-l10n-args');
    }

    // if l10n resources are ready, translate now;
    // if not, the element will be translated along with the document anyway.
    if (gReadyState === 'complete') {
      translateElement(element);
    }
  }


  /**
   * Startup & Public API
   *
   * This section is quite specific to the B2G project: old browsers are not
   * supported and the API is slightly different from the standard webl10n one.
   */

  // load the default locale on startup
  function l10nStartup() {
    gDefaultLocale = document.documentElement.lang || gDefaultLocale;
    gReadyState = 'interactive';
    consoleLog('loading [', navigator.language, '] resources, ',
        (gAsyncResourceLoading ? 'asynchronously.' : 'synchronously.'));

    // load the default locale and translate the document if required
    var translationRequired =
      (document.documentElement.lang !== navigator.language);
    loadLocale(navigator.language, translationRequired);
  }

  // the B2G build system doesn't expose any `document'...
  if (typeof(document) !== 'undefined') {
    if (document.readyState === 'complete' ||
      document.readyState === 'interactive') {
      window.setTimeout(l10nStartup);
    } else {
      document.addEventListener('DOMContentLoaded', l10nStartup);
    }
  }

  // load the appropriate locale if the language setting has changed
  if ('mozSettings' in navigator && navigator.mozSettings) {
    navigator.mozSettings.addObserver('language.current', function(event) {
      loadLocale(event.settingValue, true);
    });
  }

  // public API
  navigator.mozL10n = {
    // get a localized string
    get: function l10n_get(key, args) {
      var data = getL10nData(key, args);
      if (!data) {
        consoleWarn('#', key, ' is undefined.');
        return '';
      } else {
        return data._;
      }
    },

    // get|set the document language and direction
    get language() {
      return {
        // get|set the document language (ISO-639-1)
        get code() { return gLanguage; },
        set code(lang) { loadLocale(lang, true); },

        // get the direction (ltr|rtl) of the current language
        get direction() {
          // http://www.w3.org/International/questions/qa-scripts
          // Arabic, Hebrew, Farsi, Pashto, Urdu
          var rtlList = ['ar', 'he', 'fa', 'ps', 'ur'];
          return (rtlList.indexOf(gLanguage) >= 0) ? 'rtl' : 'ltr';
        }
      };
    },

    // translate an element or document fragment
    translate: translateFragment,

    // localize an element (= set its data-l10n-* attributes and translate it)
    localize: localizeElement,

    // get (a part of) the dictionary for the current locale
    getDictionary: getSubDictionary,

    // this can be used to prevent race conditions
    get readyState() { return gReadyState; },
    ready: function l10n_ready(callback) {
      if (!callback) {
        return;
      }
      if (gReadyState == 'complete') {
        window.setTimeout(callback);
      } else {
        window.addEventListener('localized', callback);
      }
    }
  };

  consoleLog('library loaded.');
})(this);


define("shared/js/l10n", function(){});

/* -*- Mode: js; js-indent-level: 2; indent-tabs-mode: nil -*- */
/* vim: set shiftwidth=2 tabstop=2 autoindent cindent expandtab: */



/**
 * This lib relies on `l10n.js' to implement localizable date/time strings.
 *
 * The proposed `DateTimeFormat' object should provide all the features that are
 * planned for the `Intl.DateTimeFormat' constructor, but the API does not match
 * exactly the ES-i18n draft.
 *   - https://bugzilla.mozilla.org/show_bug.cgi?id=769872
 *   - http://wiki.ecmascript.org/doku.php?id=globalization:specification_drafts
 *
 * Besides, this `DateTimeFormat' object provides two features that aren't
 * planned in the ES-i18n spec:
 *   - a `toLocaleFormat()' that really works (i.e. fully translated);
 *   - a `fromNow()' method to handle relative dates ("pretty dates").
 *
 * WARNING: this library relies on the non-standard `toLocaleFormat()' method,
 * which is specific to Firefox -- no other browser is supported.
 */

navigator.mozL10n.DateTimeFormat = function(locales, options) {
  var _ = navigator.mozL10n.get;

  // https://developer.mozilla.org/en-US/docs/JavaScript/Reference/Global_Objects/Date/toLocaleFormat
  function localeFormat(d, format) {
    var tokens = format.match(/(%E.|%O.|%.)/g);

    for (var i = 0; tokens && i < tokens.length; i++) {
      var value = '';

      // http://pubs.opengroup.org/onlinepubs/007908799/xsh/strftime.html
      switch (tokens[i]) {
        // localized day/month names
        case '%a':
          value = _('weekday-' + d.getDay() + '-short');
          break;
        case '%A':
          value = _('weekday-' + d.getDay() + '-long');
          break;
        case '%b':
        case '%h':
          value = _('month-' + d.getMonth() + '-short');
          break;
        case '%B':
          value = _('month-' + d.getMonth() + '-long');
          break;
        case '%Eb':
          value = _('month-' + d.getMonth() + '-genitive');
          break;

        // like %H, but in 12-hour format and without any leading zero
        case '%I':
          value = d.getHours() % 12 || 12;
          break;

        // like %d, without any leading zero
        case '%e':
          value = d.getDate();
          break;

        // %p: 12 hours format (AM/PM)
        case '%p':
          value = d.getHours() < 12 ? _('time_am') : _('time_pm');
          break;

        // localized date/time strings
        case '%c':
        case '%x':
        case '%X':
          // ensure the localized format string doesn't contain any %c|%x|%X
          var tmp = _('dateTimeFormat_' + tokens[i]);
          if (tmp && !(/(%c|%x|%X)/).test(tmp)) {
            value = localeFormat(d, tmp);
          }
          break;

        // other tokens don't require any localization
      }

      format = format.replace(tokens[i], value || d.toLocaleFormat(tokens[i]));
    }

    return format;
  }

  /**
   * Returns the parts of a number of seconds
   */
  function relativeParts(seconds) {
    seconds = Math.abs(seconds);
    var descriptors = {};
    var units = [
      'years', 86400 * 365,
      'months', 86400 * 30,
      'weeks', 86400 * 7,
      'days', 86400,
      'hours', 3600,
      'minutes', 60
    ];

    if (seconds < 60) {
      return {
        minutes: Math.round(seconds / 60)
      };
    }

    for (var i = 0, uLen = units.length; i < uLen; i += 2) {
      var value = units[i + 1];
      if (seconds >= value) {
        descriptors[units[i]] = Math.floor(seconds / value);
        seconds -= descriptors[units[i]] * value;
      }
    }
    return descriptors;
  }

  /**
   * Returns a translated string which respresents the
   * relative time before or after a date.
   * @param {String|Date} time before/after the currentDate.
   * @param {String} useCompactFormat whether to use a compact display format.
   * @param {Number} maxDiff returns a formatted date if the diff is greater.
   */
  function prettyDate(time, useCompactFormat, maxDiff) {
    maxDiff = maxDiff || 86400 * 10; // default = 10 days

    switch (time.constructor) {
      case String: // timestamp
        time = parseInt(time);
        break;
      case Date:
        time = time.getTime();
        break;
    }

    var secDiff = (Date.now() - time) / 1000;
    if (isNaN(secDiff)) {
      return _('incorrectDate');
    }

    if (secDiff > maxDiff) {
      return localeFormat(new Date(time), '%x');
    }

    var f = useCompactFormat ? '-short' : '-long';
    var parts = relativeParts(secDiff);

    var affix = secDiff >= 0 ? '-ago' : '-until';
    for (var i in parts) {
      return _(i + affix + f, { value: parts[i]});
    }
  }

  // API
  return {
    localeDateString: function localeDateString(d) {
      return localeFormat(d, '%x');
    },
    localeTimeString: function localeTimeString(d) {
      return localeFormat(d, '%X');
    },
    localeString: function localeString(d) {
      return localeFormat(d, '%c');
    },
    localeFormat: localeFormat,
    fromNow: prettyDate,
    relativeParts: relativeParts
  };
};

define("shared/js/l10n_date", function(){});

// This module declares a dependency on the shared "l10n_date" module which
// itself depends on the shared "l10n" module (see the application's "shim"
// configuration for Alameda). Declaring the dependencies in this way ensures
// that this module exports the "l10n" module's global variable only after both
// shared libraries have been loaded in the correct order.
define('l10n',['shared/js/l10n_date'], function() {
  
  return navigator.mozL10n;
});



(function(window) {

  function dispatch(name) {
    if (!window.mozPerfHasListener) {
      return;
    }

    var now = window.performance.now();

    setTimeout(function() {
      //console.log('PerformanceTestingHelper: dispatching event', name);

      var detail = {
        name: name,
        timestamp: now
      };
      var evt = new CustomEvent('x-moz-perf', { detail: detail });
      window.dispatchEvent(evt);
    });
  }

  window.PerformanceTestingHelper = {
    dispatch: dispatch
  };

})(window);

define("shared/js/performance_testing_helper", (function (global) {
    return function () {
        var ret, fn;
        return ret || global.PerformanceTestingHelper;
    };
}(this)));

define('app',['require','tabs','view','l10n','shared/js/performance_testing_helper'],function(require) {


var Tabs = require('tabs');
var View = require('view');
var mozL10n = require('l10n');
var PerformanceTestingHelper = require('shared/js/performance_testing_helper');
var rAF = mozRequestAnimationFrame || requestAnimationFrame;
/**
 * Global Application event handling and paging
 */
var App = {
  /**
   * Load the Tabs and Panels, attach events and navigate to the default view.
   */
  init: function() {
    this.tabs = new Tabs(document.getElementById('clock-tabs'));

    window.addEventListener('hashchange', this);
    window.addEventListener('localized', this);
    window.addEventListener('visibilitychange', this);

    // we wait for the app to be l10n ready before initializing, so call
    // the onlocalized once at startup
    this.onlocalized();

    this.visible = !document.hidden;
    this.panels = Array.prototype.map.call(
      document.querySelectorAll('[data-panel-id]'),
      function(element) {
        var panel = {
          el: element,
          fragment: element.dataset.panelId.replace('_', '-') + '-panel',
          instance: null
        };

        return panel;
      }.bind(this)
    );
    this.navigate({ hash: '#alarm-panel' }, function() {
      // Dispatch an event to mark when we've finished loading.
      PerformanceTestingHelper.dispatch('startup-path-done');
    });
    return this;
  },

  /**
   * Load and instantiate the specified panel (when necessary).
   *
   * @param {Object} panel - An object describing the panel. It must contain
   *                         either an `el` attribute (defining the panel's
   *                         containing element) or an `instance` attribute
   *                         (defining the instantiated Panel itself).
   * @param {Function} [callback] - A function that will be invoked with the
   *                                instantiated panel once it is loaded.
   */
  loadPanel: function(panel, callback) {
    if (panel.instance) {
      callback && setTimeout(callback, 0, panel);
      return;
    }

    var moduleId = 'panels/' + panel.el.dataset.panelId + '/main';

    require([moduleId], function(PanelModule) {
      panel.instance = View.instance(panel.el, PanelModule);
      callback && callback(panel);
    });
  },

  /**
   * split each event handler into it's own method
   */
  handleEvent: function(event) {
    var handler = this['on' + event.type];
    if (handler) {
      return handler.apply(this, arguments);
    }
  },

  /**
   * navigate between pages.
   *
   * @param {object} data Options for navigation.
   * @param {string} data.hash The hash of the panel id.  I.E. '#alarm-panel'.
   * @param {function} callback Callback to invoke when done.
   */
  navigate: function(data, callback) {
    var currentIndex = this.panels.indexOf(this.currentPanel);

    this.panels.forEach(function(panel, panelIndex) {
      if ('#' + panel.fragment === data.hash) {
        this.loadPanel(panel, function() {
          var instance = panel.instance;
          instance.navData = data.data || null;
          instance.active = true;
          instance.visible = true;
          if (currentIndex !== -1 && currentIndex !== panelIndex) {
            var direction = currentIndex < panelIndex;
            rAF(function startAnimation(oldPanel) {
              instance.transition =
                direction ? 'slide-in-right' : 'slide-in-left';

              oldPanel.instance.transition =
                direction ? 'slide-out-left' : 'slide-out-right';
            }.bind(null, this.currentPanel));
          }
          this.currentPanel = panel;
          callback && callback();
        }.bind(this));
      } else {
        if (panel.instance) {
          panel.instance.active = false;
        }
      }
    }, this);
    this.currentHash = data.hash;
  },

  /**
   * Navigate to the new hash.
   */
  onhashchange: function(event) {
    if (this.currentHash === location.hash) {
      return;
    }
    this.navigate({ hash: location.hash });
  },

  /**
   * Reset the global localization params on the html element.  Called when
   * the language changes, and once on application startup.
   */
  onlocalized: function(event) {
    document.documentElement.lang = mozL10n.language.code;
    document.documentElement.dir = mozL10n.language.direction;
  },

  /**
   * Whenever the application gains/loses focus, inform the current panel of
   * its visibility loss.
   */
  onvisibilitychange: function(event) {
    this.visible = !document.hidden;
    if (this.currentPanel) {
      this.currentPanel.visible = this.visible;
    }
  }
};

return App;

});



define('startup_init', ['require','app','l10n'],function(require) {

var App = require('app');
var mozL10n = require('l10n');
mozL10n.ready(App.init.bind(App));
});

require(['require_config'], function() {
  requirejs(['startup_init']);
});

define("startup", function(){});

define('panel',['require','view'],function(require) {

var View = require('view');
var priv = new WeakMap();

/**
 * A Panel is a "full screen" style tab/dialog.  Panels have an active state
 * which can be true or false, and can transition in and out using CSS3
 * classes and animation events.
 *
 * @constructor
 * @param {HTMLElement} element The element to wrap.
 */
function Panel(element) {
  View.apply(this, arguments);
  priv.set(this, {
    active: element.classList.contains('active'),
    transition: false
  });
  element.addEventListener('animationend', this);
}

Panel.prototype = Object.create(View.prototype);

/**
 * Handles the "animationend" event.  Sets the transition state to false
 * and hides the element if the Panel is not active.
 */
Panel.prototype.handleEvent = function(event) {
  if (event.target !== this.element) {
    return;
  }
  // remove visibility if transition finishes on non-active view
  if (!this.active) {
    this.visible = false;
  }
  this.transition = false;
};

Object.defineProperties(Panel.prototype, {
  /**
   * Panel.prototype.active - Boolean
   *
   * Sets the internal active state, and adds or removes the "active"
   * class on the element.
   */
  active: {
    get: function() {
      return priv.get(this).active;
    },
    set: function(value) {
      var state = priv.get(this);
      value = !!value;
      if (state.active !== value) {
        state.active = value;
        if (value) {
          this.element.classList.add('active');
        } else {
          this.element.classList.remove('active');
        }
        this.emit('active', value);
      }
      return value;
    }
  },
  /**
   * Panel.prototype.transition - String or false
   *
   * Sets the internal transition state.  When set, adds the class specified
   * to the element, removing the old transition class if it exists.
   *
   * When set to false, it removes the current transition class.
   */
  transition: {
    get: function() {
      return priv.get(this).transition;
    },
    set: function(value) {
      var state = priv.get(this);
      if (value) {
        if (state.transition) {
          this.element.classList.remove(state.transition);
        }
        this.element.classList.add(value);
      } else if (state.transition) {
        this.element.classList.remove(state.transition);
      }
      state.transition = value;
      return value;
    }
  }
});

return Panel;

});

/* -*- Mode: Java; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- /
/* vim: set shiftwidth=2 tabstop=2 autoindent cindent expandtab: */



/**
 * This file defines an asynchronous version of the localStorage API, backed by
 * an IndexedDB database.  It creates a global asyncStorage object that has
 * methods like the localStorage object.
 *
 * To store a value use setItem:
 *
 *   asyncStorage.setItem('key', 'value');
 *
 * If you want confirmation that the value has been stored, pass a callback
 * function as the third argument:
 *
 *  asyncStorage.setItem('key', 'newvalue', function() {
 *    console.log('new value stored');
 *  });
 *
 * To read a value, call getItem(), but note that you must supply a callback
 * function that the value will be passed to asynchronously:
 *
 *  asyncStorage.getItem('key', function(value) {
 *    console.log('The value of key is:', value);
 *  });
 *
 * Note that unlike localStorage, asyncStorage does not allow you to store and
 * retrieve values by setting and querying properties directly. You cannot just
 * write asyncStorage.key; you have to explicitly call setItem() or getItem().
 *
 * removeItem(), clear(), length(), and key() are like the same-named methods of
 * localStorage, but, like getItem() and setItem() they take a callback
 * argument.
 *
 * The asynchronous nature of getItem() makes it tricky to retrieve multiple
 * values. But unlike localStorage, asyncStorage does not require the values you
 * store to be strings.  So if you need to save multiple values and want to
 * retrieve them together, in a single asynchronous operation, just group the
 * values into a single object. The properties of this object may not include
 * DOM elements, but they may include things like Blobs and typed arrays.
 *
 * Unit tests are in apps/gallery/test/unit/asyncStorage_test.js
 */

this.asyncStorage = (function() {

  var DBNAME = 'asyncStorage';
  var DBVERSION = 1;
  var STORENAME = 'keyvaluepairs';
  var db = null;

  function withDatabase(f) {
    if (db) {
      f();
    } else {
      var openreq = indexedDB.open(DBNAME, DBVERSION);
      openreq.onerror = function withStoreOnError() {
        console.error("asyncStorage: can't open database:", openreq.error.name);
      };
      openreq.onupgradeneeded = function withStoreOnUpgradeNeeded() {
        // First time setup: create an empty object store
        openreq.result.createObjectStore(STORENAME);
      };
      openreq.onsuccess = function withStoreOnSuccess() {
        db = openreq.result;
        f();
      };
    }
  }

  function withStore(type, callback, oncomplete) {
    withDatabase(function() {
      var transaction = db.transaction(STORENAME, type);
      if (oncomplete) {
        transaction.oncomplete = oncomplete;
      }
      callback(transaction.objectStore(STORENAME));
    });
  }

  function getItem(key, callback) {
    var req;
    withStore('readonly', function getItemBody(store) {
      req = store.get(key);
      req.onerror = function getItemOnError() {
        console.error('Error in asyncStorage.getItem(): ', req.error.name);
      };
    }, function onComplete() {
      var value = req.result;
      if (value === undefined) {
        value = null;
      }
      callback(value);
    });
  }

  function setItem(key, value, callback) {
    withStore('readwrite', function setItemBody(store) {
      var req = store.put(value, key);
      req.onerror = function setItemOnError() {
        console.error('Error in asyncStorage.setItem(): ', req.error.name);
      };
    }, callback);
  }

  function removeItem(key, callback) {
    withStore('readwrite', function removeItemBody(store) {
      var req = store.delete(key);
      req.onerror = function removeItemOnError() {
        console.error('Error in asyncStorage.removeItem(): ', req.error.name);
      };
    }, callback);
  }

  function clear(callback) {
    withStore('readwrite', function clearBody(store) {
      var req = store.clear();
      req.onerror = function clearOnError() {
        console.error('Error in asyncStorage.clear(): ', req.error.name);
      };
    }, callback);
  }

  function length(callback) {
    var req;
    withStore('readonly', function lengthBody(store) {
      req = store.count();
      req.onerror = function lengthOnError() {
        console.error('Error in asyncStorage.length(): ', req.error.name);
      };
    }, function onComplete() {
      callback(req.result);
    });
  }

  function key(n, callback) {
    if (n < 0) {
      callback(null);
      return;
    }

    var req;
    withStore('readonly', function keyBody(store) {
      var advanced = false;
      req = store.openCursor();
      req.onsuccess = function keyOnSuccess() {
        var cursor = req.result;
        if (!cursor) {
          // this means there weren't enough keys
          return;
        }
        if (n === 0 || advanced) {
          // Either 1) we have the first key, return it if that's what they
          // wanted, or 2) we've got the nth key.
          return;
        }

        // Otherwise, ask the cursor to skip ahead n records
        advanced = true;
        cursor.advance(n);
      };
      req.onerror = function keyOnError() {
        console.error('Error in asyncStorage.key(): ', req.error.name);
      };
    }, function onComplete() {
      var cursor = req.result;
      callback(cursor ? cursor.key : null);
    });
  }

  return {
    getItem: getItem,
    setItem: setItem,
    removeItem: removeItem,
    clear: clear,
    length: length,
    key: key
  };
}());


define("shared/js/async_storage", (function (global) {
    return function () {
        var ret, fn;
        return ret || global.asyncStorage;
    };
}(this)));

/* -*- Mode: Java; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- /
/* vim: set shiftwidth=2 tabstop=2 autoindent cindent expandtab: */
(function(exports) {
  

  var priv = new WeakMap();
  var rmatcher = /\$\{([^}]+)\}/g;
  var rentity = /[&<>"']/g;
  var rentities = {
    '&': '&amp;',
    '<': '&lt;',
    '>': '&gt;',
    '"': '&quot;',
    '\'': '&apos;'
  };

  function extract(node) {
    var nodeId;
    // Received an ID string? Find the appropriate node to continue
    if (typeof node === 'string') {
      nodeId = node;
      node = document.getElementById(node);
    } else if (node) {
      nodeId = node.id;
    }

    if (!node) {
      console.error(
        'Can not find the node passed to Template', nodeId
      );
      return '';
    }

    // No firstChild means no comment node.
    if (!node.firstChild) {
      console.error(
        'Node passed to Template should have a comment node', nodeId
      );
      return '';
    }

    // Starting with the container node's firstChild...
    node = node.firstChild;

    do {
      // Check if it's the comment node that we're looking for...
      if (node.nodeType === Node.COMMENT_NODE) {
        return (node.nodeValue || '').trim();
      }
      // If the current child of the container node isn't
      // a comment node, it's likely a text node, so hop to
      // the nextSibling and repeat the operation.
    } while ((node = node.nextSibling));

    console.error(
      'Nodes passed to Template should have a comment node', nodeId
    );
    return '';
  }


  /**
   * Template
   *
   * Initialize a template instance from a string or node
   *
   * @param {String} idOrNode id string of existing node.
   *        {Object} idOrNode existing node.
   *
   */
  function Template(idOrNode) {
    if (!(this instanceof Template)) {
      return new Template(idOrNode);
    }
    // Storing the extracted template string as a private
    // instance property prevents direct access to the
    // template once it's been initialized.
    priv.set(this, {
      tmpl: extract(idOrNode)
    });
  };

  /**
   * template.toString()
   *
   * Safe, read-only access to the template string
   *
   */
  Template.prototype.toString = function() {
    // Return a copy of the stored template string.
    return priv.get(this).tmpl.slice();
  };

  /**
   * template.interpolate
   *
   * Interpolate template string with values provided by
   * data object. Optionally allow properties to retain
   * HTML that is known to be safe.
   *
   * @param {Object} data     properties correspond to substitution.
   *                          - identifiers in template string.
   * @param {Object} options  optional.
   *                          - safe, a list of properties that contain
   *                          HTML that is known and are
   *                          "known" to ignore.
   */
  Template.prototype.interpolate = function(data, options) {
    // This _should_ be rewritten to use Firefox's support for ES6
    // default parameters:
    // ... = function(data, options = { safe: [] }) {
    //
    options = options || {};
    options.safe = options.safe || [];

    return priv.get(this).tmpl.replace(rmatcher, function(match, property) {
      property = property.trim();
      // options.safe is an array of properties that can be ignored
      // by the "suspicious" html strategy.
      return options.safe.indexOf(property) === -1 ?
        // Any field that is not explicitly listed as "safe" is
        // to be treated as suspicious
        Template.escape(data[property]) :
        // Otherwise, return the string of rendered markup
        data[property];
    });
  };

  Template.escape = function escape(str) {
    if (typeof str !== 'string') {
      return '';
    }
    return str.replace(rentity, function(s) {
      return rentities[s];
    });
  };

  exports.Template = Template;

}(this));

define("shared/js/template", (function (global) {
    return function () {
        var ret, fn;
        return ret || global.Template;
    };
}(this)));

define('template',['require','shared/js/template'],function(require) {

var Template = require('shared/js/template');

function ClockTemplate(text) {
  var srcNode = document.createElement('div');
  var comment = document.createComment(text);
  srcNode.appendChild(comment);
  return new Template(srcNode);
}

return ClockTemplate;
});

define('utils',['require','l10n'],function(require) {


var mozL10n = require('l10n');

var Utils = {};
// Maintain references to millisecond multipliers
var dateMultipliers = {
  days: 1000 * 60 * 60 * 24,
  hours: 1000 * 60 * 60,
  minutes: 1000 * 60,
  seconds: 1000,
  milliseconds: 1
};
var units = Object.keys(dateMultipliers);

/**
 * Define a singleton method that returns a unified instance
 * based on arguments.
 *
 * @param {function} constructor - A constructor function used
 *        to create a new instance.
 * @param {function} [getKey] - A function called with (arguments),
 *        and returns a lookup key for this singleton.
 * @return {object} - returns the instance either created or retrieved
 *        from the singleton-map by the key.
 */
Utils.singleton = function(constructor, getKey) {
  var singletonMap = new Map();
  return function() {
    var arglist = Array.prototype.slice.call(arguments);
    var key = (typeof getKey === 'function') ? getKey(arglist) : constructor;
    var instance = singletonMap.get(key);
    if (!instance) {
      instance = Object.create(constructor.prototype);
      constructor.apply(instance, arglist);
      singletonMap.set(key, instance);
    }
    return instance;
  };
};

Utils.memoizedDomPropertyDescriptor = function(selector) {
  var memoizedValue = null;
  return {
    get: function() {
      if (memoizedValue === null) {
        memoizedValue = document.querySelectorAll(selector)[0];
      }
      return memoizedValue;
    },
    set: function(value) {
      memoizedValue = value;
    }
  };
};

Utils.dateMath = {
  /**
   * Convert object literals containing interval length to milliseconds
   *
   * @param {Object|Date|Number} interval An object literal containing days,
   *                                      hours, minutes etc.
   *                                      Optionally a number or date object.
   * @param {Object} opts Options object with a unitsPartial property containing
   *                      (if desired) a restriction on which properties will be
   *                      searched on the interval.
   * @return {Number} Millisecond value for interval length.
   */
  toMS: function(interval, opts) {
    var converted, sign, unitsPartial;

    // if a millisecond interval or a Date is passed in, return that
    if (interval instanceof Date || typeof interval === 'number') {
      return +interval;
    }

    opts = opts || {};
    unitsPartial = opts.unitsPartial || units;
    // Allow for 'hours' or 'hour'
    unitsPartial = unitsPartial.map(function(unit) {
      // String.prototype.endsWith is available in FF17+
      return unit.endsWith('s') ? unit : unit.concat('s');
    });

    // some will exit early when it returns a truthy value
    sign = unitsPartial.some(function(unit) {
      return interval[unit] < 0;
    });
    // Using as a multiplier later
    sign = sign ? -1 : 1;
    // collect passed in units and multiply by their millisecond/unit count
    converted = unitsPartial.map(function(unit) {
      var partial;
      // we're storing the sign out of the iterator
      partial = Math.abs(interval[unit]);
      // A missing property and 0 should be treated the same
      return partial ? partial * dateMultipliers[unit] : 0;
    });

    // add up each millisecond-converted term and multiply total by sign
    return sign * converted.reduce(function(a, b) { return a + b; });
  },
  /**
   * Convert millisecond values to object literals conformable to toMS()
   *
   * @param {Number} interval A millisecond value.
   * @param {Object} [opts] Options object with a unitsPartial property
   *                        containing (if desired) a restriction on which
   *                        properties will be reated for the return value.
   * @return {Object} Object literal with properties as deliniated by opts.
   */
  fromMS: function(interval, opts) {
    var times, sign, unitsPartial;

    opts = opts || {};
    unitsPartial = opts.unitsPartial || units;
    // Allow for 'hours' or 'hour'
    unitsPartial = unitsPartial.map(function(unit) {
      // String.prototype.endsWith is available in FF17+
      return unit.endsWith('s') ? unit : unit.concat('s');
    });
    // For negative intervals (time previous to now)
    // update interval to absolute value and store the sign
    // to apply to all units
    if (interval < 0) {
      sign = -1;
      interval = Math.abs(interval);
    } else {
      sign = 1;
    }

    // divide the time interval by the highest millisecond multiplier
    // store the truncated result and subtract that from the interval
    // update the interval to be the remainder
    times = unitsPartial.map(function(unit, index) {
      var truncated, mult;
      mult = dateMultipliers[unit];
      truncated = Math.floor(interval / mult);
      interval = interval - (truncated * mult);
      // units are either all positive or negative
      // only iterate to needed specificity
      return sign * truncated;
    });

    // Populate the returned object using units as property names
    // and times for values
    return times.reduce(function(out, unitTime, index) {
      out[unitsPartial[index]] = unitTime;
      return out;
    }, {});
  }
};

Utils.extend = function(initialObject, extensions) {
  // extend({}, a, b, c ... d) -> {...}
  // rightmost properties (on 'd') take precedence
  extensions = Array.prototype.slice.call(arguments, 1);
  for (var i = 0; i < extensions.length; i++) {
    var extender = extensions[i];
    for (var prop in extender) {
      if (Object.prototype.hasOwnProperty.call(extender, prop)) {
        initialObject[prop] = extender[prop];
      }
    }
  }
  return initialObject;
};

Utils.escapeHTML = function(str, escapeQuotes) {
  var span = document.createElement('span');
  span.textContent = str;

  if (escapeQuotes) {
    return span.innerHTML.replace(/"/g, '&quot;').replace(/'/g, '&#x27;');
  }
  return span.innerHTML;
};

Utils.is12hFormat = function() {
  var localeTimeFormat = mozL10n.get('dateTimeFormat_%X');
  var is12h = (localeTimeFormat.indexOf('%p') >= 0);
  return is12h;
};

Utils.getLocaleTime = function(d) {
  var f = new mozL10n.DateTimeFormat();
  var is12h = Utils.is12hFormat();
  return {
    t: f.localeFormat(d, (is12h ? '%I:%M' : '%H:%M')).replace(/^0/, ''),
    p: is12h ? f.localeFormat(d, '%p') : ''
  };
};

Utils.changeSelectByValue = function(selectElement, value) {
  var options = selectElement.options;
  for (var i = 0; i < options.length; i++) {
    if (options[i].value == value) {
      if (selectElement.selectedIndex != i) {
        selectElement.selectedIndex = i;
      }
      break;
    }
  }
};

Utils.getSelectedValueByIndex = function(selectElement) {
  return selectElement.options[selectElement.selectedIndex].value;
};

Utils.parseTime = function(time) {
  var parsed = time.split(':');
  var hour = +parsed[0]; // cast hour to int, but not minute yet
  var minute = parsed[1];

  // account for 'AM' or 'PM' vs 24 hour clock
  var periodIndex = minute.indexOf('M') - 1;
  if (periodIndex >= 0) {
    hour = (hour == 12) ? 0 : hour;
    hour += (minute.slice(periodIndex) == 'PM') ? 12 : 0;
    minute = minute.slice(0, periodIndex);
  }

  return {
    hour: hour,
    minute: +minute // now cast minute to int
  };
};

var wakeTarget = {
  requests: {
    cpu: new Map(), screen: new Map(), wifi: new Map()
  },
  locks: {
    cpu: null, screen: null, wifi: null
  },
  timeouts: {
    cpu: null, screen: null, wifi: null
  }
};
function getLongestLock(type) {
  var max = 0;
  for (var i of wakeTarget.requests[type]) {
    var request = i[1];
    if (request.time > max) {
      max = request.time;
    }
  }
  return {
    time: max,
    lock: wakeTarget.locks[type],
    timeout: wakeTarget.timeouts[type]
  };
}
Utils.safeWakeLock = function(opts, fn) {
    /*
     * safeWakeLock
     *
     * Create a Wake lock that is automatically released after
     * timeoutMs. Locks are reentrant, and have no meaningful mutual
     * exclusion behavior.
     *
     * @param {Object} options - an object containing
     *                 [type] {string} a string passed to requestWakeLock
     *                                 default = 'cpu'. This string can be any
     *                                 resource exposed by the environment that
     *                                 this application was designed to run in.
     *                                 Gaia exposes three of them: 'cpu',
     *                                 'screen', and 'wifi'. Certified apps may
     *                                 expose more.
     *                 timeoutMs {number} number of milliseconds to hold
     *                                    the lock.
     * @param {Function} callback - a function to be called after all other
     *                              generated callbacks have been called.
     *                              function ([err]) -> undefined.
     */
  opts = opts || {};
  var type = opts.type || 'cpu';
  var timeoutMs = opts.timeoutMs | 0;
  var now = Date.now();
  var myKey = {};
  wakeTarget.requests[type].set(myKey, {
    time: now + timeoutMs
  });
  var max = getLongestLock(type);
  var unlockFn = function() {
    if (!myKey) {
      return;
    }
    wakeTarget.requests[type]. delete(myKey);
    var now = Date.now();
    var max = getLongestLock(type);
    if (max.time > now) {
      clearTimeout(wakeTarget.timeouts[type]);
      wakeTarget.timeouts[type] = setTimeout(unlockFn, max.time - now);
    } else {
      if (wakeTarget.locks[type]) {
        wakeTarget.locks[type].unlock();
      }
      wakeTarget.locks[type] = null;
      clearTimeout(wakeTarget.timeouts[type]);
      wakeTarget.timeouts[type] = null;
    }
    myKey = null;
  };
  clearTimeout(wakeTarget.timeouts[type]);
  wakeTarget.timeouts[type] = setTimeout(unlockFn, max.time - now);
  try {
    if (!wakeTarget.locks[type] && max.time > now) {
      wakeTarget.locks[type] = navigator.requestWakeLock(type);
    }
    fn(unlockFn);
  } catch (err) {
    unlockFn();
    throw err;
  }
};

Utils.repeatString = function rep(str, times) {
  var built = [], cur = str;
  for (var i = 0, j = 1; j <= times; i++) {
    if ((times & j) > 0) {
      built.push(cur);
    }
    cur = cur + cur;
    j = j << 1;
  }
  return built.join('');
};

Utils.format = {
  time: function(hour, minute, opts) {
    var period = '';
    opts = opts || {};
    opts.meridian = typeof opts.meridian === 'undefined' ? true : opts.meridian;
    var padHours = typeof opts.padHours === 'undefined' ? false : opts.padHours;
    opts.padHours = padHours;

    if (opts.meridian && Utils.is12hFormat()) {
      period = hour < 12 ? 'AM' : 'PM';
      hour = hour % 12;
      hour = (hour === 0) ? 12 : hour;
    }

    if (opts.padHours && hour < 10) {
      hour = '0' + hour;
    }

    if (hour === 0) {
      hour = '00';
    }

    if (minute < 10) {
      minute = '0' + minute;
    }

    return hour + ':' + minute + period;
  },
  hms: function(sec, format) {
    var hour = 0;
    var min = 0;

    if (sec >= 3600) {
      hour = Math.floor(sec / 3600);
      sec -= hour * 3600;
    }

    if (sec >= 60) {
      min = Math.floor(sec / 60);
      sec -= min * 60;
    }

    hour = (hour < 10) ? '0' + hour : hour;
    min = (min < 10) ? '0' + min : min;
    sec = (sec < 10) ? '0' + sec : sec;

    if (typeof format !== 'undefined') {
      format = format.replace('hh', hour);
      format = format.replace('mm', min);
      format = format.replace('ss', sec);

      return format;
    }
    return hour + ':' + min + ':' + sec;
  },
  durationMs: function(ms) {
    var dm = Utils.dateMath.fromMS(ms, {
      unitsPartial: ['minutes', 'seconds', 'milliseconds']
    });
    var puts = function(x, n) {
      x = String(x);
      return Utils.repeatString('0', Math.max(0, n - x.length)) + x;
    };
    return [
      puts(dm.minutes, 2), ':',
      puts(dm.seconds, 2), '.',
      puts((dm.milliseconds / 10) | 0, 2)
    ].join('');
  }
};


Utils.async = {

  generator: function(latchCallback) {
    /*
     * Generator
     *
     * Create an async generator. Each time the generator is
     * called, it will return a new callback. When all issued
     * callbacks have been called, the latchCallback is called.
     *
     * If any of the callbacks are called with and error as
     * the first argument, the latchCallback will be called
     * immediately with that error.
     *
     * @latchCallback {Function} a function to be called after
     *           all other generated callbacks have been
     *           called
     *           function ([err]) -> undefined
     */
    var tracker = new Map();
    var issuedCallbackCount = 0;
    var disabled = false;
    var testFn = function(err) {
      var trackerSize;
      if (!disabled) {
        // FF18 defines size to be a method, so we need to test here:
        // Remove with FF18 support
        if (typeof tracker.size === 'function') {
          trackerSize = tracker.size();
        } else {
          trackerSize = tracker.size;
        }
        if (err || trackerSize === issuedCallbackCount) {
          disabled = true;
          latchCallback && latchCallback(err);
        }
      }
    };
    return function() {
      return (function() {
        var i = issuedCallbackCount++;
        return function(err) {
          tracker.set(i, true);
          testFn(err);
        };
      })();
    };
  },

  namedParallel: function(names, latchCallback) {
    /*
     * namedParallel
     *
     * Create an async namedParallel.
     *
     * The return value is an object containing the parameters
     * specified in the names array. Each parameter is set to
     * a callback. When all callbacks have been called, latchCallback
     * is called.
     *
     * If any named callback is called with an error as the first
     * parameter, latchCallback is immediately called with that
     * error. Future calls to callbacks are then no-ops.
     *
     * @names {List<String>} - A list of strings to be used as
     *        parameter names for callbacks on the returned object.
     */
    var generator = Utils.async.generator(latchCallback);
    var done = generator();
    var ret = {};
    for (var i = 0; i < names.length; i++) {
      ret[names[i]] = generator();
    }
    done();
    return ret;
  }

};

Utils.data = {

  defaultCompare: function ud_defaultCompare(a, b) {
    if (typeof a === 'number' && typeof b === 'number') {
      var diff = a - b;
      return diff !== 0 ? diff / Math.abs(diff) : diff;
    } else if ((typeof a === 'string' || a instanceof String) &&
               (typeof b === 'string' || b instanceof String)) {
      return (a < b) ? -1 : ((a > b) ? 1 : 0);
    } else if (Array.isArray(a) && Array.isArray(b)) {
      var commonLength = Math.min(a.length, b.length);
      for (var i = 0; i < commonLength; i++) {
        var compareResult = Utils.data.defaultCompare(a[i], b[i]);
        if (compareResult !== 0) {
          return compareResult;
        }
      }
      return b.length - a.length;
    } else {
      throw new Error('Cannot compare ' + JSON.stringify([a, b]));
    }
  },

  keyedCompare: function ud_keyedCompare(key) {
    return function internal_keyedCompare(a, b) {
      return Utils.data.defaultCompare(a[key], b[key]);
    };
  },

  binarySearch: function ud_binarySearch(key, list, compare) {
    compare = compare || Utils.data.defaultCompare;
    var botI = 0;
    var topI = list.length;
    var midI, comp, value;
    if (list.length > 0) {
      do {
        midI = botI + ((topI - botI) / 2) | 0;
        value = list[midI];
        comp = compare(value, key);
        if (comp < 0) {
          botI = midI + 1;
        } else if (comp > 0) {
          topI = midI - 1;
        }
      } while (comp !== 0 && botI < topI);
    }
    midI = comp === 0 ? midI : topI;
    value = list[midI];
    if (comp === 0 || value && compare(value, key) === 0) {
      return { match: true, index: midI, value: value };
    } else {
      var index;
      if (0 > midI) {
        index = 0;
      } else if (midI >= list.length) {
        index = list.length;
      } else {
        index = midI + (compare(value, key) < 0 ? 1 : 0);
      }
      return {
        match: false,
        index: index
      };
    }
  },

  sortedInsert: function ud_sortedInsert(item, list, compare, unique) {
    compare = compare || Utils.data.defaultCompare;
    var bs = Utils.data.binarySearch(item, list, compare);
    var inserted = !bs.match || !unique;
    if (inserted) {
      list.splice(bs.index, 0, item);
    }
    return (inserted) ? bs.index : null;
  },

  sortedRemove: function ud_sortedRemove(item, list, compare, multiple) {
    compare = compare || Utils.data.defaultCompare;
    var removed = false;
    if (!multiple) {
      var bs = Utils.data.binarySearch(item, list, compare);
      if (bs.match) {
        list.splice(bs.index, 1);
        removed = true;
      }
    } else {
      var biasedCompare = function(target, result, ic) {
        return function(a, b) {
          if (ic(a, target) === 0) {
            return result;
          } else {
            return ic(a, b);
          }
        };
      };
      var leftBound = Utils.data.binarySearch(item, list,
        biasedCompare(item, 1, compare));
      var rightBound = Utils.data.binarySearch(item, list,
        biasedCompare(item, -1, compare));
      if (leftBound.index < rightBound.index) {
        list.splice(leftBound.index, rightBound.index - leftBound.index);
        removed = true;
      }
    }
    return removed;
  }
};

return Utils;

});

define('text',{
  pluginBuilder: './text_builder',
  load: function(name, req, onload, config) {
    var url = req.toUrl(name),
        xhr = new XMLHttpRequest();

    xhr.open('GET', url, true);
    xhr.onreadystatechange = function(evt) {
      var status, err;
      if (xhr.readyState === 4) {
        status = xhr.status;
        if (status > 399 && status < 600) {
          //An http 4xx or 5xx error. Signal an error.
          err = new Error(url + ' HTTP status: ' + status);
          err.xhr = xhr;
          onload.error(err);
        } else {
          onload(xhr.responseText);
        }
      }
    };
    xhr.responseType = 'text';
    xhr.send(null);
  }
});


define('text!banner/banner.html',[],function () { return '<p>${notice}</p>\n';});

define('banner/main',['require','template','utils','l10n','text!banner/banner.html'],function(require) {
  

  var Template = require('template');
  var Utils = require('utils');
  var mozL10n = require('l10n');
  var html = require('text!banner/banner.html');

  function Banner(node) {
    // Accept a reference to an element or the element id
    if (typeof node === 'string') {
      this.notice = document.getElementById(node);
    } else {
      this.notice = node;
    }
    // Accept an optional reference to template element id
    this.tmpl = new Template(html);
    // Store a reference to timeout to debounce banner
    this.timeout = null;
    return this;
  }

  Banner.prototype = {

    constructor: Banner,

    render: function bn_render(alarmTime) {
      var timeLeft, tl, countdownType, localTimes, unitObj;

      timeLeft = +alarmTime - Date.now();
      // generate human readable numbers to pass to localization function
      tl = Utils.dateMath.fromMS(timeLeft, {
        unitsPartial: ['days', 'hours', 'minutes']
      });

      // Match properties to localizations string types
      // e.g. minutes maps to nMinutes if there are no hours but
      // nRemainMinutes if hours > 0
      if (tl.days) {
        //countdown-moreThanADay localized only for en-US while 913466 is open
        countdownType = 'countdown-moreThanADay';
        localTimes = [
          ['days', 'nRemainDays', tl.days],
          ['hours', 'nAndRemainHours', tl.hours]
        ];
      } else if (tl.hours > 0) {
        countdownType = 'countdown-moreThanAnHour';
        localTimes = [
          ['hours', 'nHours', tl.hours],
          ['minutes', 'nRemainMinutes', tl.minutes]
        ];
      } else {
        countdownType = 'countdown-lessThanAnHour';
        localTimes = [
          ['minutes', 'nMinutes', tl.minutes]
        ];
      }

      // Create an object to pass to mozL10n.get
      // e.g. {minutes: mozL10n.get('nMinutes', {n: 3})}
      unitObj = localTimes.reduce(function(lcl, time) {
        lcl[time[0]] = mozL10n.get(time[1], {n: time[2]});
        return lcl;
      }, {});

      // mozL10n.get interpolates the units in unitObj inside the
      // localization string for countdownType
      return mozL10n.get(countdownType, unitObj);
    },

    show: function bn_show(alarmTime) {
      // Render the Banner notice
      this.notice.innerHTML = this.tmpl.interpolate(
        {notice: this.render(alarmTime)},
        // Localization strings contain <strong> tags
        {safe: ['notice']}
      );
      // 'visible' class controls the animation
      this.notice.classList.add('visible');
      // use this object rather than a function to retain context
      this.notice.addEventListener('click', this);
      // Debounce timer in case alarms are added more quickly than 4 seconds
      if (this.timeout) {
        clearTimeout(this.timeout);
      }
      // After 4 seconds, remove the banner
      this.timeout = setTimeout(this.hide.bind(this), 4000);
    },

    hide: function bn_hide() {
      this.notice.classList.remove('visible');
      this.notice.removeEventListener('click', this);
    },

    handleEvent: function bn_handleEvent() {
      this.hide();
    }
  };

  return Banner;
});

define('constants',['require','exports','module'],function(require, exports) {
  

  // ---------------------------------------------------------
  // Constants

  exports.DAYS = [
    'monday', 'tuesday', 'wednesday', 'thursday', 'friday',
    'saturday', 'sunday'
  ];

  exports.RDAYS = exports.DAYS.map(function(_, n) {
    return n;
  });

  exports.WEEKDAYS = [0, 1, 2, 3, 4].map(function(x) {
    return exports.DAYS[x];
  });

  exports.WEEKENDS = [5, 6].map(function(x) {
    return exports.DAYS[x];
  });

});

define('alarm',['require','exports','module','alarmsdb','utils','constants','l10n'],function(require, exports, module) {

  

  var AlarmsDB = require('alarmsdb');
  var Utils = require('utils');
  var constants = require('constants');
  var mozL10n = require('l10n');

  // define WeakMaps for protected properties
  var protectedProperties = (function() {
    var protectedWeakMaps = new Map();
    ['id', 'repeat', 'registeredAlarms'].forEach(function(x) {
      protectedWeakMaps.set(x, new WeakMap());
    });
    return protectedWeakMaps;
  })();

  // define variables
  var validPropertiesSet = null; // memoizes validProperties() method
  var idMap = protectedProperties.get('id');
  var repeatMap = protectedProperties.get('repeat');
  var registeredAlarmsMap = protectedProperties.get('registeredAlarms');

  // ---------------------------------------------------------
  // Alarm Object

  function Alarm(config) {
    if (config instanceof Alarm) {
      config = config.toSerializable();
    }
    var econfig = Utils.extend(this.defaultProperties(), config || {});
    this.extractProtected(econfig);
    Utils.extend(this, econfig);
  }

  Alarm.prototype = {

    constructor: Alarm,

    // ---------------------------------------------------------
    // Initialization methods

    extractProtected: function(config) {
      var valids = this.validProperties();
      for (var i in config) {
        if (protectedProperties.has(i)) {
          var map = protectedProperties.get(i);
          map.set(this, config[i]);
          delete config[i];
        }
        if (!valids.has(i)) {
          delete config[i];
        }
      }
    },

    defaultProperties: function() {
      var now = new Date();
      return {
        registeredAlarms: {}, // set -> this.schedule & this.cancel
        repeat: {},
        hour: now.getHours(),
        minute: now.getMinutes(),

        // Raw Fields
        label: '',
        sound: 'ac_classic_clock_alarm.opus',
        vibrate: 1,
        snooze: 5,
        color: 'Darkorange'
      };
    },

    validProperties: function() {
      if (validPropertiesSet !== null) {
        return new Set(validPropertiesSet);
      }
      var ret = new Set();
      var keys = Object.keys(this.defaultProperties());
      keys = keys.concat(['id']);
      for (var i in keys) {
        ret.add(keys[i]);
      }
      validPropertiesSet = ret;
      return new Set(ret);
    },

    // ---------------------------------------------------------
    // Persisted form

    toSerializable: function alarm_toSerializable() {
      var retval = {};
      for (var i in this) {
        if (this.hasOwnProperty(i)) {
          retval[i] = this[i];
        }
      }
      for (var kv of protectedProperties) {
        var prop = kv[0], map = kv[1];
        if (map.has(this) && map.get(this) !== undefined) {
          retval[prop] = map.get(this);
        }
      }
      return retval;
    },

    // ---------------------------------------------------------
    // Getters and Setters

    set time(x) {
      // destructure passed array
      this.minute = +x[1];
      this.hour = +x[0];
    },

    get time() {
      return [this.hour, this.minute];
    },

    get id() {
      return idMap.get(this) || undefined;
    },

    // this is needed because the unit tests need to set ID,
    // and 'use strict' forbids setting if there's a getter
    set id(id) {
      idMap.set(this, id);
    },

    get registeredAlarms() {
      return registeredAlarmsMap.get(this) || {};
    },

    set repeat(x) {
      var rep = {};
      for (var y of constants.DAYS) {
        if (x[y] === true) {
          rep[y] = true;
        }
      }
      repeatMap.set(this, rep);
    },

    get repeat() {
      return repeatMap.get(this);
    },

    set enabled(x) {
      throw 'use setEnabled to set (async requires callback)';
    },

    get enabled() {
      for (var i in this.registeredAlarms) {
        if (i === 'normal') {
          return true;
        }
      }
      return false;
    },

    // ---------------------------------------------------------
    // Time Handling

    summarizeDaysOfWeek: function alarm_summarizeRepeat() {
      var _ = mozL10n.get;
      var i, dayName;
      // Build a bitset
      var value = 0;
      for (i = 0; i < constants.DAYS.length; i++) {
        dayName = constants.DAYS[i];
        if (this.repeat[dayName] === true) {
          value |= (1 << i);
        }
      }
      var summary;
      if (value === 127) { // 127 = 0b1111111
        summary = _('everyday');
      } else if (value === 31) { // 31 = 0b0011111
        summary = _('weekdays');
      } else if (value === 96) { // 96 = 0b1100000
        summary = _('weekends');
      } else if (value !== 0) { // any day was true
        var weekdays = [];
        for (i = 0; i < constants.DAYS.length; i++) {
          dayName = constants.DAYS[i];
          if (this.repeat[dayName]) {
            // Note: here, Monday is the first day of the week
            // whereas in JS Date(), it's Sunday -- hence the (+1) here.
            weekdays.push(_('weekday-' + ((i + 1) % 7) + '-short'));
          }
          summary = weekdays.join(', ');
        }
      } else { // no day was true
        summary = _('never');
      }
      return summary;
    },

    isAlarmPassedToday: function alarm_isAlarmPassedToday() {
      var now = new Date();
      if (this.hour > now.getHours() ||
           (this.hour === now.getHours() &&
            this.minute > now.getMinutes())) {
        return false;
      }
      return true;
    },

    isDateInRepeat: function alarm_isDateInRepeat(date) {
      // return true if repeat contains date
      var day = constants.DAYS[(date.getDay() + 6) % 7];
      return !!this.repeat[day];
    },

    repeatDays: function alarm_repeatDays() {
      var count = 0;
      for (var i in this.repeat) {
        if (this.repeat[i]) {
          count++;
        }
      }
      return count;
    },

    isRepeating: function alarm_isRepeating() {
      return this.repeatDays() !== 0;
    },

    getNextAlarmFireTime: function alarm_getNextAlarmFireTime() {
      var now = new Date(), nextFire = new Date();
      nextFire.setHours(this.hour, this.minute, 0, 0);
      while (nextFire <= now ||
              !(this.repeatDays() === 0 ||
                this.isDateInRepeat(nextFire))) {
        nextFire.setDate(nextFire.getDate() + 1);
      }
      return nextFire;
    },

    getNextSnoozeFireTime: function alarm_getNextSnoozeFireTime() {
      if (this.snooze && (typeof this.snooze) === 'number') {
        var now = new Date();
        now.setMinutes(now.getMinutes() + this.snooze);
        return now;
      }
      return null;
    },

    // ---------------------------------------------------------
    // Wholistic methods (Alarm API and Database)

    setEnabled: function alarm_setEnabled(value, callback) {
      if (value) {
        var scheduleWithID = function(err, alarm) {
          this.schedule({
            type: 'normal',
            first: true
          }, this.saveCallback(callback));
        };
        if (!this.id) {
          // if we don't have an ID yet, save to IndexedDB to
          // get one, and then call scheduleWithID
          this.save(scheduleWithID.bind(this));
        } else {
          // otherwise, just call scheduleWithID
          setTimeout(scheduleWithID.bind(this, null, this), 0);
        }
      } else if (this.enabled) {
        this.cancel();
        this.save(callback);
      } else if (callback) {
        setTimeout(callback.bind(undefined, null, this), 0);
      }
    },

    delete: function alarm_delete(callback) {
      this.cancel();
      AlarmsDB.deleteAlarm(this.id,
        function alarm_innerDelete(err, alarm) {
        callback(err, this);
      }.bind(this));
    },

    // ---------------------------------------------------------
    // Database Integration

    saveCallback: function alarm_saveCallback(callback) {
      return function(err, value) {
        if (!err) {
          this.save(callback);
        } else {
          if (callback) {
            callback(err, value);
          }
        }
      }.bind(this);
    },

    save: function alarm_save(callback) {
      AlarmsDB.putAlarm(this, function(err, alarm) {
        idMap.set(this, alarm.id);
        callback && callback(err, this);
      }.bind(this));
    },

    // ---------------------------------------------------------
    // Alarm API

    scheduleHelper: function alarm_scheduleHelper(type, date, callback) {
      var data = {
        id: this.id,
        type: type
      };
      var request = navigator.mozAlarms.add(
        date, 'ignoreTimezone', data);
      request.onsuccess = (function(ev) {
        var registeredAlarms = registeredAlarmsMap.get(this) || {};
        registeredAlarms[type] = ev.target.result;
        registeredAlarmsMap.set(this, registeredAlarms);
        if (callback) {
          callback(null, this);
        }
      }).bind(this);
      request.onerror = function(ev) {
        if (callback) {
          callback(ev.target.error);
        }
      };
    },

    schedule: function alarm_schedule(options, callback) {
      /*
       * Schedule
       *
       * Schedule a mozAlarm to wake up the app at a certain time.
       *
       * @options {Object} an object containing parameters for the
       *                   scheduled alarm.
       *          - type: 'normal' or 'snooze'
       *          - first: {boolean}
       *
       * First is used true when an alarm is "first" in a sequence
       * of repeating normal alarms.
       * For no-repeat alarms, the sequence of length 1, and so
       * the alarm is always first.
       * Snooze alarms are never first, since they have a normal
       * alarm parent.
       *
       */
      options = options || {}; // defaults
      if (typeof options.type === 'undefined') {
        options.type = 'normal';
      }
      if (typeof options.first === 'undefined') {
        options.first = true;
      }
      if (!options.first && !this.isRepeating()) {
        this.cancel('normal');
        callback(null, this);
        return;
      }
      this.cancel(options.type);
      var firedate;
      if (options.type === 'normal') {
        firedate = this.getNextAlarmFireTime();
      } else if (options.type === 'snooze') {
        firedate = this.getNextSnoozeFireTime();
      }
      this.scheduleHelper(options.type, firedate, callback);
    },

    cancel: function alarm_cancel(cancelType) {
      // cancel an alarm type ('normal' or 'snooze')
      // type == false to cancel all
      function removeAlarm(type, id) {
        /* jshint validthis:true */
        navigator.mozAlarms.remove(id);
        var registeredAlarms = this.registeredAlarms;
        delete registeredAlarms[type];
        registeredAlarmsMap.set(this, registeredAlarms);
      }
      if (!cancelType) {
        for (var type in this.registeredAlarms) {
          removeAlarm.call(this, type, this.registeredAlarms[type]);
        }
      } else {
        removeAlarm.call(this, cancelType, this.registeredAlarms[cancelType]);
      }
    }

  };

  // ---------------------------------------------------------
  // Export

  module.exports = Alarm;

});

define('alarmsdb',['require','exports','module','utils','alarm','alarm'],function(require, exports) {


var Utils = require('utils');

var BaseIndexDB = function(objectStoreOptions, upgradeHandler) {

  this.query = function ad_query(dbName, storeName, func, callback, data) {
    var indexedDB = window.indexedDB || window.webkitIndexedDB ||
        window.mozIndexedDB || window.msIndexedDB;

    var upgradeRequired = false;

    var request = indexedDB.open(dbName, 6);

    request.onsuccess = (function(event) {
      if (upgradeRequired && typeof upgradeHandler === 'function') {
        request.result.close();
        upgradeHandler(function(err) {
          if (!err) {
            // retry query to avoid transaction issues
            this.query(dbName, storeName, func, callback, data);
          } else {
            console.log('Error during database upgrade:', err.message);
          }
        }.bind(this));
      } else {
        func(request.result, storeName, callback, data);
      }
    }).bind(this);

    request.onerror = function(event) {
      console.error('Can\'t open database', dbName, event);
    };

    // DB init
    request.onupgradeneeded = function(event) {
      console.log('Upgrading db');
      upgradeRequired = true;
      var db = event.target.result;
      if (!db.objectStoreNames.contains(storeName)) {
        db.createObjectStore(storeName, objectStoreOptions);
      }
      console.log('Upgrading db done');
    };
  };

  this.put = function ad_put(database, storeName, callback, item) {
    var txn = database.transaction(storeName, 'readwrite');
    var store = txn.objectStore(storeName);
    var putreq = store.put(item);

    putreq.onsuccess = function(event) {
      item.id = event.target.result;
      callback && callback(null, item);
    };

    putreq.onerror = function(e) {
      callback && callback({
        database: database,
        store: storeName,
        message: e.message,
        code: putreq.errorCode
      });
    };
  };

  this.load = function ad_load(database, storeName, callback) {
    var alarms = [];
    var txn = database.transaction(storeName);
    var store = txn.objectStore(storeName);
    var cursor = store.openCursor(null, 'prev');

    cursor.onsuccess = function(event) {
      var item = event.target.result;
      if (item) {
        alarms.push(item.value);
        item.continue();
      } else {
        txn.db.close();
        callback && callback(null, alarms);
      }
    };

    cursor.onerror = function(event) {
      callback && callback(event);
    };
  };

  this.get = function ad_get(database, storeName, callback, key) {
    var txn = database.transaction(storeName);
    var store = txn.objectStore(storeName);
    var request = store.get(key);

    request.onsuccess = function(event) {
      txn.db.close();
      callback && callback(null, request.result);
    };

    request.onerror = function(event) {
      callback && callback({
        database: database,
        store: storeName,
        message: event.message,
        code: request.errorCode
      });
    };
  };

  this.delete = function ad_delete(database, storeName, callback, key) {

    var txn = database.transaction(storeName, 'readwrite');
    var store = txn.objectStore(storeName);
    var request = store.delete(key);

    request.onsuccess = function(e) {
      txn.db.close();
      callback && callback(null, e);
    };

    request.onerror = function(e) {
      callback && callback({
        database: database,
        store: storeName,
        message: event.message,
        code: request.errorCode
      });
    };
  };
};

exports.DBNAME = 'alarms';
exports.STORENAME = 'alarms';

  // Database methods
  exports.getAlarmList = function ad_getAlarmList(callback) {
    function getAlarmList_mapper(err, list) {
      callback(err, (list || []).map(function(x) {
        return new (require('alarm'))(x);
      }));
    }
    this.query(this.DBNAME, this.STORENAME, this.load, getAlarmList_mapper);
  };

function convertTo12(alarm) {
  // Detect the version and return a correct 1.2 serializable.
  var ret = Utils.extend({
    registeredAlarms: {},
    repeat: {}
  }, alarm);
  if (typeof alarm.enabled !== 'undefined') {
    delete ret.enabled;
  }
  // Extract a normalAlarmId
  if (typeof alarm.normalAlarmId !== 'undefined') {
    ret.registeredAlarms.normal = alarm.normalAlarmId;
    delete ret.normalAlarmId;
  }
  // Extract a snoozeAlarmId
  if (typeof alarm.snoozeAlarmId !== 'undefined') {
    ret.registeredAlarms.snooze = alarm.snoozeAlarmId;
    delete ret.snoozeAlarmId;
  }
  // Map '1111100' string bitmap to a 1.2 repeat object with day name
  // properties.
  if (typeof alarm.repeat === 'string') {
    var days = ['monday', 'tuesday', 'wednesday', 'thursday', 'friday',
                'saturday', 'sunday'];
    ret.repeat = {};
    for (var i = 0; i < alarm.repeat.length && i < days.length; i++) {
      if (alarm.repeat[i] === '1') {
        ret.repeat[days[i]] = true;
      }
    }
  } else {
    ret.repeat = Utils.extend({}, alarm.repeat);
  }
  return ret;
}

  /**
   * convertAlarms - converts from v1.0 or v1.1 alarm representation to 1.2.
   *
   * @param {Function} callback Called when the conversion completes, with
   *                            (err).
   */
  exports.convertAlarms = function ad_convertAlarms(callback) {
    console.log('Converting alarms to new database storage');
    var gen = Utils.async.generator(function(err) {
      // All done, call the callback.
      console.log('Conversion complete', JSON.stringify(err));
      callback && callback(err);
    });
    var done = gen();
    this.query(this.DBNAME, this.STORENAME, this.load, function(err, list) {
      if (err) {
        done(err);
        return;
      }
      for (var i = 0; i < list.length; i++) {
        this.query(this.DBNAME, this.STORENAME, this.put, gen(),
          convertTo12(list[i]));
      }
      done();
    }.bind(exports));
  };

  exports.putAlarm = function ad_putAlarm(alarm, callback) {
    this.query(this.DBNAME, this.STORENAME, this.put, callback,
      alarm.toSerializable());
  };

  exports.getAlarm = function ad_getAlarm(key, callback) {
    this.query(this.DBNAME, this.STORENAME, this.get,
      function(err, result) {
        callback(err, new (require('alarm'))(result));
      }, key);
  };

  exports.deleteAlarm = function ad_deleteAlarm(key, callback) {
    this.query(this.DBNAME, this.STORENAME, this.delete, callback, key);
  };

Utils.extend(exports, new BaseIndexDB({
  keyPath: 'id',
  autoIncrement: true
}, exports.convertAlarms.bind(exports)));

});

define('alarm_manager',['require','utils','alarmsdb'],function(require) {

/* An Alarm's ID:
 * ID in Clock app                              ID in mozAlarms API
 * id (unique)                                  id (unique)
 *                                              type: 'normal' or 'snooze'
 *
 *
 * An alarm has its own id in the Clock app's indexDB(alarmsdb.js).
 * In order to maintain(add,remove) an alarm by mozAlarms API,
 * we prepare an registeredAlarms object that contains each alarm type:
 * 'snooze' and 'normal'
 *
 * In order to identify the active alarm which comes from mozAlarms API,
 * we pass id and type in JSON object data during adding an alarm by API.
 * id:    sync with each alarm's own id in Clock app's
 * type:  'normal', 'snooze' corresponding alarm type
 *
 *
 * An Alarm's Life:
 * We maintain an alarm's life cycle immediately when the alarm goes off.
 * If user click the snooze button when the alarm goes off,
 * we request a snooze alarm immediately.
 *
 *
 * Example:
 * (): set a alarm in start state
 * []: alarm goes off
 * O:  an once alarm
 * R:  a repeatable alarm
 * S:  a snooze alarm
 *
 * ====>: the flow of normal alarm
 * ---->: the flow of snooze alarm
 * |:     User click the snooze button
 *
 * Flow map:
 * i.  Once Alarm:
 *     (O) ====> [O]
 *
 * or  (O) ====> [O]
 *                |
 *                |  ----> [S] ----> [S]
 *
 *
 * ii. Repeat Alarm:
 *     (R) ====> [R] ====> [R]
 *                |
 *                |  ----> [S] ----> [S]
 *
 *                                              |  ----> [S] ----> [S]
 *                          |  ----> [S]        |
 *                          |                   |
 * or  (R) ====> [R] ====> [R] ====> [R] ====> [R] ====> [R] ====> [R]
 *                |
 *                |  ----> [S] ----> [S]
 */

var Utils = require('utils');
var AlarmsDB = require('alarmsdb');

var AlarmManager = {

  toggleAlarm: function am_toggleAlarm(alarm, enabled, callback) {
    alarm.setEnabled(enabled, callback);
  },

  updateAlarmStatusBar: function am_updateAlarmStatusBar() {
    /* jshint loopfunc:true */
    var request = navigator.mozAlarms.getAll();
    request.onsuccess = function(e) {
      var hasAlarmEnabled = false;
      var generator = Utils.async.generator(function(err) {
        if (!err && navigator.mozSettings) {
          navigator.mozSettings.createLock().set({
            'alarm.enabled': hasAlarmEnabled
          });
        }
      });
      var endCb = generator();
      for (var i = 0; i < e.target.result.length && !hasAlarmEnabled; i++) {
        var data = e.target.result[i].data;
        if (!data.id || ['normal', 'snooze'].indexOf(data.type) === -1) {
          return;
        }
        AlarmsDB.getAlarm(data.id,
          (function(mozAlarm, doneCb) {
          return function(err, alarm) {
            if (!err) {
              for (var j in alarm.registeredAlarms) {
                if (alarm.registeredAlarms[j] === mozAlarm.id) {
                  hasAlarmEnabled = true;
                }
              }
            }
            doneCb();
          };
        })(e.target.result[i], generator()));
      }
      endCb();
    };
    request.onerror = function(e) {
      console.error('get all alarm fail');
    };
  },

  regUpdateAlarmEnableState: function am_regUpdateAlarmEnableState(handler) {
    this._updateAlarmEableStateHandler = handler;
  }

};

return AlarmManager;
});

define('text!panels/alarm/list_item.html',[],function () { return '<label class="alarmList alarmEnable">\n  <input class="input-enable" data-id="${id}" type="checkbox" ${checked}>\n  <span></span>\n</label>\n<a href="#alarm-edit-panel" class="alarm-item${withRepeat}" data-id="${id}">\n  <span class="time">\n    ${time} <span class="period">${meridian}</span>\n  </span>\n  <span class="label">${label}</span>\n  <span class="repeat">${repeat}</span>\n</a>\n';});

define('panels/alarm/alarm_list',['require','banner/main','alarmsdb','alarm_manager','utils','template','l10n','app','text!panels/alarm/list_item.html'],function(require) {


var Banner = require('banner/main');
var AlarmsDB = require('alarmsdb');
var AlarmManager = require('alarm_manager');
var Utils = require('utils');
var Template = require('template');
var mozL10n = require('l10n');
var App = require('app');
var alarmHtml = require('text!panels/alarm/list_item.html');

var _ = mozL10n.get;

var AlarmList = {

  alarmList: [],

  // Lookup table mapping alarm IDs to the number "toggle" operations currently
  // in progress.
  toggleOperations: {},
  count: 0,

  get alarms() {
    delete this.alarms;
    return (this.alarms = document.getElementById('alarms'));
  },

  get title() {
    delete this.title;
    return (this.title = document.getElementById('alarms-title'));
  },

  get newAlarmButton() {
    delete this.newAlarmButton;
    return (this.newAlarmButton = document.getElementById('alarm-new'));
  },

  template: null,

  handleEvent: function al_handleEvent(evt) {

    var link = evt.target;
    if (!link) {
      return;
    }

    if (link === this.newAlarmButton) {
      this.alarmEditView();
      evt.preventDefault();
    } else if (link.classList.contains('input-enable')) {
      this.toggleAlarmEnableState(link.checked,
        this.getAlarmFromList(parseInt(link.dataset.id, 10)));
    } else if (link.classList.contains('alarm-item')) {
      this.alarmEditView(this.getAlarmFromList(
        parseInt(link.dataset.id, 10)));
      evt.preventDefault();
    }
  },

  alarmEditView: function(alarm) {
    App.navigate({ hash: '#alarm-edit-panel', data: alarm });
  },

  init: function al_init() {
    this.template = new Template(alarmHtml);
    this.newAlarmButton.addEventListener('click', this);
    this.alarms.addEventListener('click', this);
    this.banner = new Banner('banner-countdown');

    // Bind this.refresh so that the listener can be easily removed.
    this.refresh = this.refresh.bind(this);
    // Update the dropdown when the language changes.
    window.addEventListener('localized', this.refresh);
    this.refresh();
    AlarmManager.regUpdateAlarmEnableState(this.refreshItem.bind(this));
  },

  refresh: function al_refresh() {
    AlarmsDB.getAlarmList(function al_gotAlarmList(err, list) {
      if (!err) {
        this.fillList(list);
      } else {
        console.error(err);
      }
    }.bind(this));
  },

  render: function al_render(alarm) {
    var repeat = alarm.isRepeating() ?
      alarm.summarizeDaysOfWeek() : '';
    var withRepeat = alarm.isRepeating() ? ' with-repeat' : '';
    // Because `0` is a valid value for these attributes, check for their
    // presence with the `in` operator.
    var isActive = 'normal' in alarm.registeredAlarms ||
      'snooze' in alarm.registeredAlarms;
    var checked = !!isActive ? 'checked=true' : '';

    var d = new Date();
    d.setHours(alarm.hour);
    d.setMinutes(alarm.minute);

    var id = alarm.id + '';
    var time = Utils.getLocaleTime(d);
    var label = alarm.label ? alarm.label : _('alarm');

    return this.template.interpolate({
      id: id,
      checked: checked,
      label: label,
      meridian: time.p,
      repeat: repeat,
      withRepeat: withRepeat,
      time: time.t
    });
  },

  createItem: function al_createItem(alarm, prependTarget) {
    /**
     * createItem
     *
     * Render and then prepend an alarm to the DOM element
     * prependTarget
     *
     */
    var count = this.getAlarmCount();
    var li = document.createElement('li');
    li.className = 'alarm-cell';
    li.id = 'alarm-' + alarm.id;
    li.innerHTML = this.render(alarm);

    if (prependTarget) {
      prependTarget.insertBefore(li, prependTarget.firstChild);

      if (this.count !== count) {
        this.count = count;
        // TODO: Address this circular dependency
        require(['panels/alarm/clock_view'], function(ClockView) {
        ClockView.resizeAnalogClock();
        });
      }
    }
    return li;
  },

  refreshItem: function al_refreshItem(alarm) {
    var li;
    var id = alarm.id;

    if (!this.getAlarmFromList(id)) {
      this.alarmList.push(alarm);
      this.alarmList.sort(function(a, b) {
        return a.id - b.id;
      });
      this.createItem(alarm, this.alarms);
    } else {
      this.setAlarmFromList(id, alarm);
      li = this.alarms.querySelector('#alarm-' + id);
      li.innerHTML = this.render(alarm);

      // clear the refreshing alarm's flag
      if (id in this.toggleOperations) {
        delete this.toggleOperations[id];
      }
    }
  },

  fillList: function al_fillList(alarmList) {
    /**
     * fillList
     *
     * Render all alarms in alarmList to the DOM in
     * decreasing order
     *
     */
    this.alarms.innerHTML = '';
    this.alarmList = alarmList;

    alarmList.sort(function(a, b) {
      return a.id - b.id;
    }).forEach(function al_fillEachList(alarm) {
      // prepend the rendered alarm to the alarm list
      this.createItem(alarm, this.alarms);
    }.bind(this));
  },

  getAlarmFromList: function al_getAlarmFromList(id) {
    for (var i = 0; i < this.alarmList.length; i++) {
      if (this.alarmList[i].id === id) {
        return this.alarmList[i];
      }
    }
    return null;
  },

  setAlarmFromList: function al_setAlarmFromList(id, alarm) {
    for (var i = 0; i < this.alarmList.length; i++) {
      if (this.alarmList[i].id === id) {
        this.alarmList[i] = alarm;
        return;
      }
    }
  },

  getAlarmCount: function al_getAlarmCount() {
    return this.alarmList.length;
  },

  toggleAlarmEnableState: function al_toggleAlarmEnableState(enabled, alarm) {
    var changed = false;
    var toggleOps = this.toggleOperations;
    // has a snooze active
    if (alarm.registeredAlarms.snooze !== undefined) {
      if (!enabled) {
        alarm.cancel('snooze');
        changed = true;
      }
    }
    // normal state needs to change
    if (alarm.enabled !== enabled) {
      toggleOps[alarm.id] = (toggleOps[alarm.id] || 0) + 1;
      // setEnabled saves to database
      alarm.setEnabled(!alarm.enabled, function al_putAlarm(err, alarm) {
        toggleOps[alarm.id]--;

        // If there are any pending toggle operations, the current state of
        // the alarm is volatile, so do not update the DOM.
        if (toggleOps[alarm.id] > 0) {
          return;
        }
        delete toggleOps[alarm.id];

        if (alarm.enabled) {
          this.banner.show(alarm.getNextAlarmFireTime());
        }
        this.refreshItem(alarm);
        AlarmManager.updateAlarmStatusBar();
      }.bind(this));
    } else {
      if (changed) {
        alarm.save();
      }
      AlarmManager.updateAlarmStatusBar();
    }
  }
};

return AlarmList;
});

define('panels/alarm/clock_view',['require','shared/js/async_storage','panels/alarm/alarm_list','utils','l10n'],function(require) {


var asyncStorage = require('shared/js/async_storage');
var AlarmList = require('panels/alarm/alarm_list');
var Utils = require('utils');
var SETTINGS_CLOCKMODE = 'settings_clockoptions_mode';
var mozL10n = require('l10n');
var viewMode = null;

// Retrieve stored view mode data as early as possible.
asyncStorage.getItem(SETTINGS_CLOCKMODE, function(value) {

  // If no value has been stored, don't update
  // the viewMode closure.
  if (value === null) {
    return;
  }
  // If the ClockView hasn't initialized yet,
  // and the stored value is different from
  // the arbitrarily chosen default view (analog)
  // then update the viewMode closure.
  if (!ClockView.isInitialized && viewMode !== value) {
    viewMode = value;
  }
});

var ClockView = {
  get mode() {
    // Closure value, stored in settings,
    // or the default (analog)
    return viewMode;
  },

  set mode(value) {
    // If the `mode` is being updated to a new value:
    //
    //    - Update the viewMode closure
    //    - Store the new value in persistent data storage
    //
    // Always return `value`
    if (viewMode !== value) {
      viewMode = value;
      asyncStorage.setItem(
        SETTINGS_CLOCKMODE, value
      );
    }
    return viewMode;
  },

  timeouts: {
    analog: null,
    dayDate: null,
    digital: null
  },

  get digital() {
    delete this.digital;
    return (this.digital = document.getElementById('digital-clock'));
  },

  get analog() {
    delete this.analog;
    return (this.analog = document.getElementById('analog-clock'));
  },

  get time() {
    delete this.time;
    return (this.time = document.getElementById('clock-time'));
  },

  get hourState() {
    delete this.hourState;
    return (this.hourState = document.getElementById('clock-hour24-state'));
  },

  get dayDate() {
    delete this.dayDate;
    return (this.dayDate = document.getElementById('clock-day-date'));
  },

  get container() {
    delete this.container;
    return (this.container =
      document.getElementById('analog-clock-container'));
  },
  isInitialized: false,

  init: function cv_init() {
    var handler = this.handleEvent.bind(this);

    document.addEventListener('visibilitychange', handler);

    this.analog.addEventListener('click', handler, false);
    this.digital.addEventListener('click', handler, false);
    this.hands = {};
    ['second', 'minute', 'hour'].forEach(function(hand) {
      this.hands[hand] = document.getElementById(hand + 'hand');
    }, this);
    // Kick off the day date display (upper left string)
    this.updateDayDate();

    // If the attempt to request and set the viewMode
    // closure early has failed to respond before the
    // call to ClockView.init(), make an async request,
    // passing the response value as an argument to this.show()
    if (this.mode === null) {
      asyncStorage.getItem(
        SETTINGS_CLOCKMODE, this.show.bind(this)
      );
    } else {
      // Display the clock face
      this.show();
    }

    this.isInitialized = true;
  },

  updateDayDate: function cv_updateDayDate() {
    var d = new Date();
    var f = new mozL10n.DateTimeFormat();
    var format = mozL10n.get('dateFormat');

    // If the date of the month is part of the locale format as a
    // number, insert bold tags to accentuate the number itself. %d
    // and %e are strings that represent the day of the month (1-31).
    format = format.replace(/(%d|%e)/g, '<b>$1</b>');

    var remainMillisecond = (24 - d.getHours()) * 3600 * 1000 -
                            d.getMinutes() * 60 * 1000 -
                            d.getMilliseconds();

    this.dayDate.innerHTML = f.localeFormat(d, format);

    this.timeouts.dayDate = setTimeout(
      this.updateDayDate.bind(this), remainMillisecond
    );
  },

  update: function cv_update(opts) {
    opts = opts || {};

    if (this.mode === 'digital') {
      this.updateDigitalClock(opts);
    } else {
      this.updateAnalogClock(opts);
    }
  },

  updateDigitalClock: function cv_updateDigitalClock(opts) {
    opts = opts || {};

    var d = new Date();
    var time = Utils.getLocaleTime(d);
    this.time.textContent = time.t;
    this.hourState.textContent = time.p || '  '; // 2 non-break spaces

    this.timeouts.digital = setTimeout(
      this.updateDigitalClock.bind(this), (60 - d.getSeconds()) * 1000
    );
  },

  updateAnalogClock: function cv_updateAnalogClock(opts) {
    opts = opts || {};

    if (opts.needsResize) {
      this.resizeAnalogClock();
    }
    var now = new Date();
    var sec, min, hour;
    sec = now.getSeconds();
    min = now.getMinutes();
    // hours progress gradually
    hour = (now.getHours() % 12) + min / 60;
    this.setTransform('second', sec);
    this.setTransform('minute', min);
    this.setTransform('hour', hour);
    // update again in one second
    this.timeouts.analog = setTimeout(
      this.updateAnalogClock.bind(this), 1000 - now.getMilliseconds()
    );
  },

  setTransform: function cv_setTransform(id, angle) {
    var hand = this.hands[id];
    // return correct angle for different hands
    function conv(timeFrag) {
      var mult;
      // generate a conformable number to rotate about
      // 30 degrees per hour 6 per second and minute
      mult = id === 'hour' ? 30 : 6;
      // we generate the angle from the fractional sec/min/hour
      return (timeFrag * mult);
    }
    // Use transform rotate on the rect itself vs on a child element
    // avoids unexpected behavior if either dur and fill are set to defaults
    // Use translateZ to force it on its own layer, which will invoke the GPU
    // and thus do the minimum amount of work required (reduces power usage)
    hand.style.transform = 'rotate(' + conv(angle) + 'deg) translateZ(1px)';
  },

  handleEvent: function cv_handleEvent(event) {
    var newMode, target;

    switch (event.type) {
      case 'visibilitychange':
        if (document.hidden) {
          if (this.timeouts.dayDate) {
            clearTimeout(this.timeouts.dayDate);
          }
          if (this.timeouts.digital) {
            clearTimeout(this.timeouts.digital);
          }
          if (this.timeouts.analog) {
            clearTimeout(this.timeouts.analog);
          }
          return;
        } else if (!document.hidden) {
          // Refresh the view when app return to foreground.
          this.updateDayDate();

          if (this.mode === 'digital') {
            this.updateDigitalClock();
          } else if (this.mode === 'analog') {
            this.updateAnalogClock();
          }
        }
        break;

      case 'click':
        target = event.target;

        if (!target) {
          return;
        }

        if (this.digital.contains(target) ||
            target.id === 'digital-clock') {

          newMode = 'analog';
        }

        if (this.analog.contains(target) ||
            target.id === 'analog-clock') {

          newMode = 'digital';
        }

        if (newMode) {
          this.show(newMode);
        }

        break;
    }
  },

  calAnalogClockType: function cv_calAnalogClockType(count) {
    var type = 'small';
    if (count < 2) {
      type = 'large';
    } else if (count === 2) {
      type = 'medium';
    }
    return type;
  },

  resizeAnalogClock: function cv_resizeAnalogClock() {
    var type = this.calAnalogClockType(AlarmList.getAlarmCount());
    this.container.className = type;
    document.getElementById('alarms').className = 'count' + type;
  },

  show: function cv_show(mode) {
    var isAnalog = false;
    var previous, hiding, showing;

    if (location.hash !== '#alarm-panel') {
      location.hash = '#alarm-panel';
    }

    // The clock display mode is either
    //
    //    - Explicitly passed as the mode param
    //    - Set as a property of ClockView
    //    - Default to "analog"
    //
    mode = mode || this.mode || 'analog';

    isAnalog = mode === 'analog';

    // Determine what to hide and what to show
    previous = isAnalog ? 'digital' : 'analog';
    hiding = isAnalog ? this.digital : this.analog;
    showing = isAnalog ? this.analog : this.digital;

    // Clear any previously created timeouts.
    if (this.timeouts[previous]) {
      clearTimeout(
        this.timeouts[previous]
      );
    }

    hiding.classList.remove('visible');
    showing.classList.add('visible');

    // Update the locally stored `mode`.
    this.mode = mode;

    // This MUST be called after this.mode is set
    // to ensure that the correct mode is used for
    // updating the clockface
    this.update({
      needsResize: true
    });
  }
};

return ClockView;
});

define('timer',['require','emitter','shared/js/async_storage','utils'],function(require) {


var Emitter = require('emitter');
var asyncStorage = require('shared/js/async_storage');
var Utils = require('utils');

var timerPrivate = new WeakMap();

/**
 * Timer
 *
 * Create new or revive existing timer objects.
 *
 * @param {Object} opts Optional timer object to create or revive
 *                      a new or existing timer object.
 *                 - startTime, number time in ms.
 *                 - duration, time to count from `start`.
 *                 - configuredDuration, time requested by user.
 *                 - sound, string sound name.
 *                 - vibrate, boolean, vibrate or not.
 *                 - id, integer, mozAlarm API id number.
 */
function Timer(opts) {
  opts = opts || {};
  Emitter.call(this);

  var now = Date.now();
  if (opts.id !== undefined) {
    delete opts.id;
  }
  // private properties
  timerPrivate.set(this, Utils.extend({
    state: Timer.INITIAL
  }, extractProtected(opts)));
  // public properties
  Utils.extend(this, {
    startTime: now,
    duration: null,
    configuredDuration: null,
    sound: 'ac_classic_clock_alarm.opus',
    vibrate: true
  }, opts);
}

Timer.prototype = Object.create(Emitter.prototype);
Timer.prototype.constructor = Timer;

/**
 * request - get the persisted Timer object.
 *
 * @param {function} [callback] - called with (err, timer_raw).
 */
Timer.request = function timerRequest(callback) {
  asyncStorage.getItem('active_timer', function(obj) {
    callback && callback(null, obj || null);
  });
};

/**
 * singleton - get the unique persisted Timer object.
 *
 * @param {function} [callback] - called with (err, timer).
 */
var timerSingleton = Utils.singleton(Timer);
Timer.singleton = function tm_singleton(callback) {
  Timer.request(function(err, obj) {
    var ts = timerSingleton(obj);
    callback && callback(null, ts);
  });
};

function extractProtected(config) {
  var ret = {};
  var protectedProperties = new Set(['state']);
  for (var i in config) {
    if (protectedProperties.has(i)) {
      ret[i] = config[i];
      delete config[i];
    }
  }
  return ret;
}

/**
 * toSerializable - convert `this` to a serialized format.
 *
 * @return {object} - object representation of this Timer.
 */
Timer.prototype.toSerializable = function timerToSerializable() {
  var ret = {};
  var props = Utils.extend({}, this, timerPrivate.get(this));
  [
    'startTime', 'duration', 'configuredDuration', 'sound', 'vibrate',
    'state'
  ].forEach(function(x) {
    ret[x] = props[x];
  });
  return ret;
};

/**
 * save - Save the timer to the database.
 *
 * @param {function} [callback] - callback to call after the timer
 *                                has been saved.
 */
Timer.prototype.save = function timerSave(callback) {
  asyncStorage.setItem('active_timer', this.toSerializable(), function() {
    callback && callback(null, this);
  }.bind(this));
};

/**
 * register - Register the timer with mozAlarm API.
 *
 * @param {function} [callback] - callback to call after the timer
 *                                has been registered.
 */
Timer.prototype.register = function timerRegister(callback) {
  var data = {
    type: 'timer'
  };
  var request;

  // Remove previously-created mozAlarm for this alarm, if necessary.
  this.unregister();

  request = navigator.mozAlarms.add(
    new Date(Date.now() + this.remaining), 'ignoreTimezone', data
  );

  request.onsuccess = (function(ev) {
    this.id = ev.target.result;
    callback && callback(null, this);
  }.bind(this));
  request.onerror = function(ev) {
    callback && callback(ev.target.error);
  };
};

/**
 * commit - save and register the timer as necessary.
 *
 * @param {function} [callback] - callback to call after the timer
 *                                has been registered.
 */
Timer.prototype.commit = function timerCommit(callback) {
  var saveSelf = this.save.bind(this, callback);
  if (this.state === Timer.STARTED) {
    this.register(saveSelf);
  } else {
    this.unregister();
    saveSelf();
  }
};

Timer.prototype.unregister = function timerUnregister() {
  if (typeof this.id === 'number') {
    navigator.mozAlarms.remove(this.id);
  }
};

Object.defineProperty(Timer.prototype, 'remaining', {
  get: function() {
    if (this.state === Timer.INITIAL) {
      return this.configuredDuration;
    } else if (this.state === Timer.PAUSED) {
      return this.duration;
    } else if (this.state === Timer.STARTED) {
      if (typeof this.startTime === 'undefined' ||
          typeof this.duration === 'undefined') {
        return 0;
      }
      var r = (this.startTime + this.duration) - Date.now();
      return r >= 0 ? r : 0;
    }
  }
});

Object.defineProperty(Timer.prototype, 'state', {
  get: function() {
    var priv = timerPrivate.get(this);
    return priv.state;
  }
});

Timer.prototype.start = function timerStart() {
  if (this.state !== Timer.STARTED) {
    var priv = timerPrivate.get(this);
    priv.state = Timer.STARTED;
    this.startTime = Date.now();
    this.duration = (typeof this.duration === 'number') ? this.duration :
      this.configuredDuration;
    this.emit('start');
  }
};

Timer.prototype.pause = function timerPause() {
  if (this.state === Timer.STARTED) {
    this.duration = this.remaining; // remaining getter observes private state
    var priv = timerPrivate.get(this);
    priv.state = Timer.PAUSED;
    this.startTime = null;
    this.emit('pause');
  }
};

Timer.prototype.cancel = function timerReset() {
  if (this.state !== Timer.INITIAL) {
    var priv = timerPrivate.get(this);
    priv.state = Timer.INITIAL;
    this.startTime = null;
    this.duration = this.configuredDuration;
    this.emit('end');
  }
};

/**
 * plus Increase the duration and extend the endAt time
 *
 * @param {Number} seconds The time in seconds to add.
 *
 * @return {Timer} Timer instance.
 */
Timer.prototype.plus = function timerPlus(seconds) {
  // Convert to ms
  var ms = seconds * 1000;

  this.duration += ms;

  return this;
};

/**
 * Static "const" Timer states.
 */
Object.defineProperties(Timer, {
  INITIAL: { value: 0 },
  STARTED: { value: 1 },
  PAUSED: { value: 2 }
});

return Timer;
});

define('panels/alarm/active_alarm',['require','app','alarm_manager','panels/alarm/alarm_list','alarmsdb','timer','utils'],function(require) {
  

  var App = require('app');
  var AlarmManager = require('alarm_manager');
  var AlarmList = require('panels/alarm/alarm_list');
  var AlarmsDB = require('alarmsdb');
  var Timer = require('timer');
  var Utils = require('utils');

  function ActiveAlarm() {
    this.childWindow = null;
    this.initialized = false;
    this.ringerWaitList = [];
  }

  ActiveAlarm.singleton = Utils.singleton(ActiveAlarm);

  var messageHandlerMapping = {
    normal: 'onAlarm',
    snooze: 'onAlarm',
    scheduleSnooze: 'scheduleSnooze',
    timer: 'onTimer',
    ringer: 'onRingerReady',
    'close-alarm': 'onClose',
    'close-timer': 'onClose'
  };

  ActiveAlarm.prototype = {
    constructor: ActiveAlarm,

    init: function am_init() {
      if (!this.initialized) {
        var handler = this.handler.bind(this);
        navigator.mozSetMessageHandler('alarm', handler);
        // Add a handler to make integration tests easier:
        window.addEventListener('test-alarm', handler);
        navigator.mozSetMessageHandler('message', handler);
        window.addEventListener('message', handler, false);
        AlarmManager.updateAlarmStatusBar();
        this.initialized = true;
      }
    },

    handler: function aac_handler(message) {
      // If this is a 'test-alarm' CustomEvent, data is stored in 'detail'.
      var data = message.data || message.detail;
      data.date = message.date || new Date();

      // Set a watchdog to avoid locking the CPU wake lock too long,
      // because it'd exhaust the battery quickly which is very bad.
      // This could probably happen if the app failed to launch or
      // handle the alarm message due to any unexpected reasons.
      Utils.safeWakeLock({timeoutMs: 30000}, function(done) {
        try {
          this[messageHandlerMapping[data.type]].call(
            this, data, done);
        } catch (err) {
          console.error('Error calling handler', err);
          done();
          throw err;
        }
      }.bind(this));
    },

    onRingerReady: function aac_ringerReady(data, done) {
      if (data.status === 'READY') {
        while (true) {
          var el = this.ringerWaitList.shift();
          if (!el) {
            break;
          }
          if (typeof el === 'function') {
            el();
          }
        }
      }
      done();
    },

    popAlert: function aac_popAlert(callback) {
      if (this.childwindow && !this.childwindow.closed) {
        this.childwindow.postMessage({
          type: 'stop'
        }, window.location.origin);
        this.ringerWaitList.push(callback.bind(null, this.childwindow));
      } else {
        // prepare to pop out attention screen, ring the ringtone, vibrate
        var childwindow = this.childwindow = window.open(
          window.location.origin + '/onring.html', '_blank', 'attention');
        this.ringerWaitList.push(callback.bind(null, childwindow));
      }
    },

    onAlarm: function aac_onAlarm(data, done) {
      /*
       * We maintain an alarm's life cycle immediately when the alarm goes off.
       * If user click the snooze button when the alarm goes off,
       * we request a snooze alarm with snoozeAlarmId immediately.
       *
       * If multiple alarms goes off in a period of time (even if in the same
       * time), we always stop the previous notification and handle it by
       * its setting. Such as following case:
       *   An once alarm should be turned off.
       *   A repeat alarm should be requested its next alarm.
       *   A snooze alarm should be turned off.
       */
      // receive and parse the alarm id from the message
      var id = data.id;
      var date = data.date;
      var type = data.type;

      // Unlock the CPU when these functions have been called
      var finalizer = Utils.async.namedParallel([
        'onReschedule',
        'onReceivedAlarm'
      ], function(err) {
        AlarmList.refresh();
        AlarmManager.updateAlarmStatusBar();
        done();
      });

      AlarmsDB.getAlarm(id, function aac_gotAlarm(err, alarm) {
        if (err) {
          done();
          return;
        }
        if (type === 'normal') {
          alarm.schedule({
            type: 'normal',
            first: false
          }, alarm.saveCallback(finalizer.onReschedule));
        } else {
          alarm.cancel('snooze');
          alarm.save(finalizer.onReschedule);
        }
        // prepare to pop out attention screen, ring the ringtone, vibrate
        this.popAlert(function(childWindow) {
          childWindow.postMessage({
            type: 'alarm',
            date: date,
            alarm: alarm.toSerializable()
          }, window.location.origin);
        });
        finalizer.onReceivedAlarm();
      }.bind(this));
    },

    onTimer: function aac_onTimer(data, done) {
      Timer.request(function(err, timer) {
        if (err && !timer) {
          timer = Timer.singleton().toSerializable();
        }
        this.popAlert(function(childWindow) {
          childWindow.postMessage({
            type: 'timer',
            timer: timer
          }, window.location.origin);
          done();
        }.bind(this));
      }.bind(this));
    },

    scheduleSnooze: function aac_scheduleSnooze(data, done) {
      var id = data.id;
      AlarmsDB.getAlarm(id, function aac_gotAlarm(err, alarm) {
        if (err) {
          return;
        }
        alarm.schedule({
          type: 'snooze'
        }, alarm.saveCallback(function(err, alarm) {
          AlarmList.refreshItem(alarm);
          AlarmManager.updateAlarmStatusBar();
          done();
        }));
      });
    },

    onClose: function aac_onClose(data, done) {
      this.childwindow = null;
      switch (data.type) {
        case 'close-timer':
          Timer.singleton(function(err, timer) {
            if (!err) {
              timer.cancel();
              timer.save();
            }
            App.navigate({ hash: '#timer-panel' }, done);
          });
          break;
        case 'close-alarm':
          App.navigate({ hash: '#alarm-panel' }, done);
          break;
      }
    }
  };

  return ActiveAlarm;
});

define('text!panels/alarm/panel.html',[],function () { return '<div id="clock-view">\n  <div id="analog-clock">\n    <div id="analog-clock-container">\n      <div id="analog-clock-face">\n        <div id="analog-clock-hands">\n          <div class="analog-clock-hand" id="secondhand"></div>\n          <div class="analog-clock-hand" id="minutehand"></div>\n          <div class="analog-clock-hand" id="hourhand"></div>\n        </div>\n      </div>\n    </div>\n  </div>\n  <div id="digital-clock">\n    <div id="digital-clock-face">\n      <span id="clock-time"></span>\n      <span id="clock-hour24-state"></span>\n    </div>\n  </div>\n  <div id="clock-day-date"></div>\n  <!--  create new alarm icon -->\n  <button id="alarm-new"></button>\n  <!-- list of exisiting alarms, to be populated -->\n  <ul id="alarms"></ul>\n</div>\n<section id="banner-countdown" role="status">\n  <!-- this will be replaced dynamically -->\n</section>\n';});

define('panels/alarm/main',['require','panel','panels/alarm/clock_view','panels/alarm/alarm_list','panels/alarm/active_alarm','text!panels/alarm/panel.html'],function(require) {


var Panel = require('panel');
var ClockView = require('panels/alarm/clock_view');
var AlarmList = require('panels/alarm/alarm_list');
var ActiveAlarm = require('panels/alarm/active_alarm');
var html = require('text!panels/alarm/panel.html');

function AlarmPanel() {
  Panel.apply(this, arguments);

  this.element.innerHTML = html;
  ClockView.init();
  AlarmList.init();
  ActiveAlarm.singleton().init();
}

AlarmPanel.prototype = Object.create(Panel.prototype);

return AlarmPanel;
});
