
// outer IIFE
define('form_button',['require','utils'],function(require) {


var Utils = require('utils');

function createButton(formButton) {
  var button = document.createElement(formButton.tagName);
  button.className = formButton.className;
  if (formButton.id) {
    button.id = formButton.id;
  }
  var input = formButton.input;
  input.parentNode.insertBefore(button, input.nextSibling);
  formButton.button = button;
}

/**
 * A FormButton is a button that triggers an input. The text
 * of the currently selected value will display on the buttons's face.
 *
 * The `config` paramater supports the following optional properties.
 * `formatLabel` - A function that is given the current value of the input
 * and should return a string which will be used as the textContent of
 * the button.
 *
 * `tagName` - The name of the tag to create and insert into the
 * document as the main button used to trigger the input. The default
 * value is 'button'
 *
 * `className` The value of the className property that will be assigned to
 *  the button element the default value is 'icon icon-dialog'.
 *
 * `id` - A string that is used as the id of the button element.
 *
 * @constructor
 * @param {HTMLElement} input The input element to trigger.
 * @param {Object} config An optional config object.
 *
 */
function FormButton(input, config) {
  config = config || {};
  Utils.extend(this, config);

  this.input = input;
  createButton(this);

  this.input.classList.add('form-button-input');
  // hide input
  this.input.classList.add('form-button-hide');

  // set isSelect
  Object.defineProperty(this, 'isSelect', {
    value: this.input.nodeName === 'SELECT'
  });

  this.button.addEventListener('click', this.focus.bind(this), false);

  input.addEventListener('change', this.refresh.bind(this), false);
  input.addEventListener('blur', this.refresh.bind(this), false);

  // Bind this.refresh so that the listener can be easily removed.
  this.refresh = this.refresh.bind(this);
  // Update the dropdown when the language changes.
  window.addEventListener('localized', this.refresh);
}

FormButton.prototype = {

  /** Remove all event handlers. */
  destroy: function() {
    window.removeEventListener('localized', this.refresh);
  },

  /**
   * focus Triggers a focus event on the input associated with this
   * FormButton.
   *
   * @param {Object} event an event object.
   */
  focus: function(event) {
    event.preventDefault();
    setTimeout(this.input.focus.bind(this.input), 10);
  },

  /**
   * refresh Updates the label text on the button to reflect
   * the current value of the input.
   *
   */
  refresh: function() {
    var value = this.value;
    this.button.textContent = this.formatLabel(value);
  },

  /**
   * value Returns the current value of the input.
   *
   * @return {String|Object} The value of the input.
   *
   */
  get value() {
    if (this.isSelect) {
      if (this.input.multiple) {
        var selectedOptions = {};
        var options = this.input.options;
        for (var i = 0; i < options.length; i++) {
          if (options[i].selected) {
            selectedOptions[options[i].value] = true;
          }
        }
        return selectedOptions;
      }
      if (this.input.selectedIndex !== -1) {
        return Utils.getSelectedValueByIndex(this.input);
      }
      return null;
    }
    // input node
    return this.input.value;
  },

  /**
   * value sets the current value of the input and update's the
   * button text.
   *
   * @param {String|Object} value A string of the current values or an
   * object with properties that map to input options if the input is
   * a multi select.
   *
   */
  set value(value) {
    if (this.isSelect) {
      if (this.input.multiple) {
        // multi select
        var options = this.input.options;
        for (var i = 0; i < options.length; i++) {
          options[i].selected = value[options[i].value] === true;
        }
      } else {
        // normal select element
        Utils.changeSelectByValue(this.input, value);
      }
    } else {
      // input element
      this.input.value = value;
    }
    // Update the text on the button to reflect the new input value
    this.refresh();
  },

  /**
   * An overrideable method that is called when updating the textContent
   * of the button.
   *
   * @return {String} The formatted text to display in the label.
   *
   */
  formatLabel: function(value) {
    return value;
  },

  /**
   * tagName The the name of the tag to insert into the document to use
   * as the button element.
   */
  tagName: 'button',

  /**
   * class The value to assign to the className property on the
   * generated button element.
   */
  className: 'icon icon-dialog'

};

  return FormButton;

// end outer IIFE
});

define('sounds',['require','exports','module','l10n'],function(require, exports) {
  
  
  var _ = require('l10n').get;

  // Sadly, this is needed because when sound l10n ids change, they no
  // longer match up with the sound filename.
  var SOUND_FILE_TO_L10N_ID = {
    '0': 'noSound',
    'ac_classic_clock_alarm.opus': 'ac_classic_clock_alarm_opus',
    'ac_classic_clock_alarm_prog.opus': 'ac_classic_clock_alarm_prog_opus2',
    'ac_classic_clock_radio.opus': 'ac_classic_clock_radio_opus',
    'ac_normal_gem_echoes.opus': 'ac_normal_gem_echoes_opus',
    'ac_normal_ringing_strings.opus': 'ac_normal_ringing_strings_opus',
    'ac_soft_humming_waves.opus': 'ac_soft_humming_waves_opus',
    'ac_soft_into_the_void.opus': 'ac_soft_into_the_void_opus',
    'ac_soft_smooth_strings.opus': 'ac_soft_smooth_strings_opus'
  };

  /**
   * Given a sound ID, return the label to be displayed, for instance,
   * on a FormButton.
   */
  exports.formatLabel = function(sound) {
    return (sound === null || sound === '0') ?
      _('noSound') : _(SOUND_FILE_TO_L10N_ID[sound]);
  };
});

define('text!panels/alarm_edit/panel.html',[],function () { return '<header>\n  <button id="alarm-close"><span class="icon icon-close"></span></button>\n  <menu type="toolbar">\n    <button id="alarm-done" data-l10n-id="done"></button>\n  </menu>\n  <h1 id="alarm-title">\n    <span class="new-alarm-title" data-l10n-id="newAlarm">NeW AlarM</span>\n    <span class="edit-alarm-title" data-l10n-id="editAlarm">Edit AlarM</span>\n  </h1>\n</header>\n<ul id="edit-alarm" class="compact">\n  <li>\n    <input type="text" name="alarm.label" id="alarm-name" data-l10n-id="alarmName" placeholder="Alarm name" maxLength="50" dir="auto"/>\n  </li>\n  <li>\n    <label data-l10n-id="time">Time</label>\n    <input id="time-select" type="time" />\n  </li>\n  <li>\n    <label data-l10n-id="repeat">Repeat</label>\n    <select id="repeat-select" multiple="true">\n      <!-- NOTE: These are reordered based upon the value for\n                 the l10n variable \'weekStartsOnMonday\'. -->\n      <option value="monday" data-l10n-id="weekday-1-long">Monday</option>\n      <option value="tuesday" data-l10n-id="weekday-2-long">Tuesday</option>\n      <option value="wednesday" data-l10n-id="weekday-3-long">Wednesday</option>\n      <option value="thursday" data-l10n-id="weekday-4-long">Thursday</option>\n      <option value="friday" data-l10n-id="weekday-5-long">Friday</option>\n      <option value="saturday" data-l10n-id="weekday-6-long">Saturday</option>\n      <option value="sunday" id="repeat-select-sunday"\n              data-l10n-id="weekday-0-long">Sunday</option>\n    </select>\n  </li>\n  <li>\n    <label data-l10n-id="sound">Sound</label>\n    <select id="sound-select">\n      <option value="0" data-l10n-id="noSound">No Sound</option>\n      <option value="ac_classic_clock_alarm.opus" data-l10n-id="ac_classic_clock_alarm_opus"></option>\n      <option value="ac_classic_clock_alarm_prog.opus" data-l10n-id="ac_classic_clock_alarm_prog_opus2"></option>\n      <option value="ac_classic_clock_radio.opus" data-l10n-id="ac_classic_clock_radio_opus"></option>\n      <option value="ac_normal_gem_echoes.opus" data-l10n-id="ac_normal_gem_echoes_opus"></option>\n      <option value="ac_normal_ringing_strings.opus" data-l10n-id="ac_normal_ringing_strings_opus"></option>\n      <option value="ac_soft_humming_waves.opus" data-l10n-id="ac_soft_humming_waves_opus"></option>\n      <option value="ac_soft_into_the_void.opus" data-l10n-id="ac_soft_into_the_void_opus"></option>\n      <option value="ac_soft_smooth_strings.opus" data-l10n-id="ac_soft_smooth_strings_opus"></option>\n    </select>\n  </li>\n  <li>\n    <label data-l10n-id="vibrate">Vibrate</label>\n    <select id="vibrate-select">\n      <option value="1" data-l10n-id="vibrateOn">On</option>\n      <option value="0" data-l10n-id="vibrateOff">Off</option>\n    </select>\n  </li>\n  <li>\n    <label class="view-alarm-lbl snooze-lbl" data-l10n-id="snooze-label">Snooze</label>\n    <select id="snooze-select">\n      <option data-l10n-id="nMinutes" data-l10n-args=\'{"n": "5"}\'  value="5">  5 minutes</option>\n      <option data-l10n-id="nMinutes" data-l10n-args=\'{"n": "10"}\' value="10">10 minutes</option>\n      <option data-l10n-id="nMinutes" data-l10n-args=\'{"n": "15"}\' value="15">15 minutes</option>\n      <option data-l10n-id="nMinutes" data-l10n-args=\'{"n": "20"}\' value="20">20 minutes</option>\n    </select>\n  </li>\n  <li id="delete-menu">\n    <button id="alarm-delete" class="danger full" data-l10n-id="delete">Delete</button>\n  </li>\n</ul>\n';});


/* global KeyEvent */
define('panels/alarm_edit/main',['require','alarm','panels/alarm/alarm_list','alarm_manager','panels/alarm/clock_view','form_button','sounds','utils','l10n','panel','text!panels/alarm_edit/panel.html','constants'],function(require) {
var Alarm = require('alarm');
var AlarmList = require('panels/alarm/alarm_list');
var AlarmManager = require('alarm_manager');
var ClockView = require('panels/alarm/clock_view');
var FormButton = require('form_button');
var Sounds = require('sounds');
var Utils = require('utils');
var mozL10n = require('l10n');
var Panel = require('panel');
var _ = mozL10n.get;
var html = require('text!panels/alarm_edit/panel.html');
var constants = require('constants');

var AlarmEdit = function() {
  Panel.apply(this, arguments);
  this.element.innerHTML = html;
  mozL10n.translate(this.element);
  var handleDomEvent = this.handleDomEvent.bind(this);
  this.on('visibilitychange', this.handleVisibilityChange.bind(this));

  this.selects = {};
  [
    'time', 'repeat', 'sound', 'vibrate', 'snooze'
  ].forEach(function(id) {
    this.selects[id] = this.element.querySelector('#' + id + '-select');
  }, this);

  this.inputs = {
    name: this.element.querySelector('#alarm-name')
  };

  this.buttons = {};
  [
    'delete', 'close', 'done'
  ].forEach(function(id) {
    this.buttons[id] = this.element.querySelector('#alarm-' + id);
  }, this);

  this.buttons.time = new FormButton(this.selects.time, {
    formatLabel: function(value) {
      var time = Utils.parseTime(value);
      return Utils.format.time(time.hour, time.minute);
    }.bind(this)
  });
  this.buttons.repeat = new FormButton(this.selects.repeat, {
    selectOptions: constants.DAYS,
    id: 'repeat-menu',
    formatLabel: function(daysOfWeek) {
      return this.alarm.summarizeDaysOfWeek(daysOfWeek);
    }.bind(this)
  });
  this.buttons.sound = new FormButton(this.selects.sound, {
    id: 'sound-menu',
    formatLabel: Sounds.formatLabel
  });
  this.buttons.vibrate = new FormButton(this.selects.vibrate, {
    formatLabel: function(vibrate) {
      return (vibrate === null || vibrate === '0') ?
        _('vibrateOff') :
        _('vibrateOn');
    }
  });
  this.buttons.snooze = new FormButton(this.selects.snooze, {
    id: 'snooze-menu',
    formatLabel: function(snooze) {
      return _('nMinutes', {n: snooze});
    }
  });

  // When the system pops up the ValueSelector, it inadvertently
  // messes with the scrollTop of the current panel. This is a
  // workaround for bug 981255 until the Edit panel becomes a new
  // window per bug 922651.
  this.element.addEventListener('scroll', function() {
    this.element.scrollTop = 0;
  }.bind(this));

  mozL10n.translate(this.element);
  // When the language changes, the value of 'weekStartsOnMonday'
  // might change. Since that's more than a simple text string, we
  // can't just use mozL10n.translate().
  window.addEventListener('localized', this.updateL10n.bind(this));
  this.updateL10n();

  this.buttons.close.addEventListener('click', handleDomEvent);
  this.buttons.done.addEventListener('click', handleDomEvent);
  this.selects.sound.addEventListener('change', handleDomEvent);
  this.selects.sound.addEventListener('blur', handleDomEvent);
  this.selects.repeat.addEventListener('change', handleDomEvent);
  this.buttons.delete.addEventListener('click', handleDomEvent);
  this.inputs.name.addEventListener('keypress', this.handleNameInput);

  // If the phone locks during preview, pause the sound.
  // TODO: When this is no longer a singleton, unbind the listener.
  window.addEventListener('visibilitychange', function() {
    if (document.hidden) {
      this.stopPreviewSound();
    }
  }.bind(this));
};

AlarmEdit.prototype = Object.create(Panel.prototype);

var selectors = {
  scrollList: '#edit-alarm',
  labelInput: 'input[name="alarm.label"]',
  timeSelect: '#time-select',
  timeMenu: '#time-menu',
  alarmTitle: '#alarm-title',
  repeatMenu: '#repeat-menu',
  repeatSelect: '#repeat-select',
  sundayListItem: '#repeat-select-sunday',
  soundMenu: '#sound-menu',
  soundSelect: '#sound-select',
  vibrateMenu: '#vibrate-menu',
  vibrateSelect: '#vibrate-select',
  snoozeMenu: '#snooze-menu',
  snoozeSelect: '#snooze-select',
  deleteButton: '#alarm-delete',
  backButton: '#alarm-close',
  doneButton: '#alarm-done'
};
Object.keys(selectors).forEach(function(attr) {
  var selector = selectors[attr];
  Object.defineProperty(AlarmEdit.prototype, attr, {
    get: function() {
      var element = this.element.querySelector(selector);
      Object.defineProperty(this, attr, {
        value: element
      });
      return element;
    },
    configurable: true
  });
});

Utils.extend(AlarmEdit.prototype, {

  alarm: null,
  alarmRef: null,
  timePicker: {
    hour: null,
    minute: null,
    hour24State: null,
    is12hFormat: false
  },
  previewRingtonePlayer: null,

  handleNameInput: function(evt) {
    // If the user presses enter on the name label, dismiss the
    // keyboard to allow them to continue filling out the other
    // fields. This is not in the `handleEvent` function because we
    // only want to call `.preventDefault` sometimes.
    if (evt.keyCode === KeyEvent.DOM_VK_RETURN) {
      evt.preventDefault();
      evt.target.blur();
    }
  },

  updateL10n: function() {
    // Move the weekdays around to properly account for whether the
    // week starts on Sunday or Monday.
    var weekStartsOnMonday = parseInt(_('weekStartsOnMonday'), 10);
    var parent = this.sundayListItem.parentElement;
    if (weekStartsOnMonday) {
      // Sunday gets moved to the end.
      parent.appendChild(this.sundayListItem);
    } else {
      // Sunday goes first.
      parent.insertBefore(this.sundayListItem, parent.firstChild);
    }
  },

  // The name `handleEvent` is already defined by the Panel class, so this
  // method must be named uniquely to avoid overriding that functionality.
  handleDomEvent: function aev_handleDomEvent(evt) {
    evt.preventDefault();
    var input = evt.target;
    if (!input) {
      return;
    }

    switch (input) {
      case this.buttons.close:
        ClockView.show();
        break;
      case this.buttons.done:
        ClockView.show();
        this.save(function aev_saveCallback(err, alarm) {
          if (err) {
            return;
          }
          AlarmList.refreshItem(alarm);
        });
        break;
      case this.selects.sound:
        switch (evt.type) {
          case 'change':
            this.previewSound();
            break;
          case 'blur':
            this.stopPreviewSound();
            break;
        }
        break;
      case this.buttons.delete:
        ClockView.show();
        this.delete();
        break;
      case this.selects.repeat:
        this.alarm.repeat = this.buttons.repeat.value;
        break;
    }
  },

  focusMenu: function aev_focusMenu(menu) {
    setTimeout(function() { menu.focus(); }, 10);
  },

  handleVisibilityChange: function aev_show(isVisible) {
    var alarm;
    if (!isVisible) {
      return;
    }
    // `navData` is set by the App module in `navigate`.
    alarm = this.navData;
    // scroll to top of form list
    this.scrollList.scrollTop = 0;

    this.element.classList.toggle('new', !alarm);
    this.alarm = new Alarm(alarm); // alarm may be null

    // Set to empty string if the Alarm doesn't have an ID,
    // otherwise dataset will automatically stringify it
    // to be "undefined" rather than "".
    this.element.dataset.id = this.alarm.id || '';
    this.inputs.name.value = this.alarm.label;

    // Init time, repeat, sound, snooze selection menu.
    this.initTimeSelect();
    this.initRepeatSelect();
    this.initSoundSelect();
    this.initVibrateSelect();
    this.initSnoozeSelect();
    location.hash = '#alarm-edit-panel';
  },

  initTimeSelect: function aev_initTimeSelect() {
    // The format of input type="time" should be in HH:MM
    var opts = { meridian: false, padHours: true };
    var time = Utils.format.time(this.alarm.hour, this.alarm.minute, opts);
    this.buttons.time.value = time;
  },

  getTimeSelect: function aev_getTimeSelect() {
    return Utils.parseTime(this.selects.time.value);
  },
  initRepeatSelect: function aev_initRepeatSelect() {
    this.buttons.repeat.value = this.alarm.repeat;
  },

  initSoundSelect: function aev_initSoundSelect() {
    this.buttons.sound.value = this.alarm.sound;
  },

  getSoundSelect: function aev_getSoundSelect() {
    return this.buttons.sound.value;
  },

  previewSound: function aev_previewSound() {
    var ringtonePlayer = this.previewRingtonePlayer;
    if (!ringtonePlayer) {
      this.previewRingtonePlayer = new Audio();
      ringtonePlayer = this.previewRingtonePlayer;
    } else {
      ringtonePlayer.pause();
    }

    var ringtoneName = this.getSoundSelect();
    var previewRingtone = 'shared/resources/media/alarms/' + ringtoneName;
    ringtonePlayer.mozAudioChannelType = 'alarm';
    ringtonePlayer.src = previewRingtone;
    ringtonePlayer.play();
  },

  stopPreviewSound: function aev_stopPreviewSound() {
    if (this.previewRingtonePlayer) {
      this.previewRingtonePlayer.pause();
    }
  },

  initVibrateSelect: function aev_initVibrateSelect() {
    this.buttons.vibrate.value = this.alarm.vibrate;
  },

  getVibrateSelect: function aev_getVibrateSelect() {
    return this.buttons.vibrate.value;
  },

  initSnoozeSelect: function aev_initSnoozeSelect() {
    this.buttons.snooze.value = this.alarm.snooze;
  },

  getSnoozeSelect: function aev_getSnoozeSelect() {
    return this.buttons.snooze.value;
  },

  getRepeatSelect: function aev_getRepeatSelect() {
    return this.buttons.repeat.value;
  },

  save: function aev_save(callback) {
    if (this.element.dataset.id !== '') {
      this.alarm.id = parseInt(this.element.dataset.id, 10);
    } else {
      delete this.alarm.id;
    }
    var error = false;

    this.alarm.label = this.inputs.name.value;

    var time = this.getTimeSelect();
    this.alarm.time = [time.hour, time.minute];
    this.alarm.repeat = this.buttons.repeat.value;
    this.alarm.sound = this.getSoundSelect();
    this.alarm.vibrate = this.getVibrateSelect();
    this.alarm.snooze = parseInt(this.getSnoozeSelect(), 10);

    if (!error) {
      this.alarm.cancel();
      this.alarm.setEnabled(true, function(err, alarm) {
        if (err) {
          callback && callback(err, alarm);
          return;
        }
        AlarmList.refreshItem(alarm);
        AlarmList.banner.show(alarm.getNextAlarmFireTime());
        AlarmManager.updateAlarmStatusBar();
        callback && callback(null, alarm);
      });
    } else {
      // error
      if (callback) {
        callback(error);
      }
    }

    return !error;
  },

  delete: function aev_delete(callback) {
    if (!this.alarm.id) {
      setTimeout(callback.bind(null, new Error('no alarm id')), 0);
      return;
    }

    this.alarm.delete(function aev_delete(err, alarm) {
      AlarmList.refresh();
      AlarmManager.updateAlarmStatusBar();
      callback && callback(err, alarm);
    });
  }

});

return AlarmEdit;
});
