// This Source Code Form is subject to the terms of the Mozilla Public
// License, v. 2.0. If a copy of the MPL was not distributed with this
// file, You can obtain one at https://mozilla.org/MPL/2.0/.

use inherent::inherent;

use super::{CommonMetricData, MetricId};

use crate::ipc::need_ipc;

/// Developer-facing API for recording URL metrics.
///
/// Instances of this class type are automatically generated by the parsers
/// at build time, allowing developers to record values that were previously
/// registered in the metrics.yaml file.
#[derive(Clone)]
pub enum UrlMetric {
    Parent(glean::private::UrlMetric),
    Child(UrlMetricIpc),
}
#[derive(Clone, Debug)]
pub struct UrlMetricIpc;

impl UrlMetric {
    /// Create a new Url metric.
    pub fn new(_id: MetricId, meta: CommonMetricData) -> Self {
        if need_ipc() {
            UrlMetric::Child(UrlMetricIpc)
        } else {
            UrlMetric::Parent(glean::private::UrlMetric::new(meta))
        }
    }

    #[cfg(test)]
    pub(crate) fn child_metric(&self) -> Self {
        match self {
            UrlMetric::Parent(_) => UrlMetric::Child(UrlMetricIpc),
            UrlMetric::Child(_) => panic!("Can't get a child metric from a child metric"),
        }
    }
}

#[inherent]
impl glean::traits::Url for UrlMetric {
    pub fn set<S: Into<std::string::String>>(&self, value: S) {
        match self {
            UrlMetric::Parent(p) => p.set(value),
            UrlMetric::Child(_) => {
                log::error!("Unable to set Url metric in non-main process. Ignoring.");
                // TODO: Record an error.
            }
        };
    }

    pub fn test_get_value<'a, S: Into<Option<&'a str>>>(
        &self,
        ping_name: S,
    ) -> Option<std::string::String> {
        let ping_name = ping_name.into().map(|s| s.to_string());
        match self {
            UrlMetric::Parent(p) => p.test_get_value(ping_name),
            UrlMetric::Child(_) => {
                panic!("Cannot get test value for Url metric in non-parent process!")
            }
        }
    }

    pub fn test_get_num_recorded_errors(&self, error: glean::ErrorType) -> i32 {
        match self {
            UrlMetric::Parent(p) => p.test_get_num_recorded_errors(error),
            UrlMetric::Child(_) => panic!(
                "Cannot get the number of recorded errors for Url metric in non-parent process!"
            ),
        }
    }
}

#[cfg(test)]
mod test {
    use crate::{common_test::*, ipc, metrics};

    #[test]
    fn sets_url_value() {
        let _lock = lock_test();

        let metric = &metrics::test_only_ipc::a_url;

        metric.set("https://example.com");

        assert_eq!(
            "https://example.com",
            metric.test_get_value("store1").unwrap()
        );
    }

    #[test]
    fn url_ipc() {
        // UrlMetric doesn't support IPC.
        let _lock = lock_test();

        let parent_metric = &metrics::test_only_ipc::a_url;

        parent_metric.set("https://example.com/parent");

        {
            let child_metric = parent_metric.child_metric();

            let _raii = ipc::test_set_need_ipc(true);

            // Instrumentation calls do not panic.
            child_metric.set("https://example.com/child");

            // (They also shouldn't do anything,
            // but that's not something we can inspect in this test)
        }

        assert!(ipc::replay_from_buf(&ipc::take_buf().unwrap()).is_ok());

        assert!(
            "https://example.com/parent" == parent_metric.test_get_value("store1").unwrap(),
            "Url metrics should only work in the parent process"
        );
    }
}
