/* -*- Mode: C++; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* vim: set ts=8 sts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "DragEvent.h"

#include "mozilla/dom/MouseEventBinding.h"
#include "mozilla/ImageInputTelemetry.h"
#include "mozilla/MouseEvents.h"
#include "nsContentUtils.h"

namespace mozilla::dom {

DragEvent::DragEvent(EventTarget* aOwner, nsPresContext* aPresContext,
                     WidgetDragEvent* aEvent)
    : MouseEvent(
          aOwner, aPresContext,
          aEvent ? aEvent : new WidgetDragEvent(false, eVoidEvent, nullptr)) {
  if (aEvent) {
    mEventIsInternal = false;
  } else {
    mEventIsInternal = true;
    mEvent->mRefPoint = LayoutDeviceIntPoint(0, 0);
    mEvent->AsMouseEvent()->mInputSource =
        MouseEvent_Binding::MOZ_SOURCE_UNKNOWN;
  }
}

void DragEvent::InitDragEventInternal(
    const nsAString& aType, bool aCanBubble, bool aCancelable,
    nsGlobalWindowInner* aView, int32_t aDetail, double aScreenX,
    double aScreenY, double aClientX, double aClientY, bool aCtrlKey,
    bool aAltKey, bool aShiftKey, bool aMetaKey, uint16_t aButton,
    EventTarget* aRelatedTarget, DataTransfer* aDataTransfer) {
  NS_ENSURE_TRUE_VOID(!mEvent->mFlags.mIsBeingDispatched);

  MouseEvent::InitMouseEventInternal(aType, aCanBubble, aCancelable, aView,
                                     aDetail, aScreenX, aScreenY, aClientX,
                                     aClientY, aCtrlKey, aAltKey, aShiftKey,
                                     aMetaKey, aButton, aRelatedTarget);
  if (mEventIsInternal) {
    mEvent->AsDragEvent()->mDataTransfer = aDataTransfer;
  }
}

DataTransfer* DragEvent::GetDataTransfer() {
  // the dataTransfer field of the event caches the DataTransfer associated
  // with the drag. It is initialized when an attempt is made to retrieve it
  // rather that when the event is created to avoid duplicating the data when
  // no listener ever uses it.
  if (!mEvent || mEvent->mClass != eDragEventClass) {
    NS_WARNING("Tried to get dataTransfer from non-drag event!");
    return nullptr;
  }

  WidgetDragEvent* dragEvent = mEvent->AsDragEvent();
  // for synthetic events, just use the supplied data transfer object even if
  // null
  if (!mEventIsInternal) {
    nsresult rv = nsContentUtils::SetDataTransferInEvent(dragEvent);
    NS_ENSURE_SUCCESS(rv, nullptr);
  }

  // Only collect image input telemetry on external drop events, unless in
  // automation. In tests we skip this check using the pref, since we cannot
  // synthesize external events.
  if ((!mEventIsInternal ||
       StaticPrefs::privacy_imageInputTelemetry_enableTestMode()) &&
      !mImageInputTelemetryCollected) {
    ImageInputTelemetry::MaybeRecordDropImageInputTelemetry(
        dragEvent, GetParentObject()->PrincipalOrNull());
    mImageInputTelemetryCollected = true;
  }

  return dragEvent->mDataTransfer;
}

// static
already_AddRefed<DragEvent> DragEvent::Constructor(
    const GlobalObject& aGlobal, const nsAString& aType,
    const DragEventInit& aParam) {
  nsCOMPtr<EventTarget> t = do_QueryInterface(aGlobal.GetAsSupports());
  RefPtr<DragEvent> e = new DragEvent(t, nullptr, nullptr);
  bool trusted = e->Init(t);
  e->InitDragEventInternal(
      aType, aParam.mBubbles, aParam.mCancelable, aParam.mView, aParam.mDetail,
      aParam.mScreenX, aParam.mScreenY, aParam.mClientX, aParam.mClientY,
      aParam.mCtrlKey, aParam.mAltKey, aParam.mShiftKey, aParam.mMetaKey,
      aParam.mButton, aParam.mRelatedTarget, aParam.mDataTransfer);
  e->InitializeExtraMouseEventDictionaryMembers(aParam);
  e->SetTrusted(trusted);
  e->SetComposed(aParam.mComposed);
  return e.forget();
}

}  // namespace mozilla::dom

using namespace mozilla;
using namespace mozilla::dom;

already_AddRefed<DragEvent> NS_NewDOMDragEvent(EventTarget* aOwner,
                                               nsPresContext* aPresContext,
                                               WidgetDragEvent* aEvent) {
  RefPtr<DragEvent> event = new DragEvent(aOwner, aPresContext, aEvent);
  return event.forget();
}
