/* >>>>>>>>>> BEGIN source/__preamble__.js */
/*! @license
==========================================================================
CoreTest 1.0 - JavaScript Testing Library
copyright 2006-2009, Sprout Systems Inc., Apple Inc. and contributors.

Permission is hereby granted, free of charge, to any person obtaining a 
copy of this software and associated documentation files (the "Software"), 
to deal in the Software without restriction, including without limitation 
the rights to use, copy, modify, merge, publish, distribute, sublicense, 
and/or sell copies of the Software, and to permit persons to whom the 
Software is furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in 
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
DEALINGS IN THE SOFTWARE.

CoreTest is part of the SproutCore project.

SproutCore and the SproutCore logo are trademarks of Sprout Systems, Inc.

For more information visit http://www.sproutcore.com/core_test

==========================================================================
@license */

"use modules false";
"use loader false";

/* >>>>>>>>>> BEGIN package_info.js */
;tiki.register('core_test', {
  "packages": {
    "tiki": {
      "scripts": [
        {
          "url": "/static/tiki/en/9f76cda2d7e44b7acc4c94bb51f3277709b0cb56/javascript.js",
          "id": "tiki:en/9f76cda2d7e44b7acc4c94bb51f3277709b0cb56/javascript.js"
        }
      ]
    }
  },
  "stylesheets": [
    {
      "url": "/static/core_test/en/7cad3ee1925b727665f4dadcb6cc4ca4cd6b94d2/stylesheet.css",
      "id": "core_test:en/7cad3ee1925b727665f4dadcb6cc4ca4cd6b94d2/stylesheet.css"
    }
  ],
  "depends": [
    "tiki"
  ],
  "scripts": [
    {
      "url": "/static/core_test/en/7cad3ee1925b727665f4dadcb6cc4ca4cd6b94d2/javascript.js",
      "id": "core_test:en/7cad3ee1925b727665f4dadcb6cc4ca4cd6b94d2/javascript.js"
    }
  ]
});

/* >>>>>>>>>> BEGIN source/lib/assert.js */
tiki.module('core_test:assert',function(require,exports,module,tiki){// ==========================================================================
// Project:   CoreTest Unit Testing Library
// Copyright: ©2010 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================

var Ct    = require('core'),
    utils = require('utils'),
    assert = {};

require('system/equiv'); // add Ct.equiv

/** @file

  Implements a CommonJS compatible assertion API.  Combined with the CommonJS
  standard "test" module, you can implement generic tests that should run on
  any CommonJS platform.
  
  h2. Example

  {{{
    var assert = require('assert', 'core_test'),

    exports.testFooIsBar = function() {
      assert.ok('foo' typeof 'bar', 'this will actually fail') ;
    };
    
    // other tests here - beginning with the word "test"
    
    // run tests like this
    require('test', 'core_test').run(exports);
  }}}
  
*/

// ..........................................................
// PUBLIC API
// 

/**
  Core assertion API defined as a mixin.  This API is mixed into the Ct.Test
  object and also wrapped here to export the standardized API.
*/
exports.xCoreTestAssertable = assert;
exports.AssertionError = Ct.AssertionError;

/**
  Tests whether a value is truthy (as determined by !!guard).  This is 
  equivalent to assert.equal(true, guard).  To test for strict equality use 
  assert.strictEqual(true, guard).
  
  @param {Object} value
    value to test
  
  @param {String} message
    optional message
    
  @returns {void}
*/
assert.ok = function ok(value, message) {
  this.assert(!!value, message, !!value, true);
};

/**
  Tests whether two value are coersively equal (i.e. ==).  To test for strict
  equality use assert.strictEqual(actual, expected).
  
  @param {Object} actual
    The computed value to test
    
  @param {Object} expected
    The expected value
    
  @param {String} message
    optional message 
    
  @returns {void}
*/
assert.equal = function(actual, expected, message) {
  message = (message||'') + ' should be equal';
  return this.assert(actual==expected, message, actual, expected);
};

/**
  Tests whether two value are coersively not equal (i.e. ==).  To test for 
  strict inequality use assert.strictNotEqual(actual, expected).
  
  @param {Object} actual
    The computed value to test
    
  @param {Object} expected
    The expected value
    
  @param {String} message
    optional message 
    
  @returns {void}
*/
assert.notEqual = function(actual, expected, message) {
  message = (message||'') + ' should not be equal';
  return this.assert(actual != expected, message, actual, expected);
};


/**
  Tests whether two value are equal by recursively evaluating each property on 
  both objects.
  
  @param {Object} actual
    The computed value to test
    
  @param {Object} expected
    The expected value
    
  @param {String} message
    optional message 
    
  @returns {void}
*/
assert.deepEqual = function(actual, expected, message) {
  message = message + ' should be deep equal';
  return this.assert(Ct.equiv(actual, expected), message, actual, expected);
};

/**
  Tests that two value are not equal by recursively evaluating each property 
  on both objects.
  
  @param {Object} actual
    The computed value to test
    
  @param {Object} expected
    The expected value
    
  @param {String} message
    optional message 
    
  @returns {void}
*/
assert.notDeepEqual = function(actual, expected, message) {
  message = message + ' should not be deep equal';
  return this.assert(!Ct.equiv(actual, expected), message, actual, expected);
};

/**
  Tests that two value are strictly equal (i.e. ===).  To test for coercive
  equality use assert.equal(actual, expected).
  
  @param {Object} actual
    The computed value to test
    
  @param {Object} expected
    The expected value
    
  @param {String} message
    optional message 
    
  @returns {void}
*/
assert.strictEqual = function(actual, expected, message) {
  message = message + ' should be strictly';
  return this.assert(actual === expected, message, actual, expected);
};

/**
  Tests whether two value are not strictly equal (i.e. ==).  To test for 
  coercive inequality use assert.notEqual(actual, expected).
  
  @param {Object} actual
    The computed value to test
    
  @param {Object} expected
    The expected value
    
  @param {String} message
    optional message 
    
  @returns {void}
*/
assert.notStrictEqual = function(actual, expected, message) {
  message = message + ' should not be strictly equal';
  return this.assert(actual !== expected, message, actual, expected);
};


function _throws (context, shouldThrow, block, err, message) {
  var exception = null,
      didThrow = false,
      typeMatters = true,
      pass;

  // normalize
  if (err) {
    if ('string' === typeof err) {
      message = err;
      err     = null;
      typeMatters = false;
    }
  } else typeMatters = false ;

  message = message || "";

  // no execute the passed function and save results
  try {
    block();
  } catch (e) {
    didThrow = true;
    exception = e;
  }

  // handle case where we expected it throw...
  if (shouldThrow) {
    
    if (typeMatters) {
      if ("function" === typeof err) {
        pass = didThrow && (exception instanceof err);
      } else {
        pass = didThrow && (exception === err);
      }
      
      message = message + " expected exception of type " + err.toString();

    } else {
      pass = didThrow ;
      message = message + " expected exception";
    }

  // did not expect an exception of specified type (or any one)
  } else {
    
    if (typeMatters) {
      if ("function" === typeof err) {
        pass = !(didThrow || (exception instanceof err));
      } else {
        pass = !(didThrow || (exception === err));
      }

      message = message+" did not expect exception of type "+err.toString();
    
    } else {
      pass = !didThrow;
      message = message + " did not expect exception";
    }
  }
  
  if (err && err.name) err = err.name;
  context.assert(pass, message, exception, err);
}

/**
  Invokes the passed function, expecting it to throw an exception.  If you 
  pass an Error object as a second parameter, then verifies that the thrown
  exception matches the passed type.
  
  @param {Function} func
    Callback to invoke
    
  @param {Error} errorOpt
    (Optional) exception to test against
    
  @param {String} message
    (Optional) message 
    
  @returns {void}
*/
assert.throws = function(func, errorOpt, message) {
  return _throws(this, true, func, errorOpt, message);
};

/**
  Invokes the passed function, expecting it to not throw an exception.  If you 
  pass an Error object as a second parameter, then verifies that if an 
  exception is throw, it does not match the specified type.  This is useful
  for example if you want to verify that a specific error condition does NOT
  occur.
  
  @param {Function} func
    Callback to invoke
    
  @param {Error} errorOpt
    (Optional) exception to test against
    
  @param {String} message
    (Optional) message 
    
  @returns {void}
*/
assert.doesNotThrow = function(func, errorOpt, message) {
  return _throws(this, false, func, errorOpt, message);
};

// ..........................................................
// EXPORT STANDARD API
// 

// bind assertion API to CoreTest namespace
function _bind(key) {
  var func = assert[key];
  return function() { return func.apply(Ct, arguments); };
}

for(var k in assert) {
  if (!assert.hasOwnProperty(k)) continue ;
  exports[k] = _bind(k);
}
;});
/* >>>>>>>>>> BEGIN source/lib/core.js */
tiki.module('core_test:core',function(require,exports,module,tiki){// ==========================================================================
// Project:   SproutCore Unit Testing Library
// Copyright: ©2006-2009 Sprout Systems, Inc. and contributors.
//            Portions ©2008-2009 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================
/*globals CoreTest exports */

var utils = require('utils'), 
    Promise = require('promise', 'tiki'),
    CoreTest, Ct;

/** @namespace

  CoreTest is the unit testing library for SproutCore.  It includes a test 
  runner based on QUnit with some useful extensions for testing SproutCore-
  based applications.
  
  You can use CoreTest just like you would use QUnit in your tests directory.
*/
CoreTest = {
  
  /**
    @class

    Error thrown by assertions that have failed.  For these assertions you may
    also set the CoreTest.throwsOnFailure property to false.

    When throwing an error, pass any properties you want copied onto the 
    error. The message, actual, and expected are required to properly log the 
    output.

    h2. Examples

    Throwing an error:

    {{{
      new AssertionError({ message: "foo", expected: "bar", actual: "baz" });
    }}}

    Optional (non-standard) method using parameters:

    {{{
      new AssertionError('actual', 'expected', 'message');
    }}}

    @since SproutCore 1.1
  */
  AssertionError: utils.extend(Error, {

    init: function(actual, expected, message) {
      if (arguments.length === 1) utils.mixin(this, actual);
      else {
        this.actual   = actual;
        this.expected = expected;
        this.message  = message;
      }

      this.desc = this.message;

      var ret = ['AssertionError:'];
      if (this.message) ret.push(this.message);
      if ((this.actual!==undefined) || (this.expected!==undefined)) {
        var act = Ct.dump(this.actual),
            exp = Ct.dump(this.expected);
        ret.push(utils.fmt('(actual = "%@" - expected = "%@")', act, exp));
      }
      this.message = ret.join(' ');

      return this ;
    },

    toString: function() {
      return this.message;
    }

  }),

  // ..........................................................
  // PRIMITIVE API
  // 

  /** @private 
    set or return the current tail promise for the schedule
  */
  _schedule: function(pr) {
    if (arguments.length===0) {
      if (!(pr = this._scheduleTail)) {
        pr = this._scheduleTail = Promise.create('CoreTest.head');
        tiki.ready(pr, pr.resolve);
      }
      return pr ;
    } else {
      this._scheduleTail = pr;
      return this;
    }
  },
  
  /**
    Adds a plan's modules and tests to the current test schedule.  If the 
    schedule has not been started yet, it will begin to run when the browser
    is ready to start handling events.
    
    Normally you will not call this method with any parameters, but instead
    let is schedule the default plan for you instead (which you fill in with
    the test planning API).  
    
    However, if you want to provide your own custom plan or filter the tests
    that will be run, you may want to pass your own plan object and filters.
    
    @param {Plan} plan
      (Optional) Plan to schedule or null if you want to use the default plan
      
    @param {Hash} filter
      (Optional) Nested hash of module names and test names describing the 
      tests you want to actually run.  Only those tests will be added to the
      schedule.
      
    @returns {CoreTest} receiver
  */
  run: function(plan, filter) {
    if (!plan) plan = this._defaultPlan();

    var pr = this._schedule();
    this._schedule(plan.schedule(pr, filter));
    
    // reset default plan once it has been scheduled to run
    if (plan === this.defaultPlan) this.defaultPlan = null ; 
    return this;
  },
  
  /**
    Used for testing.  Adds the passed callback to the schedule, executing it
    after the current end of schedule completes.  Note that if you do not 
    pass a cancelHandler, your successHandler will be invoked in either case.
    
    @param {Object} context 
      (Optional) context object
    
    @param {Function} successHandler
      function to invoke if previous promise succeeds
      
    @param {Function} cancelHandler
      function to invoke if previous promise is cancelled
      
    @returns {Promise} the next promise in the schedule
  */
  then: function(context, successHandler, cancelHandler) {
    
    // normalize arguments
    if (arguments.length<3 && ('function' === typeof context)) {
      cancelHandler = successHandler;
      successHandler = context;
      context = this;
    }  
    if (!cancelHandler) cancelHandler = successHandler;
    
    var pr = this._schedule();
    pr = pr.then(context, successHandler, cancelHandler);
    this._schedule(pr);
    
    return pr ;
  },

  /**
    Primitive method logs an assertion.  All other assertion methods boil down
    to this one.  You can only call this method from within the setup, 
    teardown or test functions of a test.  Outside of these functions, this
    method will throw an exception.
    
    @param {Boolean} pass 
      true if the assertion passed, false if it failed

    @param {String} message
      (Optional) descriptive message
      
    @param {Object} expected
      (Optional) expected value
      
    @param {Object} actual
      (Optional) actual value
      
    @returns {CoreTest} receiver
  */
  assert: function(pass, message, expected, actual) {
    if (!this.currentTest) throw "cannot assert outside of a plan";
    this.currentTest.assert(pass, message, expected, actual);
    return this ;
  },
  
  /**
    Primitive logs an error. (i.e. an exception, not test failure)  
    
    @param {String|Error} err
      The error object or a message to log
    
    @returns {CoreTest} receiver
  */
  error: function(err) {
    if (!this.currentTest) throw "cannot log error outside of a plan";
    this.currentTest.error(err);
    return this ;
  },

  /**
    Primitive logs non-test information.  
    
    @param {String} message
      The message to log
    
    @returns {CoreTest} receiver
  */
  info: function(message) {
    if (!this.currentTest) throw "cannot log error outside of a plan";
    this.currentTest.info(message);
    return this ;
  },
  
  // ..........................................................
  // DEFAULT PLANNING API
  // 
  // these setup and schedule a default plan
  
  
  /**
    The current plan being planned.
    
    @property {Ct.Plan}
  */
  defaultPlan: null,

  _defaultPlan: function() {
    if (!this.defaultPlan) this.defaultPlan = new Ct.Plan('default');
    return this.defaultPlan;
  },
  
  module: function(moduleName, opts) {
    this._defaultPlan().module(moduleName, opts);
    return this;
  },
  
  setup: function(handler) {
    this._defaultPlan().setup(handler);
    return this;
  },

  teardown: function(handler) {
    this._defaultPlan().teardown(handler);
    return this;
  },

  test: function(testName, handler) {
    this._defaultPlan().test(testName, handler);
    return this;
  },
  
  // noop makes it easy to turn of tests - just change test() to notest()
  notest: function(testName, handler) {
    
  },
  
  // ..........................................................
  // LOGGER CONSTANTS
  // 
  
  /** Test is OK */
  OK: 'passed',
  
  /** alternative to OK */
  PASS: 'passed',
  
  /** Test failed */
  FAIL: 'failed',
  
  /** Test raised exception */
  ERROR: 'errors',
  
  /** Test raised warning */
  WARN: 'warnings',

  /** Test logged some info */
  INFO: 'info',
  
  /** In setup mode */
  SETUP_MODE: 'setup',
  
  /** In teardown mode */
  TEARDOWN_MODE: 'teardown',
  
  /** In test mode */
  TEST_MODE: 'test',
  
  /** Test is in planning mode (i.e. not running) */
  PLANNING_MODE: 'planning'
  
};

Ct = CoreTest;

// ..........................................................
// FINAL SETUP
// 

exports = module.exports = Ct;
exports.CoreTest = exports.Ct = Ct;

// Import some other modules required by code referenced here
require('system/dump');
require('system/plan');

// Choose a default logger based on platform.  Used in case no override is
// specified
var DefaultLogger;
if (require('system', 'default').platform === 'tiki') {
   DefaultLogger = require('loggers/browser');
} else DefaultLogger = require('loggers/default');

/**
  Default logger used by a plan if you don't override it by setting Ct.logger
  or Plan.logger
  
  @property {DefaultLogger}
*/
Ct.defaultLogger = new DefaultLogger('default');


utils.setupDisplayNames(CoreTest, 'CoreTest');

;});
/* >>>>>>>>>> BEGIN source/lib/index.js */
tiki.module('core_test:index',function(require,exports,module,tiki){// ==========================================================================
// Project:   SproutCore Unit Testing Library
// Copyright: ©2006-2009 Sprout Systems, Inc. and contributors.
//            Portions ©2008-2009 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================
/*globals CoreTest exports module */

// Notify the build tools of the exports we plan to manually define
"use exports CoreTest ok same equal notEqual deepEqual deepNotEqual raises equals shouldThrow plan module setup teardown test htmlbody expect";

// Core exported API
exports = module.exports = require('core');

require('system/ext');

require('system/plan');
require('system/module');
require('system/test');

// Additional public API - these enhance CoreTest
require('system/dump');
require('system/equiv');
require('system/stub');
require('system/suite');

;});
/* >>>>>>>>>> BEGIN source/lib/loggers/browser.js */
tiki.module('core_test:loggers/browser',function(require,exports,module,tiki){// ==========================================================================
// Project:   CoreTest Unit Testing Library
// Copyright: ©2010 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================

var utils = require('utils'),
    Ct    = require('core'),
    $     = require('private/jquery');
    
require('loggers/default'); // add Ct.DefaultLogger

var textDiv, textNode;    

// convert a string into HTML-escaped text.
function _text(str) {
  if (!textNode) textNode = document.createTextNode('');
  if (!textDiv) {
    textDiv = document.createElement('div');
    textDiv.appendChild(textNode);
  }
  
  textNode.nodeValue = str;
  return textDiv.innerHTML.toString().replace(/\n/g, '<br>');
}

// knows how to emit a single assertion
var AssertionEntry = utils.extend({
  init: function(owner, status, message) {
    this.owner   = owner;
    this.message = message;
    this.status  = status;
  },
  
  // expects status, message
  template: '<li class="assertion %@1"><span class="name">%@2</span><span class="status">%@1</span></li>',
  
  emit: function() {
    return utils.fmt(this.template, _text(this.status), _text(this.message));
  }

});

// knows how to emit a single test (along with its assertions)
var TestEntry = utils.extend({
  init: function(owner, testName) {
    this.owner = owner;
    this.name  = testName;
    this.assertions = [];
    this.status = { passed: 0, failed: 0, errors: 0, warnings: 0 };
  },
  
  // add an assertion to the log - also updates stats on the assertion
  add: function(status, message) {
    var entry = new AssertionEntry(this, status, message);
    this.assertions.push(entry);
    if (this.status[status] !== undefined) this.status[status]++;
    if (this.plan()[status] !== undefined) this.plan()[status]++;
    this.owner.pass(status === 'passed');
    this.plan().assertions++;
    
    return entry ;
  },
  
  plan: function() { return this.owner.plan(); },
  
  
  // expects status, name, assertions, passed, failed, errors, warnings
  template: ['<li class="test %@1">',
    '<span class="name">%@2</span>',
    '<span class="status">',
      '<span class="passed">%@4</span>',
      '<span class="warnings">%@7</span>',
      '<span class="failed">%@5</span>',
      '<span class="errors">%@6</span>',
    '</span>',
    '<ul>%@3</ul>',
  '</li>'].join(''),
  
  // emits the result
  emit: function() {
    var statsum = [], 
        status  = this.status, 
        assertions = [], 
        key, len, idx ;
        
    for(key in status) if (status[key]>0) statsum.push(_text(key));
    
    len = this.assertions.length;
    for(idx=0;idx<len;idx++) assertions.push(this.assertions[idx].emit());
    assertions = assertions.join('');
    
    return utils.fmt(this.template, statsum, _text(this.name), assertions, status.passed, status.failed, status.errors, status.warnings);
  }
  
});

// knows how to emit a module
var ModuleEntry = utils.extend({
  
  init: function(owner, moduleName) {
    this.name = moduleName;
    this.owner = owner;
    this.entries = []; // add another module or test here
    this._modules = {}; // for lookup
    this._tests = {}; // for lookup
    this.didPass = true;
  },

  // add or get module by name
  module: function(moduleName) {
    if (this._modules[moduleName]) return this._modules[moduleName];
    var ret = new ModuleEntry(this, moduleName);
    this._modules[moduleName] = ret;
    this.entries.push(ret);
    return ret ;
  },
  
  // add or get a test by name
  test: function(testName) {
    if (this._tests[testName]) return this._tests[testName];
    var ret = new TestEntry(this, testName);
    this._tests[testName] = ret ;
    this.entries.push(ret);
    this.plan().tests++;
    return ret ;
  },
  
  pass: function(aFlag) {
    if (!aFlag) this.didPass = false;
    this.owner.pass(aFlag);
  },
  
  plan: function() { return this.owner.plan(); },
  
  template: ['<li class="module %@3">',
    '<span class="name">%@1</span>',
    '<ul>%@2</ul>',
  '</li>'].join(''),
  
  emit: function() {
    var assertions = [],
        key, len, idx;
        
    len = this.entries.length;
    for(idx=0;idx<len;idx++) assertions.push(this.entries[idx].emit());
    assertions = assertions.join('');
    
    var passed = this.didPass ? 'passed' : '';
    return utils.fmt(this.template, _text(this.name), assertions, passed);
  }

});

// knows how to emit a full plan
var PlanEntry = utils.extend({
  
  init: function(planName) {
    this.name = planName;
    this.entries = [];
    this._modules = {};
    this.passed = this.errors = this.failed = this.warnings = 0;
    this.tests = this.assertions = 0 ;
  },
  
  pass: function() { 
    // noop
  },
  
  plan: function() { return this; },
  
  module: function(moduleName) {
    if (this._modules[moduleName]) return this._modules[moduleName];
    var ret = new ModuleEntry(this, moduleName);
    this._modules[moduleName] = ret ;
    this.entries.push(ret);
    return ret;
  },
  
  template: ['<li class="plan">',
    '<span class="name">%@1</span>',
    '<ul>%@2</ul>',
  '</li>'].join(''),
  
  emit: function() {
    var assertions = [],
        key, len, idx;
        
    len = this.entries.length;
    for(idx=0;idx<len;idx++) assertions.push(this.entries[idx].emit());
    assertions = assertions.join('');
    
    return utils.fmt(this.template, _text(this.name), assertions);
  }

});


// default template used for display.  Note we use classes here - not IDs.  
// This way multiple instances can be on the same page at once.
var html = ['<div class="core-test">',
  '<div class="useragent">UserAgent</div>',
  '<div class="testresult">',
    '<label class="hide-passed">',
      '<input type="checkbox" checked="" /> Hide passed tests',
    '</label>',
    '<span class="final-status">Running...</span>',
  '</div>',
  '<ul class="detail">',
  '</ul>',
'</div>'].join('');


/**
  BrowserLogger logs output to the HTML display in a browser.
  
  @extends Ct.DefaultLogger
*/
Ct.BrowserLogger = utils.extend(Ct.DefaultLogger, 
  /** @scope Ct.DummyLogger.prototype */{
  
  name: 'browser',

  init: function() {
    Ct.DefaultLogger.prototype.init.apply(this, arguments);
    this.plans = [];
    
    this.status = { 
      passed: 0, 
      failed: 0, 
      errors: 0, 
      warnings: 0, 
      tests: 0, 
      assertions: 0 
    }; 
  },
  
  setupDisplay: function() {
    if (this._displayWasSetup) {
      return;
    }
    
    var layer, logger;
    
    layer = this.layer = $(html);
    $('body').append(layer);
    
    // write in the user agent
    layer.find('.useragent').text(navigator.userAgent);
    
    // listen to change event
    this.checkboxLayer = layer.find('.hide-passed input[type=checkbox]');
    logger = this;
    this.checkboxLayer.change(function() {
      logger.hidePassedTestsDidChange(logger.checkboxLayer.attr('checked'));
    }); 
    
  },
  
  hidePassedTestsDidChange: function(aFlag) {
    this.setupDisplay();
    if (aFlag) this.layer.find('ul.detail').addClass('hide-passed');
    else this.layer.find('ul.detail').removeClass('hide-passed');
  },

  emit: function(plan) {
    this.setupDisplay();
    var status = this.status, 
        ret, idx, key;
        
    ret = plan.emit();
    this.layer.find('ul.detail').append($(ret));    

    for(key in status) {
      if (!status.hasOwnProperty(key)) continue;
      status[key] += plan[key];
    }
    this.summarize(status);
  },

  summarize: function(status) {
    this.setupDisplay();
    var ret = [];
    ret.push(utils.fmt('Completed %@ assertions in %@ tests: ', status.assertions, status.tests));
    ret.push(utils.fmt('<span class="passed">%@ passed</span>', status.passed));
    
    var key, hasErrors;
    hasErrors = (status.failed + status.errors + status.warnings)>0;
    if (hasErrors) {
      for(key in status) {
        if (!status.hasOwnProperty(key) || (key==='passed')) continue;
        if ((key==='tests') || (key==='assertions')) continue;
        ret.push(utils.fmt('<span class="%@1">%@2 %@1</span>', key, status[key]));
      }
    }
    
    this.layer.find('.final-status').html(ret.join(''));
    
    var checkbox = this.layer.find('.hide-passed input');
    if (hasErrors) {
      checkbox.attr('disabled', false).attr('checked', true);
      this.hidePassedTestsDidChange(true);
    } else {
      checkbox.attr('disabled', true).attr('checked', false);
      this.hidePassedTestsDidChange(false);
    }
    checkbox = null;
  },
  
  // ..........................................................
  // CORE API - Overide in your subclass
  // 
  
  begin: function(planName) {
    var plan = new PlanEntry(planName);
    this.plans.push(plan);
    this.currentPlan = plan;
  },
  
  end: function(planName) {
    this.emit(this.currentPlan); 
    this.currentPlan = null;
  },

  
  add: function(status, testInfo, message) {
    
    var plan = this.currentPlan;
    if (!plan) throw "add called outside of plan";
    
    var testName = testInfo.testName,
        moduleNames = testInfo.moduleNames,
        len = moduleNames ? moduleNames.length : 0,
        idx, cur ;

    if (len===0) {
      moduleNames = ['default']; len = 1;  
    }
    
    cur = plan;
    for(idx=0;idx<len;idx++) cur = cur.module(moduleNames[idx]);
    cur = cur.test(testName);
    cur.add(status, message);
  }
});

// make available to those directly importing this module
exports = module.exports = Ct.BrowserLogger;
exports.BrowserLogger = Ct.BrowserLogger;

;});
/* >>>>>>>>>> BEGIN source/lib/loggers/default.js */
tiki.module('core_test:loggers/default',function(require,exports,module,tiki){// ==========================================================================
// Project:   CoreTest Unit Testing Library
// Copyright: ©2010 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================

"use exports DefaultLogger";

var utils = require('utils'),
    Ct    = require('core'),
    hasConsole = 'undefined' !== typeof console; 

/**
  Defines a logger class for logging test output.  You can subclass this 
  logger to redirect test output to another source.  All test loggers by 
  default can be chained, allowing you to both log to a console and capture
  output for Selenium, for example.

  To implement a logger, you need to implement the following methods:
  
  - begin(planName)
  - end(planName)
  - add(Ct.PASS | Ct.FAIL | Ct.WARN | Ct.ERROR,  testInfo, message)

  To setup a logger, just create the logger using LoggerName.create('name').
  To chain loggers use then().
  
  {{{
    LoggerA = Ct.DefaultLogger.create('default');
    LoggerB = Ct.BrowserLogger.create('browser').then(LoggerA);
  }}}
  
*/
Ct.DefaultLogger = utils.extend(/** @scope Ct.DefaultLogger.prototype */{
  
  /**
    The name of the logger.  Can be used for output.
  */
  name: 'unknown',
  
  /**
    The next logger in the chain.  Set with then().
    
    @type {Ct.DefaultLogger}
  */
  next: null,

  init: function(name) {
    this.name = name;
  },

  /**
    Sets the next logger in the chain; returns receiver
    
    @param {Ct.DefaultLogger} the next logger
    @returns {Ct.DefaultLogger} the current logger
  */
  then: function(next) {
    this.next = next; 
    return this ;
  },
  
  // ..........................................................
  // CORE API - Overide in your subclass
  // 
  
  /**
    Called when a new plan begins to run.  This can be used to setup any 
    default settings.  Only one plan can run at a time.
    
    Override this method in your subclass
    
    @param {String} planName the name of the plan to invoke
    @returns {void}
  */
  begin: function(planName) {
    var state = this.state;
    if (!state) state = this.state = {};
    
    if (state.isRunning) throw "logger only supports one plan at a time";
    state.isRunning = true;
    
    if (hasConsole && console.group) console.group(planName);
  },
  
  /**
    Called when a plan is finished funning.  This should be used to cleanup 
    any outstanding info and generate a final report based on collected stats
    
    Override this method in your subclass
    
    @param {String} planName the name of the plan to the invoke
    @returns {void}
  */
  end: function(planName) {
    var state = this.state, loc;
    if (!state || !state.isRunning) throw "plan must be running to end it";
    
    if (hasConsole && console.groupEnd) {
      if (state.testName) console.groupEnd(state.testName);

      // end nested modules
      loc = state.moduleNames ? state.moduleNames.length : -1;
      while(--loc >= 0) console.groupEnd(state.moduleNames[loc]);

      console.groupEnd(planName);
      console.log((planName||'') + ' plan complete.');
    }

    this.state = null; // clear state
  },
  
  /**
    Called to log an assertion out.  First param is the status, second is the
    message.  Override this method in your subclass
    
    @param {String} status 
      status of the message.  Must be Ct.PASS, Ct.FAIL, Ct.ERROR, Ct.WARN
      
    @param {Hash} testInfo
      describes the test.  has moduleName, testName, mode.  mode is one of 
      Ct.SETUP_MODE, Ct.TEARDOWN_MODE, Ct.TEST_MODE
      
    @param {String} message
      optional message explaining the status
      
    @returns {void}
  */
  add: function(status, testInfo, message) {
    
    var state = this.state, 
        testName, moduleNames, testMode, msg, len, idx, loc;
    
    if (!state || !state.isRunning) throw "plan must be running to log it";

    if (!hasConsole) return; // nothing to do

    moduleNames = testInfo.moduleNames;
    if (!moduleNames || moduleNames.length===0) moduleNames = ['default'];

    testName   = testInfo.testName || 'default';
    testMode   = testInfo.mode || 'test';

    // find where the old and new set of modules names diverge
    if (!state.moduleNames) loc = 0;
    else {
      len = state.moduleNames.length;
      loc = -1;
      for(idx=0;(loc<0) && (idx<len); idx++) {
        if (state.moduleNames[idx] !== moduleNames[idx]) loc = idx; 
      }
      if (loc<0) loc = len;
    }
    
    // end current module and start new one if needed
    if (loc !== moduleNames.length) {
      
      // exit current modules if there are any
      idx = state.moduleNames ? state.moduleNames.length : 0;
      if (console.groupEnd && (idx>loc)) {
        console.groupEnd(state.testName);
        while(--idx >= loc) console.groupEnd(state.moduleNames[idx]);
      }
      
      // begin new module if needed
      len = moduleNames.length;
      if (console.group && (loc<len)) {
        for(idx=loc;idx<len;idx++) console.group(moduleNames[idx]);
        console.group(testName);
      }
      
      state.moduleNames = moduleNames;
      state.testName = testName;

    // if module did not change, but test changed, handle that on its own
    } else if (state.testName !== testName) {
      if (state.testName && console.groupEnd) {
        console.groupEnd(state.testName);
      }
      
      if (console.group) console.group(testName);
      state.testName = testName ;
    }

    // now log the message itself
    if (testMode !== Ct.TEST_MODE) {
      msg = utils.fmt('%@: %@ in %@', status, message, testMode);
    } else msg = utils.fmt('%@: %@', status, message);

    switch(status) {
      case Ct.ERROR:
      case Ct.FAIL:
        if (console.error) console.error(msg);
        else console.log(msg);
        break;
        
      case Ct.WARN:
        if (console.warn) console.warn(msg);
        else console.log(msg);
        break;
        
      default:
        if (console.info) console.info(msg);
        else console.log(msg);
    }
  },
  
  // ..........................................................
  // PUBLIC API - entry points to send log
  // 

  /**
    Called whenever a new plan begins to execute.
  */
  planDidBegin: function(planName) {
    this.begin(planName);
    if (this.next) this.next.planDidBegin(plan);
    return this;
  },

  /**
    Called when a plan ends.
  */
  planDidEnd: function(planName) {
    this.end(planName);
    if (this.next) this.next.planDidEnd(plan);
    return this;
  },

  /**
    Called when an assertion passes
  */
  pass: function(testInfo, msg) {
    this.add(Ct.PASS, testInfo, msg);
    if (this.next) this.next.pass(testInfo, msg);
    return this;
  },

  /**
    Called when an assertion fails
  */
  fail: function(testInfo, msg) {
    this.add(Ct.FAIL, testInfo, msg);
    if (this.next) this.next.fail(testInfo, msg);
    return this;
  },

  /**
    Called when an assertion has an error
  */
  error: function(testInfo, msg) {
    this.add(Ct.ERROR, testInfo, msg) ;
    if (this.next) this.next.error(testInfo, msg);
    return this;
  },

  /**
    Called when an assertion as a warning
  */
  warn: function(testInfo, msg) {
    this.add(Ct.WARN, testInfo, msg) ;
    if (this.next) this.next.warn(testInfo, msg);
  },
  
  info: function(testInfo, msg) {
    this.add(Ct.INFO, testInfo, msg) ;
    if (this.next) this.next.info(testInfo, msg);
  }

});

// make available to those directly importing this module
exports = module.exports = Ct.DefaultLogger;
exports.DefaultLogger = Ct.DefaultLogger;

;});
/* >>>>>>>>>> BEGIN source/lib/loggers/dummy.js */
tiki.module('core_test:loggers/dummy',function(require,exports,module,tiki){// ==========================================================================
// Project:   CoreTest Unit Testing Library
// Copyright: ©2010 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================

var utils = require('utils'),
    Ct    = require('core');
    
require('loggers/default'); // add Ct.DefaultLogger
    

/**
  DummyLogger simply logs any results into an internal array.  Most useful 
  for testing log output then testing the test framework itself.
  
  @extends Ct.DefaultLogger
*/
Ct.DummyLogger = utils.extend(Ct.DefaultLogger, 
  /** @scope Ct.DummyLogger.prototype */{
  
  name: 'dummy',

  BEGIN: 'begin',
  
  END: 'end',
  
  TEST: 'test',
  
  /**
    Populated with items describing the log history.  Each item in this 
    array witll be a hash contains the following properties:
    
     - plan: name of the plan
     - module: name of the module
     - test: name of the test
     - message: message
  
    @property {Array}
  */
  history: null,

  /**
    Resets the log history
  */
  reset: function() {
    this.history = [];
  },
  
  init: function() {
    Ct.DefaultLogger.prototype.init.apply(this, arguments);
    this.reset();
  },

  /**
    Looks for a message matching a passed template.  Returns the message if 
    found or null if none matches.
  */
  find: function(templ) {
    var hist = this.history,
        len  = hist ? hist.length : 0,
        idx, item, key, isMatch;
        
    for(idx=0;idx<len;idx++) {
      item = hist[idx];
      isMatch = true; 
      for(key in templ) {
        if (!templ.hasOwnProperty(key)) continue;
        if (item[key] !== templ[key]) isMatch = false;
      }
      if (isMatch) return item;
    }
    
    return null;
  },
  
  /**
    Evaluates the current history against the passed templates, logging
    assertions against the named test
  */
  expect: function(test, templs) {
    var hist = this.history,
        len  = hist ? hist.length : 0,
        idx, item, templ, key;

    // normalize arguments
    if (arguments.length===1) {
      templs = test;
      test   = Ct;
    }

    test.equal(hist.length, templs.length, 'history.length');
    
    for(idx=0;idx<len;idx++) {
      item = hist[idx];
      templ = templs[idx];
      if (!templ) continue; // allow null|undefined to skip items
      
      for(key in templ) {
        if (!templ.hasOwnProperty(key)) continue;
        test.deepEqual(item[key], templ[key], utils.fmt('history[%@].%@', idx, key));
      }
    }
        
  },
  
  /**
    Temporarily redirect logging to this logger.  This will schedule the 
    redirect; not perform it immediately.
  */
  redirect: function() {
    Ct.then(this, this._redirect);
    return this;
  },
  
  _redirect: function() {
    this.previousLogger = Ct.logger;
    Ct.logger = this;  
  },
  
  /** 
    Restore a redirected logger.  This will schedule the restore; not perform
    it immediately.
  */
  restore: function() {
    Ct.then(this, this._restore);
    return this;
  },
  
  _restore: function() {
    Ct.logger = this.previousLogger;
    this.previousLogger = null;
  },
  
  // ..........................................................
  // CORE API - Overide in your subclass
  // 
  
  begin: function(planName) {
    this.history.push({ plan: planName, kind: this.BEGIN });
    this.currentPlan = planName;
  },
  
  end: function(planName) {
    this.history.push({ plan: planName, kind: this.END }) ;
    this.currentPlan = null;
  },
  
  add: function(status, testInfo, message) {
    this.history.push({
      kind: this.TEST,
      plan: this.currentPlan,
      modules: testInfo.moduleNames,
      test: testInfo.testName,
      status: status,
      message: message,
      mode: testInfo.mode
    });
  }
  
});

// make available to those directly importing this module
exports = module.exports = Ct.DummyLogger;
exports.DummyLogger = Ct.DummyLogger;

;});
/* >>>>>>>>>> BEGIN source/lib/private/jquery.js */
tiki.module('core_test:private/jquery',function(require,exports,module,tiki){/*!
 * jQuery JavaScript Library v1.4.1
 * http://jquery.com/
 *
 * Copyright 2010, John Resig
 * Dual licensed under the MIT or GPL Version 2 licenses.
 * http://jquery.org/license
 *
 * Includes Sizzle.js
 * http://sizzlejs.com/
 * Copyright 2010, The Dojo Foundation
 * Released under the MIT, BSD, and GPL Licenses.
 *
 * Date: Mon Jan 25 19:43:33 2010 -0500
 */
(function( window ) {

// Define a local copy of jQuery
var jQuery = function( selector, context ) {
		// The jQuery object is actually just the init constructor 'enhanced'
		return new jQuery.fn.init( selector, context );
	},

	// Map over jQuery in case of overwrite
	_jQuery = window.jQuery,

	// Map over the $ in case of overwrite
	_$ = window.$,

	// Use the correct document accordingly with window argument (sandbox)
	document = window.document,

	// A central reference to the root jQuery(document)
	rootjQuery,

	// A simple way to check for HTML strings or ID strings
	// (both of which we optimize for)
	quickExpr = /^[^<]*(<[\w\W]+>)[^>]*$|^#([\w-]+)$/,

	// Is it a simple selector
	isSimple = /^.[^:#\[\.,]*$/,

	// Check if a string has a non-whitespace character in it
	rnotwhite = /\S/,

	// Used for trimming whitespace
	rtrim = /^(\s|\u00A0)+|(\s|\u00A0)+$/g,

	// Match a standalone tag
	rsingleTag = /^<(\w+)\s*\/?>(?:<\/\1>)?$/,

	// Keep a UserAgent string for use with jQuery.browser
	userAgent = navigator.userAgent,

	// For matching the engine and version of the browser
	browserMatch,
	
	// Has the ready events already been bound?
	readyBound = false,
	
	// The functions to execute on DOM ready
	readyList = [],

	// The ready event handler
	DOMContentLoaded,

	// Save a reference to some core methods
	toString = Object.prototype.toString,
	hasOwnProperty = Object.prototype.hasOwnProperty,
	push = Array.prototype.push,
	slice = Array.prototype.slice,
	indexOf = Array.prototype.indexOf;

jQuery.fn = jQuery.prototype = {
	init: function( selector, context ) {
		var match, elem, ret, doc;

		// Handle $(""), $(null), or $(undefined)
		if ( !selector ) {
			return this;
		}

		// Handle $(DOMElement)
		if ( selector.nodeType ) {
			this.context = this[0] = selector;
			this.length = 1;
			return this;
		}

		// Handle HTML strings
		if ( typeof selector === "string" ) {
			// Are we dealing with HTML string or an ID?
			match = quickExpr.exec( selector );

			// Verify a match, and that no context was specified for #id
			if ( match && (match[1] || !context) ) {

				// HANDLE: $(html) -> $(array)
				if ( match[1] ) {
					doc = (context ? context.ownerDocument || context : document);

					// If a single string is passed in and it's a single tag
					// just do a createElement and skip the rest
					ret = rsingleTag.exec( selector );

					if ( ret ) {
						if ( jQuery.isPlainObject( context ) ) {
							selector = [ document.createElement( ret[1] ) ];
							jQuery.fn.attr.call( selector, context, true );

						} else {
							selector = [ doc.createElement( ret[1] ) ];
						}

					} else {
						ret = buildFragment( [ match[1] ], [ doc ] );
						selector = (ret.cacheable ? ret.fragment.cloneNode(true) : ret.fragment).childNodes;
					}

				// HANDLE: $("#id")
				} else {
					elem = document.getElementById( match[2] );

					if ( elem ) {
						// Handle the case where IE and Opera return items
						// by name instead of ID
						if ( elem.id !== match[2] ) {
							return rootjQuery.find( selector );
						}

						// Otherwise, we inject the element directly into the jQuery object
						this.length = 1;
						this[0] = elem;
					}

					this.context = document;
					this.selector = selector;
					return this;
				}

			// HANDLE: $("TAG")
			} else if ( !context && /^\w+$/.test( selector ) ) {
				this.selector = selector;
				this.context = document;
				selector = document.getElementsByTagName( selector );

			// HANDLE: $(expr, $(...))
			} else if ( !context || context.jquery ) {
				return (context || rootjQuery).find( selector );

			// HANDLE: $(expr, context)
			// (which is just equivalent to: $(context).find(expr)
			} else {
				return jQuery( context ).find( selector );
			}

		// HANDLE: $(function)
		// Shortcut for document ready
		} else if ( jQuery.isFunction( selector ) ) {
			return rootjQuery.ready( selector );
		}

		if (selector.selector !== undefined) {
			this.selector = selector.selector;
			this.context = selector.context;
		}

		return jQuery.isArray( selector ) ?
			this.setArray( selector ) :
			jQuery.makeArray( selector, this );
	},

	// Start with an empty selector
	selector: "",

	// The current version of jQuery being used
	jquery: "1.4.1",

	// The default length of a jQuery object is 0
	length: 0,

	// The number of elements contained in the matched element set
	size: function() {
		return this.length;
	},

	toArray: function() {
		return slice.call( this, 0 );
	},

	// Get the Nth element in the matched element set OR
	// Get the whole matched element set as a clean array
	get: function( num ) {
		return num == null ?

			// Return a 'clean' array
			this.toArray() :

			// Return just the object
			( num < 0 ? this.slice(num)[ 0 ] : this[ num ] );
	},

	// Take an array of elements and push it onto the stack
	// (returning the new matched element set)
	pushStack: function( elems, name, selector ) {
		// Build a new jQuery matched element set
		var ret = jQuery( elems || null );

		// Add the old object onto the stack (as a reference)
		ret.prevObject = this;

		ret.context = this.context;

		if ( name === "find" ) {
			ret.selector = this.selector + (this.selector ? " " : "") + selector;
		} else if ( name ) {
			ret.selector = this.selector + "." + name + "(" + selector + ")";
		}

		// Return the newly-formed element set
		return ret;
	},

	// Force the current matched set of elements to become
	// the specified array of elements (destroying the stack in the process)
	// You should use pushStack() in order to do this, but maintain the stack
	setArray: function( elems ) {
		// Resetting the length to 0, then using the native Array push
		// is a super-fast way to populate an object with array-like properties
		this.length = 0;
		push.apply( this, elems );

		return this;
	},

	// Execute a callback for every element in the matched set.
	// (You can seed the arguments with an array of args, but this is
	// only used internally.)
	each: function( callback, args ) {
		return jQuery.each( this, callback, args );
	},
	
	ready: function( fn ) {
		// Attach the listeners
		jQuery.bindReady();

		// If the DOM is already ready
		if ( jQuery.isReady ) {
			// Execute the function immediately
			fn.call( document, jQuery );

		// Otherwise, remember the function for later
		} else if ( readyList ) {
			// Add the function to the wait list
			readyList.push( fn );
		}

		return this;
	},
	
	eq: function( i ) {
		return i === -1 ?
			this.slice( i ) :
			this.slice( i, +i + 1 );
	},

	first: function() {
		return this.eq( 0 );
	},

	last: function() {
		return this.eq( -1 );
	},

	slice: function() {
		return this.pushStack( slice.apply( this, arguments ),
			"slice", slice.call(arguments).join(",") );
	},

	map: function( callback ) {
		return this.pushStack( jQuery.map(this, function( elem, i ) {
			return callback.call( elem, i, elem );
		}));
	},
	
	end: function() {
		return this.prevObject || jQuery(null);
	},

	// For internal use only.
	// Behaves like an Array's method, not like a jQuery method.
	push: push,
	sort: [].sort,
	splice: [].splice
};

// Give the init function the jQuery prototype for later instantiation
jQuery.fn.init.prototype = jQuery.fn;

jQuery.extend = jQuery.fn.extend = function() {
	// copy reference to target object
	var target = arguments[0] || {}, i = 1, length = arguments.length, deep = false, options, name, src, copy;

	// Handle a deep copy situation
	if ( typeof target === "boolean" ) {
		deep = target;
		target = arguments[1] || {};
		// skip the boolean and the target
		i = 2;
	}

	// Handle case when target is a string or something (possible in deep copy)
	if ( typeof target !== "object" && !jQuery.isFunction(target) ) {
		target = {};
	}

	// extend jQuery itself if only one argument is passed
	if ( length === i ) {
		target = this;
		--i;
	}

	for ( ; i < length; i++ ) {
		// Only deal with non-null/undefined values
		if ( (options = arguments[ i ]) != null ) {
			// Extend the base object
			for ( name in options ) {
				src = target[ name ];
				copy = options[ name ];

				// Prevent never-ending loop
				if ( target === copy ) {
					continue;
				}

				// Recurse if we're merging object literal values or arrays
				if ( deep && copy && ( jQuery.isPlainObject(copy) || jQuery.isArray(copy) ) ) {
					var clone = src && ( jQuery.isPlainObject(src) || jQuery.isArray(src) ) ? src
						: jQuery.isArray(copy) ? [] : {};

					// Never move original objects, clone them
					target[ name ] = jQuery.extend( deep, clone, copy );

				// Don't bring in undefined values
				} else if ( copy !== undefined ) {
					target[ name ] = copy;
				}
			}
		}
	}

	// Return the modified object
	return target;
};

jQuery.extend({
	noConflict: function( deep ) {
		window.$ = _$;

		if ( deep ) {
			window.jQuery = _jQuery;
		}

		return jQuery;
	},
	
	// Is the DOM ready to be used? Set to true once it occurs.
	isReady: false,
	
	// Handle when the DOM is ready
	ready: function() {
		// Make sure that the DOM is not already loaded
		if ( !jQuery.isReady ) {
			// Make sure body exists, at least, in case IE gets a little overzealous (ticket #5443).
			if ( !document.body ) {
				return setTimeout( jQuery.ready, 13 );
			}

			// Remember that the DOM is ready
			jQuery.isReady = true;

			// If there are functions bound, to execute
			if ( readyList ) {
				// Execute all of them
				var fn, i = 0;
				while ( (fn = readyList[ i++ ]) ) {
					fn.call( document, jQuery );
				}

				// Reset the list of functions
				readyList = null;
			}

			// Trigger any bound ready events
			if ( jQuery.fn.triggerHandler ) {
				jQuery( document ).triggerHandler( "ready" );
			}
		}
	},
	
	bindReady: function() {
		if ( readyBound ) {
			return;
		}

		readyBound = true;

		// Catch cases where $(document).ready() is called after the
		// browser event has already occurred.
		if ( document.readyState === "complete" ) {
			return jQuery.ready();
		}

		// Mozilla, Opera and webkit nightlies currently support this event
		if ( document.addEventListener ) {
			// Use the handy event callback
			document.addEventListener( "DOMContentLoaded", DOMContentLoaded, false );
			
			// A fallback to window.onload, that will always work
			window.addEventListener( "load", jQuery.ready, false );

		// If IE event model is used
		} else if ( document.attachEvent ) {
			// ensure firing before onload,
			// maybe late but safe also for iframes
			document.attachEvent("onreadystatechange", DOMContentLoaded);
			
			// A fallback to window.onload, that will always work
			window.attachEvent( "onload", jQuery.ready );

			// If IE and not a frame
			// continually check to see if the document is ready
			var toplevel = false;

			try {
				toplevel = window.frameElement == null;
			} catch(e) {}

			if ( document.documentElement.doScroll && toplevel ) {
				doScrollCheck();
			}
		}
	},

	// See test/unit/core.js for details concerning isFunction.
	// Since version 1.3, DOM methods and functions like alert
	// aren't supported. They return false on IE (#2968).
	isFunction: function( obj ) {
		return toString.call(obj) === "[object Function]";
	},

	isArray: function( obj ) {
		return toString.call(obj) === "[object Array]";
	},

	isPlainObject: function( obj ) {
		// Must be an Object.
		// Because of IE, we also have to check the presence of the constructor property.
		// Make sure that DOM nodes and window objects don't pass through, as well
		if ( !obj || toString.call(obj) !== "[object Object]" || obj.nodeType || obj.setInterval ) {
			return false;
		}
		
		// Not own constructor property must be Object
		if ( obj.constructor
			&& !hasOwnProperty.call(obj, "constructor")
			&& !hasOwnProperty.call(obj.constructor.prototype, "isPrototypeOf") ) {
			return false;
		}
		
		// Own properties are enumerated firstly, so to speed up,
		// if last one is own, then all properties are own.
	
		var key;
		for ( key in obj ) {}
		
		return key === undefined || hasOwnProperty.call( obj, key );
	},

	isEmptyObject: function( obj ) {
		for ( var name in obj ) {
			return false;
		}
		return true;
	},
	
	error: function( msg ) {
		throw msg;
	},
	
	parseJSON: function( data ) {
		if ( typeof data !== "string" || !data ) {
			return null;
		}
		
		// Make sure the incoming data is actual JSON
		// Logic borrowed from http://json.org/json2.js
		if ( /^[\],:{}\s]*$/.test(data.replace(/\\(?:["\\\/bfnrt]|u[0-9a-fA-F]{4})/g, "@")
			.replace(/"[^"\\\n\r]*"|true|false|null|-?\d+(?:\.\d*)?(?:[eE][+\-]?\d+)?/g, "]")
			.replace(/(?:^|:|,)(?:\s*\[)+/g, "")) ) {

			// Try to use the native JSON parser first
			return window.JSON && window.JSON.parse ?
				window.JSON.parse( data ) :
				(new Function("return " + data))();

		} else {
			jQuery.error( "Invalid JSON: " + data );
		}
	},

	noop: function() {},

	// Evalulates a script in a global context
	globalEval: function( data ) {
		if ( data && rnotwhite.test(data) ) {
			// Inspired by code by Andrea Giammarchi
			// http://webreflection.blogspot.com/2007/08/global-scope-evaluation-and-dom.html
			var head = document.getElementsByTagName("head")[0] || document.documentElement,
				script = document.createElement("script");

			script.type = "text/javascript";

			if ( jQuery.support.scriptEval ) {
				script.appendChild( document.createTextNode( data ) );
			} else {
				script.text = data;
			}

			// Use insertBefore instead of appendChild to circumvent an IE6 bug.
			// This arises when a base node is used (#2709).
			head.insertBefore( script, head.firstChild );
			head.removeChild( script );
		}
	},

	nodeName: function( elem, name ) {
		return elem.nodeName && elem.nodeName.toUpperCase() === name.toUpperCase();
	},

	// args is for internal usage only
	each: function( object, callback, args ) {
		var name, i = 0,
			length = object.length,
			isObj = length === undefined || jQuery.isFunction(object);

		if ( args ) {
			if ( isObj ) {
				for ( name in object ) {
					if ( callback.apply( object[ name ], args ) === false ) {
						break;
					}
				}
			} else {
				for ( ; i < length; ) {
					if ( callback.apply( object[ i++ ], args ) === false ) {
						break;
					}
				}
			}

		// A special, fast, case for the most common use of each
		} else {
			if ( isObj ) {
				for ( name in object ) {
					if ( callback.call( object[ name ], name, object[ name ] ) === false ) {
						break;
					}
				}
			} else {
				for ( var value = object[0];
					i < length && callback.call( value, i, value ) !== false; value = object[++i] ) {}
			}
		}

		return object;
	},

	trim: function( text ) {
		return (text || "").replace( rtrim, "" );
	},

	// results is for internal usage only
	makeArray: function( array, results ) {
		var ret = results || [];

		if ( array != null ) {
			// The window, strings (and functions) also have 'length'
			// The extra typeof function check is to prevent crashes
			// in Safari 2 (See: #3039)
			if ( array.length == null || typeof array === "string" || jQuery.isFunction(array) || (typeof array !== "function" && array.setInterval) ) {
				push.call( ret, array );
			} else {
				jQuery.merge( ret, array );
			}
		}

		return ret;
	},

	inArray: function( elem, array ) {
		if ( array.indexOf ) {
			return array.indexOf( elem );
		}

		for ( var i = 0, length = array.length; i < length; i++ ) {
			if ( array[ i ] === elem ) {
				return i;
			}
		}

		return -1;
	},

	merge: function( first, second ) {
		var i = first.length, j = 0;

		if ( typeof second.length === "number" ) {
			for ( var l = second.length; j < l; j++ ) {
				first[ i++ ] = second[ j ];
			}
		} else {
			while ( second[j] !== undefined ) {
				first[ i++ ] = second[ j++ ];
			}
		}

		first.length = i;

		return first;
	},

	grep: function( elems, callback, inv ) {
		var ret = [];

		// Go through the array, only saving the items
		// that pass the validator function
		for ( var i = 0, length = elems.length; i < length; i++ ) {
			if ( !inv !== !callback( elems[ i ], i ) ) {
				ret.push( elems[ i ] );
			}
		}

		return ret;
	},

	// arg is for internal usage only
	map: function( elems, callback, arg ) {
		var ret = [], value;

		// Go through the array, translating each of the items to their
		// new value (or values).
		for ( var i = 0, length = elems.length; i < length; i++ ) {
			value = callback( elems[ i ], i, arg );

			if ( value != null ) {
				ret[ ret.length ] = value;
			}
		}

		return ret.concat.apply( [], ret );
	},

	// A global GUID counter for objects
	guid: 1,

	proxy: function( fn, proxy, thisObject ) {
		if ( arguments.length === 2 ) {
			if ( typeof proxy === "string" ) {
				thisObject = fn;
				fn = thisObject[ proxy ];
				proxy = undefined;

			} else if ( proxy && !jQuery.isFunction( proxy ) ) {
				thisObject = proxy;
				proxy = undefined;
			}
		}

		if ( !proxy && fn ) {
			proxy = function() {
				return fn.apply( thisObject || this, arguments );
			};
		}

		// Set the guid of unique handler to the same of original handler, so it can be removed
		if ( fn ) {
			proxy.guid = fn.guid = fn.guid || proxy.guid || jQuery.guid++;
		}

		// So proxy can be declared as an argument
		return proxy;
	},

	// Use of jQuery.browser is frowned upon.
	// More details: http://docs.jquery.com/Utilities/jQuery.browser
	uaMatch: function( ua ) {
		ua = ua.toLowerCase();

		var match = /(webkit)[ \/]([\w.]+)/.exec( ua ) ||
			/(opera)(?:.*version)?[ \/]([\w.]+)/.exec( ua ) ||
			/(msie) ([\w.]+)/.exec( ua ) ||
			!/compatible/.test( ua ) && /(mozilla)(?:.*? rv:([\w.]+))?/.exec( ua ) ||
		  	[];

		return { browser: match[1] || "", version: match[2] || "0" };
	},

	browser: {}
});

browserMatch = jQuery.uaMatch( userAgent );
if ( browserMatch.browser ) {
	jQuery.browser[ browserMatch.browser ] = true;
	jQuery.browser.version = browserMatch.version;
}

// Deprecated, use jQuery.browser.webkit instead
if ( jQuery.browser.webkit ) {
	jQuery.browser.safari = true;
}

if ( indexOf ) {
	jQuery.inArray = function( elem, array ) {
		return indexOf.call( array, elem );
	};
}

// All jQuery objects should point back to these
rootjQuery = jQuery(document);

// Cleanup functions for the document ready method
if ( document.addEventListener ) {
	DOMContentLoaded = function() {
		document.removeEventListener( "DOMContentLoaded", DOMContentLoaded, false );
		jQuery.ready();
	};

} else if ( document.attachEvent ) {
	DOMContentLoaded = function() {
		// Make sure body exists, at least, in case IE gets a little overzealous (ticket #5443).
		if ( document.readyState === "complete" ) {
			document.detachEvent( "onreadystatechange", DOMContentLoaded );
			jQuery.ready();
		}
	};
}

// The DOM ready check for Internet Explorer
function doScrollCheck() {
	if ( jQuery.isReady ) {
		return;
	}

	try {
		// If IE is used, use the trick by Diego Perini
		// http://javascript.nwbox.com/IEContentLoaded/
		document.documentElement.doScroll("left");
	} catch( error ) {
		setTimeout( doScrollCheck, 1 );
		return;
	}

	// and execute any waiting functions
	jQuery.ready();
}

function evalScript( i, elem ) {
	if ( elem.src ) {
		jQuery.ajax({
			url: elem.src,
			async: false,
			dataType: "script"
		});
	} else {
		jQuery.globalEval( elem.text || elem.textContent || elem.innerHTML || "" );
	}

	if ( elem.parentNode ) {
		elem.parentNode.removeChild( elem );
	}
}

// Mutifunctional method to get and set values to a collection
// The value/s can be optionally by executed if its a function
function access( elems, key, value, exec, fn, pass ) {
	var length = elems.length;
	
	// Setting many attributes
	if ( typeof key === "object" ) {
		for ( var k in key ) {
			access( elems, k, key[k], exec, fn, value );
		}
		return elems;
	}
	
	// Setting one attribute
	if ( value !== undefined ) {
		// Optionally, function values get executed if exec is true
		exec = !pass && exec && jQuery.isFunction(value);
		
		for ( var i = 0; i < length; i++ ) {
			fn( elems[i], key, exec ? value.call( elems[i], i, fn( elems[i], key ) ) : value, pass );
		}
		
		return elems;
	}
	
	// Getting an attribute
	return length ? fn( elems[0], key ) : null;
}

function now() {
	return (new Date).getTime();
}
(function() {

	jQuery.support = {};

	var root = document.documentElement,
		script = document.createElement("script"),
		div = document.createElement("div"),
		id = "script" + now();

	div.style.display = "none";
	div.innerHTML = "   <link/><table></table><a href='/a' style='color:red;float:left;opacity:.55;'>a</a><input type='checkbox'/>";

	var all = div.getElementsByTagName("*"),
		a = div.getElementsByTagName("a")[0];

	// Can't get basic test support
	if ( !all || !all.length || !a ) {
		return;
	}

	jQuery.support = {
		// IE strips leading whitespace when .innerHTML is used
		leadingWhitespace: div.firstChild.nodeType === 3,

		// Make sure that tbody elements aren't automatically inserted
		// IE will insert them into empty tables
		tbody: !div.getElementsByTagName("tbody").length,

		// Make sure that link elements get serialized correctly by innerHTML
		// This requires a wrapper element in IE
		htmlSerialize: !!div.getElementsByTagName("link").length,

		// Get the style information from getAttribute
		// (IE uses .cssText insted)
		style: /red/.test( a.getAttribute("style") ),

		// Make sure that URLs aren't manipulated
		// (IE normalizes it by default)
		hrefNormalized: a.getAttribute("href") === "/a",

		// Make sure that element opacity exists
		// (IE uses filter instead)
		// Use a regex to work around a WebKit issue. See #5145
		opacity: /^0.55$/.test( a.style.opacity ),

		// Verify style float existence
		// (IE uses styleFloat instead of cssFloat)
		cssFloat: !!a.style.cssFloat,

		// Make sure that if no value is specified for a checkbox
		// that it defaults to "on".
		// (WebKit defaults to "" instead)
		checkOn: div.getElementsByTagName("input")[0].value === "on",

		// Make sure that a selected-by-default option has a working selected property.
		// (WebKit defaults to false instead of true, IE too, if it's in an optgroup)
		optSelected: document.createElement("select").appendChild( document.createElement("option") ).selected,

		// Will be defined later
		checkClone: false,
		scriptEval: false,
		noCloneEvent: true,
		boxModel: null
	};

	script.type = "text/javascript";
	try {
		script.appendChild( document.createTextNode( "window." + id + "=1;" ) );
	} catch(e) {}

	root.insertBefore( script, root.firstChild );

	// Make sure that the execution of code works by injecting a script
	// tag with appendChild/createTextNode
	// (IE doesn't support this, fails, and uses .text instead)
	if ( window[ id ] ) {
		jQuery.support.scriptEval = true;
		delete window[ id ];
	}

	root.removeChild( script );

	if ( div.attachEvent && div.fireEvent ) {
		div.attachEvent("onclick", function click() {
			// Cloning a node shouldn't copy over any
			// bound event handlers (IE does this)
			jQuery.support.noCloneEvent = false;
			div.detachEvent("onclick", click);
		});
		div.cloneNode(true).fireEvent("onclick");
	}

	div = document.createElement("div");
	div.innerHTML = "<input type='radio' name='radiotest' checked='checked'/>";

	var fragment = document.createDocumentFragment();
	fragment.appendChild( div.firstChild );

	// WebKit doesn't clone checked state correctly in fragments
	jQuery.support.checkClone = fragment.cloneNode(true).cloneNode(true).lastChild.checked;

	// Figure out if the W3C box model works as expected
	// document.body must exist before we can do this
	jQuery(function() {
		var div = document.createElement("div");
		div.style.width = div.style.paddingLeft = "1px";

		document.body.appendChild( div );
		jQuery.boxModel = jQuery.support.boxModel = div.offsetWidth === 2;
		document.body.removeChild( div ).style.display = 'none';
		div = null;
	});

	// Technique from Juriy Zaytsev
	// http://thinkweb2.com/projects/prototype/detecting-event-support-without-browser-sniffing/
	var eventSupported = function( eventName ) { 
		var el = document.createElement("div"); 
		eventName = "on" + eventName; 

		var isSupported = (eventName in el); 
		if ( !isSupported ) { 
			el.setAttribute(eventName, "return;"); 
			isSupported = typeof el[eventName] === "function"; 
		} 
		el = null; 

		return isSupported; 
	};
	
	jQuery.support.submitBubbles = eventSupported("submit");
	jQuery.support.changeBubbles = eventSupported("change");

	// release memory in IE
	root = script = div = all = a = null;
})();

jQuery.props = {
	"for": "htmlFor",
	"class": "className",
	readonly: "readOnly",
	maxlength: "maxLength",
	cellspacing: "cellSpacing",
	rowspan: "rowSpan",
	colspan: "colSpan",
	tabindex: "tabIndex",
	usemap: "useMap",
	frameborder: "frameBorder"
};
var expando = "jQuery" + now(), uuid = 0, windowData = {};
var emptyObject = {};

jQuery.extend({
	cache: {},
	
	expando:expando,

	// The following elements throw uncatchable exceptions if you
	// attempt to add expando properties to them.
	noData: {
		"embed": true,
		"object": true,
		"applet": true
	},

	data: function( elem, name, data ) {
		if ( elem.nodeName && jQuery.noData[elem.nodeName.toLowerCase()] ) {
			return;
		}

		elem = elem == window ?
			windowData :
			elem;

		var id = elem[ expando ], cache = jQuery.cache, thisCache;

		// Handle the case where there's no name immediately
		if ( !name && !id ) {
			return null;
		}

		// Compute a unique ID for the element
		if ( !id ) { 
			id = ++uuid;
		}

		// Avoid generating a new cache unless none exists and we
		// want to manipulate it.
		if ( typeof name === "object" ) {
			elem[ expando ] = id;
			thisCache = cache[ id ] = jQuery.extend(true, {}, name);
		} else if ( cache[ id ] ) {
			thisCache = cache[ id ];
		} else if ( typeof data === "undefined" ) {
			thisCache = emptyObject;
		} else {
			thisCache = cache[ id ] = {};
		}

		// Prevent overriding the named cache with undefined values
		if ( data !== undefined ) {
			elem[ expando ] = id;
			thisCache[ name ] = data;
		}

		return typeof name === "string" ? thisCache[ name ] : thisCache;
	},

	removeData: function( elem, name ) {
		if ( elem.nodeName && jQuery.noData[elem.nodeName.toLowerCase()] ) {
			return;
		}

		elem = elem == window ?
			windowData :
			elem;

		var id = elem[ expando ], cache = jQuery.cache, thisCache = cache[ id ];

		// If we want to remove a specific section of the element's data
		if ( name ) {
			if ( thisCache ) {
				// Remove the section of cache data
				delete thisCache[ name ];

				// If we've removed all the data, remove the element's cache
				if ( jQuery.isEmptyObject(thisCache) ) {
					jQuery.removeData( elem );
				}
			}

		// Otherwise, we want to remove all of the element's data
		} else {
			// Clean up the element expando
			try {
				delete elem[ expando ];
			} catch( e ) {
				// IE has trouble directly removing the expando
				// but it's ok with using removeAttribute
				if ( elem.removeAttribute ) {
					elem.removeAttribute( expando );
				}
			}

			// Completely remove the data cache
			delete cache[ id ];
		}
	}
});

jQuery.fn.extend({
	data: function( key, value ) {
		if ( typeof key === "undefined" && this.length ) {
			return jQuery.data( this[0] );

		} else if ( typeof key === "object" ) {
			return this.each(function() {
				jQuery.data( this, key );
			});
		}

		var parts = key.split(".");
		parts[1] = parts[1] ? "." + parts[1] : "";

		if ( value === undefined ) {
			var data = this.triggerHandler("getData" + parts[1] + "!", [parts[0]]);

			if ( data === undefined && this.length ) {
				data = jQuery.data( this[0], key );
			}
			return data === undefined && parts[1] ?
				this.data( parts[0] ) :
				data;
		} else {
			return this.trigger("setData" + parts[1] + "!", [parts[0], value]).each(function() {
				jQuery.data( this, key, value );
			});
		}
	},

	removeData: function( key ) {
		return this.each(function() {
			jQuery.removeData( this, key );
		});
	}
});
jQuery.extend({
	queue: function( elem, type, data ) {
		if ( !elem ) {
			return;
		}

		type = (type || "fx") + "queue";
		var q = jQuery.data( elem, type );

		// Speed up dequeue by getting out quickly if this is just a lookup
		if ( !data ) {
			return q || [];
		}

		if ( !q || jQuery.isArray(data) ) {
			q = jQuery.data( elem, type, jQuery.makeArray(data) );

		} else {
			q.push( data );
		}

		return q;
	},

	dequeue: function( elem, type ) {
		type = type || "fx";

		var queue = jQuery.queue( elem, type ), fn = queue.shift();

		// If the fx queue is dequeued, always remove the progress sentinel
		if ( fn === "inprogress" ) {
			fn = queue.shift();
		}

		if ( fn ) {
			// Add a progress sentinel to prevent the fx queue from being
			// automatically dequeued
			if ( type === "fx" ) {
				queue.unshift("inprogress");
			}

			fn.call(elem, function() {
				jQuery.dequeue(elem, type);
			});
		}
	}
});

jQuery.fn.extend({
	queue: function( type, data ) {
		if ( typeof type !== "string" ) {
			data = type;
			type = "fx";
		}

		if ( data === undefined ) {
			return jQuery.queue( this[0], type );
		}
		return this.each(function( i, elem ) {
			var queue = jQuery.queue( this, type, data );

			if ( type === "fx" && queue[0] !== "inprogress" ) {
				jQuery.dequeue( this, type );
			}
		});
	},
	dequeue: function( type ) {
		return this.each(function() {
			jQuery.dequeue( this, type );
		});
	},

	// Based off of the plugin by Clint Helfers, with permission.
	// http://blindsignals.com/index.php/2009/07/jquery-delay/
	delay: function( time, type ) {
		time = jQuery.fx ? jQuery.fx.speeds[time] || time : time;
		type = type || "fx";

		return this.queue( type, function() {
			var elem = this;
			setTimeout(function() {
				jQuery.dequeue( elem, type );
			}, time );
		});
	},

	clearQueue: function( type ) {
		return this.queue( type || "fx", [] );
	}
});
var rclass = /[\n\t]/g,
	rspace = /\s+/,
	rreturn = /\r/g,
	rspecialurl = /href|src|style/,
	rtype = /(button|input)/i,
	rfocusable = /(button|input|object|select|textarea)/i,
	rclickable = /^(a|area)$/i,
	rradiocheck = /radio|checkbox/;

jQuery.fn.extend({
	attr: function( name, value ) {
		return access( this, name, value, true, jQuery.attr );
	},

	removeAttr: function( name, fn ) {
		return this.each(function(){
			jQuery.attr( this, name, "" );
			if ( this.nodeType === 1 ) {
				this.removeAttribute( name );
			}
		});
	},

	addClass: function( value ) {
		if ( jQuery.isFunction(value) ) {
			return this.each(function(i) {
				var self = jQuery(this);
				self.addClass( value.call(this, i, self.attr("class")) );
			});
		}

		if ( value && typeof value === "string" ) {
			var classNames = (value || "").split( rspace );

			for ( var i = 0, l = this.length; i < l; i++ ) {
				var elem = this[i];

				if ( elem.nodeType === 1 ) {
					if ( !elem.className ) {
						elem.className = value;

					} else {
						var className = " " + elem.className + " ";
						for ( var c = 0, cl = classNames.length; c < cl; c++ ) {
							if ( className.indexOf( " " + classNames[c] + " " ) < 0 ) {
								elem.className += " " + classNames[c];
							}
						}
					}
				}
			}
		}

		return this;
	},

	removeClass: function( value ) {
		if ( jQuery.isFunction(value) ) {
			return this.each(function(i) {
				var self = jQuery(this);
				self.removeClass( value.call(this, i, self.attr("class")) );
			});
		}

		if ( (value && typeof value === "string") || value === undefined ) {
			var classNames = (value || "").split(rspace);

			for ( var i = 0, l = this.length; i < l; i++ ) {
				var elem = this[i];

				if ( elem.nodeType === 1 && elem.className ) {
					if ( value ) {
						var className = (" " + elem.className + " ").replace(rclass, " ");
						for ( var c = 0, cl = classNames.length; c < cl; c++ ) {
							className = className.replace(" " + classNames[c] + " ", " ");
						}
						elem.className = className.substring(1, className.length - 1);

					} else {
						elem.className = "";
					}
				}
			}
		}

		return this;
	},

	toggleClass: function( value, stateVal ) {
		var type = typeof value, isBool = typeof stateVal === "boolean";

		if ( jQuery.isFunction( value ) ) {
			return this.each(function(i) {
				var self = jQuery(this);
				self.toggleClass( value.call(this, i, self.attr("class"), stateVal), stateVal );
			});
		}

		return this.each(function() {
			if ( type === "string" ) {
				// toggle individual class names
				var className, i = 0, self = jQuery(this),
					state = stateVal,
					classNames = value.split( rspace );

				while ( (className = classNames[ i++ ]) ) {
					// check each className given, space seperated list
					state = isBool ? state : !self.hasClass( className );
					self[ state ? "addClass" : "removeClass" ]( className );
				}

			} else if ( type === "undefined" || type === "boolean" ) {
				if ( this.className ) {
					// store className if set
					jQuery.data( this, "__className__", this.className );
				}

				// toggle whole className
				this.className = this.className || value === false ? "" : jQuery.data( this, "__className__" ) || "";
			}
		});
	},

	hasClass: function( selector ) {
		var className = " " + selector + " ";
		for ( var i = 0, l = this.length; i < l; i++ ) {
			if ( (" " + this[i].className + " ").replace(rclass, " ").indexOf( className ) > -1 ) {
				return true;
			}
		}

		return false;
	},

	val: function( value ) {
		if ( value === undefined ) {
			var elem = this[0];

			if ( elem ) {
				if ( jQuery.nodeName( elem, "option" ) ) {
					return (elem.attributes.value || {}).specified ? elem.value : elem.text;
				}

				// We need to handle select boxes special
				if ( jQuery.nodeName( elem, "select" ) ) {
					var index = elem.selectedIndex,
						values = [],
						options = elem.options,
						one = elem.type === "select-one";

					// Nothing was selected
					if ( index < 0 ) {
						return null;
					}

					// Loop through all the selected options
					for ( var i = one ? index : 0, max = one ? index + 1 : options.length; i < max; i++ ) {
						var option = options[ i ];

						if ( option.selected ) {
							// Get the specifc value for the option
							value = jQuery(option).val();

							// We don't need an array for one selects
							if ( one ) {
								return value;
							}

							// Multi-Selects return an array
							values.push( value );
						}
					}

					return values;
				}

				// Handle the case where in Webkit "" is returned instead of "on" if a value isn't specified
				if ( rradiocheck.test( elem.type ) && !jQuery.support.checkOn ) {
					return elem.getAttribute("value") === null ? "on" : elem.value;
				}
				

				// Everything else, we just grab the value
				return (elem.value || "").replace(rreturn, "");

			}

			return undefined;
		}

		var isFunction = jQuery.isFunction(value);

		return this.each(function(i) {
			var self = jQuery(this), val = value;

			if ( this.nodeType !== 1 ) {
				return;
			}

			if ( isFunction ) {
				val = value.call(this, i, self.val());
			}

			// Typecast each time if the value is a Function and the appended
			// value is therefore different each time.
			if ( typeof val === "number" ) {
				val += "";
			}

			if ( jQuery.isArray(val) && rradiocheck.test( this.type ) ) {
				this.checked = jQuery.inArray( self.val(), val ) >= 0;

			} else if ( jQuery.nodeName( this, "select" ) ) {
				var values = jQuery.makeArray(val);

				jQuery( "option", this ).each(function() {
					this.selected = jQuery.inArray( jQuery(this).val(), values ) >= 0;
				});

				if ( !values.length ) {
					this.selectedIndex = -1;
				}

			} else {
				this.value = val;
			}
		});
	}
});

jQuery.extend({
	attrFn: {
		val: true,
		css: true,
		html: true,
		text: true,
		data: true,
		width: true,
		height: true,
		offset: true
	},
		
	attr: function( elem, name, value, pass ) {
		// don't set attributes on text and comment nodes
		if ( !elem || elem.nodeType === 3 || elem.nodeType === 8 ) {
			return undefined;
		}

		if ( pass && name in jQuery.attrFn ) {
			return jQuery(elem)[name](value);
		}

		var notxml = elem.nodeType !== 1 || !jQuery.isXMLDoc( elem ),
			// Whether we are setting (or getting)
			set = value !== undefined;

		// Try to normalize/fix the name
		name = notxml && jQuery.props[ name ] || name;

		// Only do all the following if this is a node (faster for style)
		if ( elem.nodeType === 1 ) {
			// These attributes require special treatment
			var special = rspecialurl.test( name );

			// Safari mis-reports the default selected property of an option
			// Accessing the parent's selectedIndex property fixes it
			if ( name === "selected" && !jQuery.support.optSelected ) {
				var parent = elem.parentNode;
				if ( parent ) {
					parent.selectedIndex;
	
					// Make sure that it also works with optgroups, see #5701
					if ( parent.parentNode ) {
						parent.parentNode.selectedIndex;
					}
				}
			}

			// If applicable, access the attribute via the DOM 0 way
			if ( name in elem && notxml && !special ) {
				if ( set ) {
					// We can't allow the type property to be changed (since it causes problems in IE)
					if ( name === "type" && rtype.test( elem.nodeName ) && elem.parentNode ) {
						jQuery.error( "type property can't be changed" );
					}

					elem[ name ] = value;
				}

				// browsers index elements by id/name on forms, give priority to attributes.
				if ( jQuery.nodeName( elem, "form" ) && elem.getAttributeNode(name) ) {
					return elem.getAttributeNode( name ).nodeValue;
				}

				// elem.tabIndex doesn't always return the correct value when it hasn't been explicitly set
				// http://fluidproject.org/blog/2008/01/09/getting-setting-and-removing-tabindex-values-with-javascript/
				if ( name === "tabIndex" ) {
					var attributeNode = elem.getAttributeNode( "tabIndex" );

					return attributeNode && attributeNode.specified ?
						attributeNode.value :
						rfocusable.test( elem.nodeName ) || rclickable.test( elem.nodeName ) && elem.href ?
							0 :
							undefined;
				}

				return elem[ name ];
			}

			if ( !jQuery.support.style && notxml && name === "style" ) {
				if ( set ) {
					elem.style.cssText = "" + value;
				}

				return elem.style.cssText;
			}

			if ( set ) {
				// convert the value to a string (all browsers do this but IE) see #1070
				elem.setAttribute( name, "" + value );
			}

			var attr = !jQuery.support.hrefNormalized && notxml && special ?
					// Some attributes require a special call on IE
					elem.getAttribute( name, 2 ) :
					elem.getAttribute( name );

			// Non-existent attributes return null, we normalize to undefined
			return attr === null ? undefined : attr;
		}

		// elem is actually elem.style ... set the style
		// Using attr for specific style information is now deprecated. Use style insead.
		return jQuery.style( elem, name, value );
	}
});
var fcleanup = function( nm ) {
	return nm.replace(/[^\w\s\.\|`]/g, function( ch ) {
		return "\\" + ch;
	});
};

/*
 * A number of helper functions used for managing events.
 * Many of the ideas behind this code originated from
 * Dean Edwards' addEvent library.
 */
jQuery.event = {

	// Bind an event to an element
	// Original by Dean Edwards
	add: function( elem, types, handler, data ) {
		if ( elem.nodeType === 3 || elem.nodeType === 8 ) {
			return;
		}

		// For whatever reason, IE has trouble passing the window object
		// around, causing it to be cloned in the process
		if ( elem.setInterval && ( elem !== window && !elem.frameElement ) ) {
			elem = window;
		}

		// Make sure that the function being executed has a unique ID
		if ( !handler.guid ) {
			handler.guid = jQuery.guid++;
		}

		// if data is passed, bind to handler
		if ( data !== undefined ) {
			// Create temporary function pointer to original handler
			var fn = handler;

			// Create unique handler function, wrapped around original handler
			handler = jQuery.proxy( fn );

			// Store data in unique handler
			handler.data = data;
		}

		// Init the element's event structure
		var events = jQuery.data( elem, "events" ) || jQuery.data( elem, "events", {} ),
			handle = jQuery.data( elem, "handle" ), eventHandle;

		if ( !handle ) {
			eventHandle = function() {
				// Handle the second event of a trigger and when
				// an event is called after a page has unloaded
				return typeof jQuery !== "undefined" && !jQuery.event.triggered ?
					jQuery.event.handle.apply( eventHandle.elem, arguments ) :
					undefined;
			};

			handle = jQuery.data( elem, "handle", eventHandle );
		}

		// If no handle is found then we must be trying to bind to one of the
		// banned noData elements
		if ( !handle ) {
			return;
		}

		// Add elem as a property of the handle function
		// This is to prevent a memory leak with non-native
		// event in IE.
		handle.elem = elem;

		// Handle multiple events separated by a space
		// jQuery(...).bind("mouseover mouseout", fn);
		types = types.split( /\s+/ );

		var type, i = 0;

		while ( (type = types[ i++ ]) ) {
			// Namespaced event handlers
			var namespaces = type.split(".");
			type = namespaces.shift();

			if ( i > 1 ) {
				handler = jQuery.proxy( handler );

				if ( data !== undefined ) {
					handler.data = data;
				}
			}

			handler.type = namespaces.slice(0).sort().join(".");

			// Get the current list of functions bound to this event
			var handlers = events[ type ],
				special = this.special[ type ] || {};

			// Init the event handler queue
			if ( !handlers ) {
				handlers = events[ type ] = {};

				// Check for a special event handler
				// Only use addEventListener/attachEvent if the special
				// events handler returns false
				if ( !special.setup || special.setup.call( elem, data, namespaces, handler) === false ) {
					// Bind the global event handler to the element
					if ( elem.addEventListener ) {
						elem.addEventListener( type, handle, false );
					} else if ( elem.attachEvent ) {
						elem.attachEvent( "on" + type, handle );
					}
				}
			}
			
			if ( special.add ) { 
				var modifiedHandler = special.add.call( elem, handler, data, namespaces, handlers ); 
				if ( modifiedHandler && jQuery.isFunction( modifiedHandler ) ) { 
					modifiedHandler.guid = modifiedHandler.guid || handler.guid; 
					modifiedHandler.data = modifiedHandler.data || handler.data; 
					modifiedHandler.type = modifiedHandler.type || handler.type; 
					handler = modifiedHandler; 
				} 
			} 
			
			// Add the function to the element's handler list
			handlers[ handler.guid ] = handler;

			// Keep track of which events have been used, for global triggering
			this.global[ type ] = true;
		}

		// Nullify elem to prevent memory leaks in IE
		elem = null;
	},

	global: {},

	// Detach an event or set of events from an element
	remove: function( elem, types, handler ) {
		// don't do events on text and comment nodes
		if ( elem.nodeType === 3 || elem.nodeType === 8 ) {
			return;
		}

		var events = jQuery.data( elem, "events" ), ret, type, fn;

		if ( events ) {
			// Unbind all events for the element
			if ( types === undefined || (typeof types === "string" && types.charAt(0) === ".") ) {
				for ( type in events ) {
					this.remove( elem, type + (types || "") );
				}
			} else {
				// types is actually an event object here
				if ( types.type ) {
					handler = types.handler;
					types = types.type;
				}

				// Handle multiple events separated by a space
				// jQuery(...).unbind("mouseover mouseout", fn);
				types = types.split(/\s+/);
				var i = 0;
				while ( (type = types[ i++ ]) ) {
					// Namespaced event handlers
					var namespaces = type.split(".");
					type = namespaces.shift();
					var all = !namespaces.length,
						cleaned = jQuery.map( namespaces.slice(0).sort(), fcleanup ),
						namespace = new RegExp("(^|\\.)" + cleaned.join("\\.(?:.*\\.)?") + "(\\.|$)"),
						special = this.special[ type ] || {};

					if ( events[ type ] ) {
						// remove the given handler for the given type
						if ( handler ) {
							fn = events[ type ][ handler.guid ];
							delete events[ type ][ handler.guid ];

						// remove all handlers for the given type
						} else {
							for ( var handle in events[ type ] ) {
								// Handle the removal of namespaced events
								if ( all || namespace.test( events[ type ][ handle ].type ) ) {
									delete events[ type ][ handle ];
								}
							}
						}

						if ( special.remove ) {
							special.remove.call( elem, namespaces, fn);
						}

						// remove generic event handler if no more handlers exist
						for ( ret in events[ type ] ) {
							break;
						}
						if ( !ret ) {
							if ( !special.teardown || special.teardown.call( elem, namespaces ) === false ) {
								if ( elem.removeEventListener ) {
									elem.removeEventListener( type, jQuery.data( elem, "handle" ), false );
								} else if ( elem.detachEvent ) {
									elem.detachEvent( "on" + type, jQuery.data( elem, "handle" ) );
								}
							}
							ret = null;
							delete events[ type ];
						}
					}
				}
			}

			// Remove the expando if it's no longer used
			for ( ret in events ) {
				break;
			}
			if ( !ret ) {
				var handle = jQuery.data( elem, "handle" );
				if ( handle ) {
					handle.elem = null;
				}
				jQuery.removeData( elem, "events" );
				jQuery.removeData( elem, "handle" );
			}
		}
	},

	// bubbling is internal
	trigger: function( event, data, elem /*, bubbling */ ) {
		// Event object or event type
		var type = event.type || event,
			bubbling = arguments[3];

		if ( !bubbling ) {
			event = typeof event === "object" ?
				// jQuery.Event object
				event[expando] ? event :
				// Object literal
				jQuery.extend( jQuery.Event(type), event ) :
				// Just the event type (string)
				jQuery.Event(type);

			if ( type.indexOf("!") >= 0 ) {
				event.type = type = type.slice(0, -1);
				event.exclusive = true;
			}

			// Handle a global trigger
			if ( !elem ) {
				// Don't bubble custom events when global (to avoid too much overhead)
				event.stopPropagation();

				// Only trigger if we've ever bound an event for it
				if ( this.global[ type ] ) {
					jQuery.each( jQuery.cache, function() {
						if ( this.events && this.events[type] ) {
							jQuery.event.trigger( event, data, this.handle.elem );
						}
					});
				}
			}

			// Handle triggering a single element

			// don't do events on text and comment nodes
			if ( !elem || elem.nodeType === 3 || elem.nodeType === 8 ) {
				return undefined;
			}

			// Clean up in case it is reused
			event.result = undefined;
			event.target = elem;

			// Clone the incoming data, if any
			data = jQuery.makeArray( data );
			data.unshift( event );
		}

		event.currentTarget = elem;

		// Trigger the event, it is assumed that "handle" is a function
		var handle = jQuery.data( elem, "handle" );
		if ( handle ) {
			handle.apply( elem, data );
		}

		var parent = elem.parentNode || elem.ownerDocument;

		// Trigger an inline bound script
		try {
			if ( !(elem && elem.nodeName && jQuery.noData[elem.nodeName.toLowerCase()]) ) {
				if ( elem[ "on" + type ] && elem[ "on" + type ].apply( elem, data ) === false ) {
					event.result = false;
				}
			}

		// prevent IE from throwing an error for some elements with some event types, see #3533
		} catch (e) {}

		if ( !event.isPropagationStopped() && parent ) {
			jQuery.event.trigger( event, data, parent, true );

		} else if ( !event.isDefaultPrevented() ) {
			var target = event.target, old,
				isClick = jQuery.nodeName(target, "a") && type === "click";

			if ( !isClick && !(target && target.nodeName && jQuery.noData[target.nodeName.toLowerCase()]) ) {
				try {
					if ( target[ type ] ) {
						// Make sure that we don't accidentally re-trigger the onFOO events
						old = target[ "on" + type ];

						if ( old ) {
							target[ "on" + type ] = null;
						}

						this.triggered = true;
						target[ type ]();
					}

				// prevent IE from throwing an error for some elements with some event types, see #3533
				} catch (e) {}

				if ( old ) {
					target[ "on" + type ] = old;
				}

				this.triggered = false;
			}
		}
	},

	handle: function( event ) {
		// returned undefined or false
		var all, handlers;

		event = arguments[0] = jQuery.event.fix( event || window.event );
		event.currentTarget = this;

		// Namespaced event handlers
		var namespaces = event.type.split(".");
		event.type = namespaces.shift();

		// Cache this now, all = true means, any handler
		all = !namespaces.length && !event.exclusive;

		var namespace = new RegExp("(^|\\.)" + namespaces.slice(0).sort().join("\\.(?:.*\\.)?") + "(\\.|$)");

		handlers = ( jQuery.data(this, "events") || {} )[ event.type ];

		for ( var j in handlers ) {
			var handler = handlers[ j ];

			// Filter the functions by class
			if ( all || namespace.test(handler.type) ) {
				// Pass in a reference to the handler function itself
				// So that we can later remove it
				event.handler = handler;
				event.data = handler.data;

				var ret = handler.apply( this, arguments );

				if ( ret !== undefined ) {
					event.result = ret;
					if ( ret === false ) {
						event.preventDefault();
						event.stopPropagation();
					}
				}

				if ( event.isImmediatePropagationStopped() ) {
					break;
				}

			}
		}

		return event.result;
	},

	props: "altKey attrChange attrName bubbles button cancelable charCode clientX clientY ctrlKey currentTarget data detail eventPhase fromElement handler keyCode layerX layerY metaKey newValue offsetX offsetY originalTarget pageX pageY prevValue relatedNode relatedTarget screenX screenY shiftKey srcElement target toElement view wheelDelta which".split(" "),

	fix: function( event ) {
		if ( event[ expando ] ) {
			return event;
		}

		// store a copy of the original event object
		// and "clone" to set read-only properties
		var originalEvent = event;
		event = jQuery.Event( originalEvent );

		for ( var i = this.props.length, prop; i; ) {
			prop = this.props[ --i ];
			event[ prop ] = originalEvent[ prop ];
		}

		// Fix target property, if necessary
		if ( !event.target ) {
			event.target = event.srcElement || document; // Fixes #1925 where srcElement might not be defined either
		}

		// check if target is a textnode (safari)
		if ( event.target.nodeType === 3 ) {
			event.target = event.target.parentNode;
		}

		// Add relatedTarget, if necessary
		if ( !event.relatedTarget && event.fromElement ) {
			event.relatedTarget = event.fromElement === event.target ? event.toElement : event.fromElement;
		}

		// Calculate pageX/Y if missing and clientX/Y available
		if ( event.pageX == null && event.clientX != null ) {
			var doc = document.documentElement, body = document.body;
			event.pageX = event.clientX + (doc && doc.scrollLeft || body && body.scrollLeft || 0) - (doc && doc.clientLeft || body && body.clientLeft || 0);
			event.pageY = event.clientY + (doc && doc.scrollTop  || body && body.scrollTop  || 0) - (doc && doc.clientTop  || body && body.clientTop  || 0);
		}

		// Add which for key events
		if ( !event.which && ((event.charCode || event.charCode === 0) ? event.charCode : event.keyCode) ) {
			event.which = event.charCode || event.keyCode;
		}

		// Add metaKey to non-Mac browsers (use ctrl for PC's and Meta for Macs)
		if ( !event.metaKey && event.ctrlKey ) {
			event.metaKey = event.ctrlKey;
		}

		// Add which for click: 1 === left; 2 === middle; 3 === right
		// Note: button is not normalized, so don't use it
		if ( !event.which && event.button !== undefined ) {
			event.which = (event.button & 1 ? 1 : ( event.button & 2 ? 3 : ( event.button & 4 ? 2 : 0 ) ));
		}

		return event;
	},

	// Deprecated, use jQuery.guid instead
	guid: 1E8,

	// Deprecated, use jQuery.proxy instead
	proxy: jQuery.proxy,

	special: {
		ready: {
			// Make sure the ready event is setup
			setup: jQuery.bindReady,
			teardown: jQuery.noop
		},

		live: {
			add: function( proxy, data, namespaces, live ) {
				jQuery.extend( proxy, data || {} );

				proxy.guid += data.selector + data.live; 
				data.liveProxy = proxy;

				jQuery.event.add( this, data.live, liveHandler, data ); 
				
			},

			remove: function( namespaces ) {
				if ( namespaces.length ) {
					var remove = 0, name = new RegExp("(^|\\.)" + namespaces[0] + "(\\.|$)");

					jQuery.each( (jQuery.data(this, "events").live || {}), function() {
						if ( name.test(this.type) ) {
							remove++;
						}
					});

					if ( remove < 1 ) {
						jQuery.event.remove( this, namespaces[0], liveHandler );
					}
				}
			},
			special: {}
		},
		beforeunload: {
			setup: function( data, namespaces, fn ) {
				// We only want to do this special case on windows
				if ( this.setInterval ) {
					this.onbeforeunload = fn;
				}

				return false;
			},
			teardown: function( namespaces, fn ) {
				if ( this.onbeforeunload === fn ) {
					this.onbeforeunload = null;
				}
			}
		}
	}
};

jQuery.Event = function( src ) {
	// Allow instantiation without the 'new' keyword
	if ( !this.preventDefault ) {
		return new jQuery.Event( src );
	}

	// Event object
	if ( src && src.type ) {
		this.originalEvent = src;
		this.type = src.type;
	// Event type
	} else {
		this.type = src;
	}

	// timeStamp is buggy for some events on Firefox(#3843)
	// So we won't rely on the native value
	this.timeStamp = now();

	// Mark it as fixed
	this[ expando ] = true;
};

function returnFalse() {
	return false;
}
function returnTrue() {
	return true;
}

// jQuery.Event is based on DOM3 Events as specified by the ECMAScript Language Binding
// http://www.w3.org/TR/2003/WD-DOM-Level-3-Events-20030331/ecma-script-binding.html
jQuery.Event.prototype = {
	preventDefault: function() {
		this.isDefaultPrevented = returnTrue;

		var e = this.originalEvent;
		if ( !e ) {
			return;
		}
		
		// if preventDefault exists run it on the original event
		if ( e.preventDefault ) {
			e.preventDefault();
		}
		// otherwise set the returnValue property of the original event to false (IE)
		e.returnValue = false;
	},
	stopPropagation: function() {
		this.isPropagationStopped = returnTrue;

		var e = this.originalEvent;
		if ( !e ) {
			return;
		}
		// if stopPropagation exists run it on the original event
		if ( e.stopPropagation ) {
			e.stopPropagation();
		}
		// otherwise set the cancelBubble property of the original event to true (IE)
		e.cancelBubble = true;
	},
	stopImmediatePropagation: function() {
		this.isImmediatePropagationStopped = returnTrue;
		this.stopPropagation();
	},
	isDefaultPrevented: returnFalse,
	isPropagationStopped: returnFalse,
	isImmediatePropagationStopped: returnFalse
};

// Checks if an event happened on an element within another element
// Used in jQuery.event.special.mouseenter and mouseleave handlers
var withinElement = function( event ) {
	// Check if mouse(over|out) are still within the same parent element
	var parent = event.relatedTarget;

	// Traverse up the tree
	while ( parent && parent !== this ) {
		// Firefox sometimes assigns relatedTarget a XUL element
		// which we cannot access the parentNode property of
		try {
			parent = parent.parentNode;

		// assuming we've left the element since we most likely mousedover a xul element
		} catch(e) {
			break;
		}
	}

	if ( parent !== this ) {
		// set the correct event type
		event.type = event.data;

		// handle event if we actually just moused on to a non sub-element
		jQuery.event.handle.apply( this, arguments );
	}

},

// In case of event delegation, we only need to rename the event.type,
// liveHandler will take care of the rest.
delegate = function( event ) {
	event.type = event.data;
	jQuery.event.handle.apply( this, arguments );
};

// Create mouseenter and mouseleave events
jQuery.each({
	mouseenter: "mouseover",
	mouseleave: "mouseout"
}, function( orig, fix ) {
	jQuery.event.special[ orig ] = {
		setup: function( data ) {
			jQuery.event.add( this, fix, data && data.selector ? delegate : withinElement, orig );
		},
		teardown: function( data ) {
			jQuery.event.remove( this, fix, data && data.selector ? delegate : withinElement );
		}
	};
});

// submit delegation
if ( !jQuery.support.submitBubbles ) {

jQuery.event.special.submit = {
	setup: function( data, namespaces, fn ) {
		if ( this.nodeName.toLowerCase() !== "form" ) {
			jQuery.event.add(this, "click.specialSubmit." + fn.guid, function( e ) {
				var elem = e.target, type = elem.type;

				if ( (type === "submit" || type === "image") && jQuery( elem ).closest("form").length ) {
					return trigger( "submit", this, arguments );
				}
			});
	 
			jQuery.event.add(this, "keypress.specialSubmit." + fn.guid, function( e ) {
				var elem = e.target, type = elem.type;

				if ( (type === "text" || type === "password") && jQuery( elem ).closest("form").length && e.keyCode === 13 ) {
					return trigger( "submit", this, arguments );
				}
			});

		} else {
			return false;
		}
	},

	remove: function( namespaces, fn ) {
		jQuery.event.remove( this, "click.specialSubmit" + (fn ? "."+fn.guid : "") );
		jQuery.event.remove( this, "keypress.specialSubmit" + (fn ? "."+fn.guid : "") );
	}
};

}

// change delegation, happens here so we have bind.
if ( !jQuery.support.changeBubbles ) {

var formElems = /textarea|input|select/i;

function getVal( elem ) {
	var type = elem.type, val = elem.value;

	if ( type === "radio" || type === "checkbox" ) {
		val = elem.checked;

	} else if ( type === "select-multiple" ) {
		val = elem.selectedIndex > -1 ?
			jQuery.map( elem.options, function( elem ) {
				return elem.selected;
			}).join("-") :
			"";

	} else if ( elem.nodeName.toLowerCase() === "select" ) {
		val = elem.selectedIndex;
	}

	return val;
}

function testChange( e ) {
		var elem = e.target, data, val;

		if ( !formElems.test( elem.nodeName ) || elem.readOnly ) {
			return;
		}

		data = jQuery.data( elem, "_change_data" );
		val = getVal(elem);

		// the current data will be also retrieved by beforeactivate
		if ( e.type !== "focusout" || elem.type !== "radio" ) {
			jQuery.data( elem, "_change_data", val );
		}
		
		if ( data === undefined || val === data ) {
			return;
		}

		if ( data != null || val ) {
			e.type = "change";
			return jQuery.event.trigger( e, arguments[1], elem );
		}
}

jQuery.event.special.change = {
	filters: {
		focusout: testChange, 

		click: function( e ) {
			var elem = e.target, type = elem.type;

			if ( type === "radio" || type === "checkbox" || elem.nodeName.toLowerCase() === "select" ) {
				return testChange.call( this, e );
			}
		},

		// Change has to be called before submit
		// Keydown will be called before keypress, which is used in submit-event delegation
		keydown: function( e ) {
			var elem = e.target, type = elem.type;

			if ( (e.keyCode === 13 && elem.nodeName.toLowerCase() !== "textarea") ||
				(e.keyCode === 32 && (type === "checkbox" || type === "radio")) ||
				type === "select-multiple" ) {
				return testChange.call( this, e );
			}
		},

		// Beforeactivate happens also before the previous element is blurred
		// with this event you can't trigger a change event, but you can store
		// information/focus[in] is not needed anymore
		beforeactivate: function( e ) {
			var elem = e.target;

			if ( elem.nodeName.toLowerCase() === "input" && elem.type === "radio" ) {
				jQuery.data( elem, "_change_data", getVal(elem) );
			}
		}
	},
	setup: function( data, namespaces, fn ) {
		for ( var type in changeFilters ) {
			jQuery.event.add( this, type + ".specialChange." + fn.guid, changeFilters[type] );
		}

		return formElems.test( this.nodeName );
	},
	remove: function( namespaces, fn ) {
		for ( var type in changeFilters ) {
			jQuery.event.remove( this, type + ".specialChange" + (fn ? "."+fn.guid : ""), changeFilters[type] );
		}

		return formElems.test( this.nodeName );
	}
};

var changeFilters = jQuery.event.special.change.filters;

}

function trigger( type, elem, args ) {
	args[0].type = type;
	return jQuery.event.handle.apply( elem, args );
}

// Create "bubbling" focus and blur events
if ( document.addEventListener ) {
	jQuery.each({ focus: "focusin", blur: "focusout" }, function( orig, fix ) {
		jQuery.event.special[ fix ] = {
			setup: function() {
				this.addEventListener( orig, handler, true );
			}, 
			teardown: function() { 
				this.removeEventListener( orig, handler, true );
			}
		};

		function handler( e ) { 
			e = jQuery.event.fix( e );
			e.type = fix;
			return jQuery.event.handle.call( this, e );
		}
	});
}

jQuery.each(["bind", "one"], function( i, name ) {
	jQuery.fn[ name ] = function( type, data, fn ) {
		// Handle object literals
		if ( typeof type === "object" ) {
			for ( var key in type ) {
				this[ name ](key, data, type[key], fn);
			}
			return this;
		}
		
		if ( jQuery.isFunction( data ) ) {
			fn = data;
			data = undefined;
		}

		var handler = name === "one" ? jQuery.proxy( fn, function( event ) {
			jQuery( this ).unbind( event, handler );
			return fn.apply( this, arguments );
		}) : fn;

		return type === "unload" && name !== "one" ?
			this.one( type, data, fn ) :
			this.each(function() {
				jQuery.event.add( this, type, handler, data );
			});
	};
});

jQuery.fn.extend({
	unbind: function( type, fn ) {
		// Handle object literals
		if ( typeof type === "object" && !type.preventDefault ) {
			for ( var key in type ) {
				this.unbind(key, type[key]);
			}
			return this;
		}

		return this.each(function() {
			jQuery.event.remove( this, type, fn );
		});
	},
	trigger: function( type, data ) {
		return this.each(function() {
			jQuery.event.trigger( type, data, this );
		});
	},

	triggerHandler: function( type, data ) {
		if ( this[0] ) {
			var event = jQuery.Event( type );
			event.preventDefault();
			event.stopPropagation();
			jQuery.event.trigger( event, data, this[0] );
			return event.result;
		}
	},

	toggle: function( fn ) {
		// Save reference to arguments for access in closure
		var args = arguments, i = 1;

		// link all the functions, so any of them can unbind this click handler
		while ( i < args.length ) {
			jQuery.proxy( fn, args[ i++ ] );
		}

		return this.click( jQuery.proxy( fn, function( event ) {
			// Figure out which function to execute
			var lastToggle = ( jQuery.data( this, "lastToggle" + fn.guid ) || 0 ) % i;
			jQuery.data( this, "lastToggle" + fn.guid, lastToggle + 1 );

			// Make sure that clicks stop
			event.preventDefault();

			// and execute the function
			return args[ lastToggle ].apply( this, arguments ) || false;
		}));
	},

	hover: function( fnOver, fnOut ) {
		return this.mouseenter( fnOver ).mouseleave( fnOut || fnOver );
	}
});

jQuery.each(["live", "die"], function( i, name ) {
	jQuery.fn[ name ] = function( types, data, fn ) {
		var type, i = 0;

		if ( jQuery.isFunction( data ) ) {
			fn = data;
			data = undefined;
		}

		types = (types || "").split( /\s+/ );

		while ( (type = types[ i++ ]) != null ) {
			type = type === "focus" ? "focusin" : // focus --> focusin
					type === "blur" ? "focusout" : // blur --> focusout
					type === "hover" ? types.push("mouseleave") && "mouseenter" : // hover support
					type;
			
			if ( name === "live" ) {
				// bind live handler
				jQuery( this.context ).bind( liveConvert( type, this.selector ), {
					data: data, selector: this.selector, live: type
				}, fn );

			} else {
				// unbind live handler
				jQuery( this.context ).unbind( liveConvert( type, this.selector ), fn ? { guid: fn.guid + this.selector + type } : null );
			}
		}
		
		return this;
	}
});

function liveHandler( event ) {
	var stop, elems = [], selectors = [], args = arguments,
		related, match, fn, elem, j, i, l, data,
		live = jQuery.extend({}, jQuery.data( this, "events" ).live);

	// Make sure we avoid non-left-click bubbling in Firefox (#3861)
	if ( event.button && event.type === "click" ) {
		return;
	}

	for ( j in live ) {
		fn = live[j];
		if ( fn.live === event.type ||
				fn.altLive && jQuery.inArray(event.type, fn.altLive) > -1 ) {

			data = fn.data;
			if ( !(data.beforeFilter && data.beforeFilter[event.type] && 
					!data.beforeFilter[event.type](event)) ) {
				selectors.push( fn.selector );
			}
		} else {
			delete live[j];
		}
	}

	match = jQuery( event.target ).closest( selectors, event.currentTarget );

	for ( i = 0, l = match.length; i < l; i++ ) {
		for ( j in live ) {
			fn = live[j];
			elem = match[i].elem;
			related = null;

			if ( match[i].selector === fn.selector ) {
				// Those two events require additional checking
				if ( fn.live === "mouseenter" || fn.live === "mouseleave" ) {
					related = jQuery( event.relatedTarget ).closest( fn.selector )[0];
				}

				if ( !related || related !== elem ) {
					elems.push({ elem: elem, fn: fn });
				}
			}
		}
	}

	for ( i = 0, l = elems.length; i < l; i++ ) {
		match = elems[i];
		event.currentTarget = match.elem;
		event.data = match.fn.data;
		if ( match.fn.apply( match.elem, args ) === false ) {
			stop = false;
			break;
		}
	}

	return stop;
}

function liveConvert( type, selector ) {
	return "live." + (type ? type + "." : "") + selector.replace(/\./g, "`").replace(/ /g, "&");
}

jQuery.each( ("blur focus focusin focusout load resize scroll unload click dblclick " +
	"mousedown mouseup mousemove mouseover mouseout mouseenter mouseleave " +
	"change select submit keydown keypress keyup error").split(" "), function( i, name ) {

	// Handle event binding
	jQuery.fn[ name ] = function( fn ) {
		return fn ? this.bind( name, fn ) : this.trigger( name );
	};

	if ( jQuery.attrFn ) {
		jQuery.attrFn[ name ] = true;
	}
});

// Prevent memory leaks in IE
// Window isn't included so as not to unbind existing unload events
// More info:
//  - http://isaacschlueter.com/2006/10/msie-memory-leaks/
if ( window.attachEvent && !window.addEventListener ) {
	window.attachEvent("onunload", function() {
		for ( var id in jQuery.cache ) {
			if ( jQuery.cache[ id ].handle ) {
				// Try/Catch is to handle iframes being unloaded, see #4280
				try {
					jQuery.event.remove( jQuery.cache[ id ].handle.elem );
				} catch(e) {}
			}
		}
	});
}
/*!
 * Sizzle CSS Selector Engine - v1.0
 *  Copyright 2009, The Dojo Foundation
 *  Released under the MIT, BSD, and GPL Licenses.
 *  More information: http://sizzlejs.com/
 */
(function(){

var chunker = /((?:\((?:\([^()]+\)|[^()]+)+\)|\[(?:\[[^[\]]*\]|['"][^'"]*['"]|[^[\]'"]+)+\]|\\.|[^ >+~,(\[\\]+)+|[>+~])(\s*,\s*)?((?:.|\r|\n)*)/g,
	done = 0,
	toString = Object.prototype.toString,
	hasDuplicate = false,
	baseHasDuplicate = true;

// Here we check if the JavaScript engine is using some sort of
// optimization where it does not always call our comparision
// function. If that is the case, discard the hasDuplicate value.
//   Thus far that includes Google Chrome.
[0, 0].sort(function(){
	baseHasDuplicate = false;
	return 0;
});

var Sizzle = function(selector, context, results, seed) {
	results = results || [];
	var origContext = context = context || document;

	if ( context.nodeType !== 1 && context.nodeType !== 9 ) {
		return [];
	}
	
	if ( !selector || typeof selector !== "string" ) {
		return results;
	}

	var parts = [], m, set, checkSet, extra, prune = true, contextXML = isXML(context),
		soFar = selector;
	
	// Reset the position of the chunker regexp (start from head)
	while ( (chunker.exec(""), m = chunker.exec(soFar)) !== null ) {
		soFar = m[3];
		
		parts.push( m[1] );
		
		if ( m[2] ) {
			extra = m[3];
			break;
		}
	}

	if ( parts.length > 1 && origPOS.exec( selector ) ) {
		if ( parts.length === 2 && Expr.relative[ parts[0] ] ) {
			set = posProcess( parts[0] + parts[1], context );
		} else {
			set = Expr.relative[ parts[0] ] ?
				[ context ] :
				Sizzle( parts.shift(), context );

			while ( parts.length ) {
				selector = parts.shift();

				if ( Expr.relative[ selector ] ) {
					selector += parts.shift();
				}
				
				set = posProcess( selector, set );
			}
		}
	} else {
		// Take a shortcut and set the context if the root selector is an ID
		// (but not if it'll be faster if the inner selector is an ID)
		if ( !seed && parts.length > 1 && context.nodeType === 9 && !contextXML &&
				Expr.match.ID.test(parts[0]) && !Expr.match.ID.test(parts[parts.length - 1]) ) {
			var ret = Sizzle.find( parts.shift(), context, contextXML );
			context = ret.expr ? Sizzle.filter( ret.expr, ret.set )[0] : ret.set[0];
		}

		if ( context ) {
			var ret = seed ?
				{ expr: parts.pop(), set: makeArray(seed) } :
				Sizzle.find( parts.pop(), parts.length === 1 && (parts[0] === "~" || parts[0] === "+") && context.parentNode ? context.parentNode : context, contextXML );
			set = ret.expr ? Sizzle.filter( ret.expr, ret.set ) : ret.set;

			if ( parts.length > 0 ) {
				checkSet = makeArray(set);
			} else {
				prune = false;
			}

			while ( parts.length ) {
				var cur = parts.pop(), pop = cur;

				if ( !Expr.relative[ cur ] ) {
					cur = "";
				} else {
					pop = parts.pop();
				}

				if ( pop == null ) {
					pop = context;
				}

				Expr.relative[ cur ]( checkSet, pop, contextXML );
			}
		} else {
			checkSet = parts = [];
		}
	}

	if ( !checkSet ) {
		checkSet = set;
	}

	if ( !checkSet ) {
		Sizzle.error( cur || selector );
	}

	if ( toString.call(checkSet) === "[object Array]" ) {
		if ( !prune ) {
			results.push.apply( results, checkSet );
		} else if ( context && context.nodeType === 1 ) {
			for ( var i = 0; checkSet[i] != null; i++ ) {
				if ( checkSet[i] && (checkSet[i] === true || checkSet[i].nodeType === 1 && contains(context, checkSet[i])) ) {
					results.push( set[i] );
				}
			}
		} else {
			for ( var i = 0; checkSet[i] != null; i++ ) {
				if ( checkSet[i] && checkSet[i].nodeType === 1 ) {
					results.push( set[i] );
				}
			}
		}
	} else {
		makeArray( checkSet, results );
	}

	if ( extra ) {
		Sizzle( extra, origContext, results, seed );
		Sizzle.uniqueSort( results );
	}

	return results;
};

Sizzle.uniqueSort = function(results){
	if ( sortOrder ) {
		hasDuplicate = baseHasDuplicate;
		results.sort(sortOrder);

		if ( hasDuplicate ) {
			for ( var i = 1; i < results.length; i++ ) {
				if ( results[i] === results[i-1] ) {
					results.splice(i--, 1);
				}
			}
		}
	}

	return results;
};

Sizzle.matches = function(expr, set){
	return Sizzle(expr, null, null, set);
};

Sizzle.find = function(expr, context, isXML){
	var set, match;

	if ( !expr ) {
		return [];
	}

	for ( var i = 0, l = Expr.order.length; i < l; i++ ) {
		var type = Expr.order[i], match;
		
		if ( (match = Expr.leftMatch[ type ].exec( expr )) ) {
			var left = match[1];
			match.splice(1,1);

			if ( left.substr( left.length - 1 ) !== "\\" ) {
				match[1] = (match[1] || "").replace(/\\/g, "");
				set = Expr.find[ type ]( match, context, isXML );
				if ( set != null ) {
					expr = expr.replace( Expr.match[ type ], "" );
					break;
				}
			}
		}
	}

	if ( !set ) {
		set = context.getElementsByTagName("*");
	}

	return {set: set, expr: expr};
};

Sizzle.filter = function(expr, set, inplace, not){
	var old = expr, result = [], curLoop = set, match, anyFound,
		isXMLFilter = set && set[0] && isXML(set[0]);

	while ( expr && set.length ) {
		for ( var type in Expr.filter ) {
			if ( (match = Expr.leftMatch[ type ].exec( expr )) != null && match[2] ) {
				var filter = Expr.filter[ type ], found, item, left = match[1];
				anyFound = false;

				match.splice(1,1);

				if ( left.substr( left.length - 1 ) === "\\" ) {
					continue;
				}

				if ( curLoop === result ) {
					result = [];
				}

				if ( Expr.preFilter[ type ] ) {
					match = Expr.preFilter[ type ]( match, curLoop, inplace, result, not, isXMLFilter );

					if ( !match ) {
						anyFound = found = true;
					} else if ( match === true ) {
						continue;
					}
				}

				if ( match ) {
					for ( var i = 0; (item = curLoop[i]) != null; i++ ) {
						if ( item ) {
							found = filter( item, match, i, curLoop );
							var pass = not ^ !!found;

							if ( inplace && found != null ) {
								if ( pass ) {
									anyFound = true;
								} else {
									curLoop[i] = false;
								}
							} else if ( pass ) {
								result.push( item );
								anyFound = true;
							}
						}
					}
				}

				if ( found !== undefined ) {
					if ( !inplace ) {
						curLoop = result;
					}

					expr = expr.replace( Expr.match[ type ], "" );

					if ( !anyFound ) {
						return [];
					}

					break;
				}
			}
		}

		// Improper expression
		if ( expr === old ) {
			if ( anyFound == null ) {
				Sizzle.error( expr );
			} else {
				break;
			}
		}

		old = expr;
	}

	return curLoop;
};

Sizzle.error = function( msg ) {
	throw "Syntax error, unrecognized expression: " + msg;
};

var Expr = Sizzle.selectors = {
	order: [ "ID", "NAME", "TAG" ],
	match: {
		ID: /#((?:[\w\u00c0-\uFFFF-]|\\.)+)/,
		CLASS: /\.((?:[\w\u00c0-\uFFFF-]|\\.)+)/,
		NAME: /\[name=['"]*((?:[\w\u00c0-\uFFFF-]|\\.)+)['"]*\]/,
		ATTR: /\[\s*((?:[\w\u00c0-\uFFFF-]|\\.)+)\s*(?:(\S?=)\s*(['"]*)(.*?)\3|)\s*\]/,
		TAG: /^((?:[\w\u00c0-\uFFFF\*-]|\\.)+)/,
		CHILD: /:(only|nth|last|first)-child(?:\((even|odd|[\dn+-]*)\))?/,
		POS: /:(nth|eq|gt|lt|first|last|even|odd)(?:\((\d*)\))?(?=[^-]|$)/,
		PSEUDO: /:((?:[\w\u00c0-\uFFFF-]|\\.)+)(?:\((['"]?)((?:\([^\)]+\)|[^\(\)]*)+)\2\))?/
	},
	leftMatch: {},
	attrMap: {
		"class": "className",
		"for": "htmlFor"
	},
	attrHandle: {
		href: function(elem){
			return elem.getAttribute("href");
		}
	},
	relative: {
		"+": function(checkSet, part){
			var isPartStr = typeof part === "string",
				isTag = isPartStr && !/\W/.test(part),
				isPartStrNotTag = isPartStr && !isTag;

			if ( isTag ) {
				part = part.toLowerCase();
			}

			for ( var i = 0, l = checkSet.length, elem; i < l; i++ ) {
				if ( (elem = checkSet[i]) ) {
					while ( (elem = elem.previousSibling) && elem.nodeType !== 1 ) {}

					checkSet[i] = isPartStrNotTag || elem && elem.nodeName.toLowerCase() === part ?
						elem || false :
						elem === part;
				}
			}

			if ( isPartStrNotTag ) {
				Sizzle.filter( part, checkSet, true );
			}
		},
		">": function(checkSet, part){
			var isPartStr = typeof part === "string";

			if ( isPartStr && !/\W/.test(part) ) {
				part = part.toLowerCase();

				for ( var i = 0, l = checkSet.length; i < l; i++ ) {
					var elem = checkSet[i];
					if ( elem ) {
						var parent = elem.parentNode;
						checkSet[i] = parent.nodeName.toLowerCase() === part ? parent : false;
					}
				}
			} else {
				for ( var i = 0, l = checkSet.length; i < l; i++ ) {
					var elem = checkSet[i];
					if ( elem ) {
						checkSet[i] = isPartStr ?
							elem.parentNode :
							elem.parentNode === part;
					}
				}

				if ( isPartStr ) {
					Sizzle.filter( part, checkSet, true );
				}
			}
		},
		"": function(checkSet, part, isXML){
			var doneName = done++, checkFn = dirCheck;

			if ( typeof part === "string" && !/\W/.test(part) ) {
				var nodeCheck = part = part.toLowerCase();
				checkFn = dirNodeCheck;
			}

			checkFn("parentNode", part, doneName, checkSet, nodeCheck, isXML);
		},
		"~": function(checkSet, part, isXML){
			var doneName = done++, checkFn = dirCheck;

			if ( typeof part === "string" && !/\W/.test(part) ) {
				var nodeCheck = part = part.toLowerCase();
				checkFn = dirNodeCheck;
			}

			checkFn("previousSibling", part, doneName, checkSet, nodeCheck, isXML);
		}
	},
	find: {
		ID: function(match, context, isXML){
			if ( typeof context.getElementById !== "undefined" && !isXML ) {
				var m = context.getElementById(match[1]);
				return m ? [m] : [];
			}
		},
		NAME: function(match, context){
			if ( typeof context.getElementsByName !== "undefined" ) {
				var ret = [], results = context.getElementsByName(match[1]);

				for ( var i = 0, l = results.length; i < l; i++ ) {
					if ( results[i].getAttribute("name") === match[1] ) {
						ret.push( results[i] );
					}
				}

				return ret.length === 0 ? null : ret;
			}
		},
		TAG: function(match, context){
			return context.getElementsByTagName(match[1]);
		}
	},
	preFilter: {
		CLASS: function(match, curLoop, inplace, result, not, isXML){
			match = " " + match[1].replace(/\\/g, "") + " ";

			if ( isXML ) {
				return match;
			}

			for ( var i = 0, elem; (elem = curLoop[i]) != null; i++ ) {
				if ( elem ) {
					if ( not ^ (elem.className && (" " + elem.className + " ").replace(/[\t\n]/g, " ").indexOf(match) >= 0) ) {
						if ( !inplace ) {
							result.push( elem );
						}
					} else if ( inplace ) {
						curLoop[i] = false;
					}
				}
			}

			return false;
		},
		ID: function(match){
			return match[1].replace(/\\/g, "");
		},
		TAG: function(match, curLoop){
			return match[1].toLowerCase();
		},
		CHILD: function(match){
			if ( match[1] === "nth" ) {
				// parse equations like 'even', 'odd', '5', '2n', '3n+2', '4n-1', '-n+6'
				var test = /(-?)(\d*)n((?:\+|-)?\d*)/.exec(
					match[2] === "even" && "2n" || match[2] === "odd" && "2n+1" ||
					!/\D/.test( match[2] ) && "0n+" + match[2] || match[2]);

				// calculate the numbers (first)n+(last) including if they are negative
				match[2] = (test[1] + (test[2] || 1)) - 0;
				match[3] = test[3] - 0;
			}

			// TODO: Move to normal caching system
			match[0] = done++;

			return match;
		},
		ATTR: function(match, curLoop, inplace, result, not, isXML){
			var name = match[1].replace(/\\/g, "");
			
			if ( !isXML && Expr.attrMap[name] ) {
				match[1] = Expr.attrMap[name];
			}

			if ( match[2] === "~=" ) {
				match[4] = " " + match[4] + " ";
			}

			return match;
		},
		PSEUDO: function(match, curLoop, inplace, result, not){
			if ( match[1] === "not" ) {
				// If we're dealing with a complex expression, or a simple one
				if ( ( chunker.exec(match[3]) || "" ).length > 1 || /^\w/.test(match[3]) ) {
					match[3] = Sizzle(match[3], null, null, curLoop);
				} else {
					var ret = Sizzle.filter(match[3], curLoop, inplace, true ^ not);
					if ( !inplace ) {
						result.push.apply( result, ret );
					}
					return false;
				}
			} else if ( Expr.match.POS.test( match[0] ) || Expr.match.CHILD.test( match[0] ) ) {
				return true;
			}
			
			return match;
		},
		POS: function(match){
			match.unshift( true );
			return match;
		}
	},
	filters: {
		enabled: function(elem){
			return elem.disabled === false && elem.type !== "hidden";
		},
		disabled: function(elem){
			return elem.disabled === true;
		},
		checked: function(elem){
			return elem.checked === true;
		},
		selected: function(elem){
			// Accessing this property makes selected-by-default
			// options in Safari work properly
			elem.parentNode.selectedIndex;
			return elem.selected === true;
		},
		parent: function(elem){
			return !!elem.firstChild;
		},
		empty: function(elem){
			return !elem.firstChild;
		},
		has: function(elem, i, match){
			return !!Sizzle( match[3], elem ).length;
		},
		header: function(elem){
			return /h\d/i.test( elem.nodeName );
		},
		text: function(elem){
			return "text" === elem.type;
		},
		radio: function(elem){
			return "radio" === elem.type;
		},
		checkbox: function(elem){
			return "checkbox" === elem.type;
		},
		file: function(elem){
			return "file" === elem.type;
		},
		password: function(elem){
			return "password" === elem.type;
		},
		submit: function(elem){
			return "submit" === elem.type;
		},
		image: function(elem){
			return "image" === elem.type;
		},
		reset: function(elem){
			return "reset" === elem.type;
		},
		button: function(elem){
			return "button" === elem.type || elem.nodeName.toLowerCase() === "button";
		},
		input: function(elem){
			return /input|select|textarea|button/i.test(elem.nodeName);
		}
	},
	setFilters: {
		first: function(elem, i){
			return i === 0;
		},
		last: function(elem, i, match, array){
			return i === array.length - 1;
		},
		even: function(elem, i){
			return i % 2 === 0;
		},
		odd: function(elem, i){
			return i % 2 === 1;
		},
		lt: function(elem, i, match){
			return i < match[3] - 0;
		},
		gt: function(elem, i, match){
			return i > match[3] - 0;
		},
		nth: function(elem, i, match){
			return match[3] - 0 === i;
		},
		eq: function(elem, i, match){
			return match[3] - 0 === i;
		}
	},
	filter: {
		PSEUDO: function(elem, match, i, array){
			var name = match[1], filter = Expr.filters[ name ];

			if ( filter ) {
				return filter( elem, i, match, array );
			} else if ( name === "contains" ) {
				return (elem.textContent || elem.innerText || getText([ elem ]) || "").indexOf(match[3]) >= 0;
			} else if ( name === "not" ) {
				var not = match[3];

				for ( var i = 0, l = not.length; i < l; i++ ) {
					if ( not[i] === elem ) {
						return false;
					}
				}

				return true;
			} else {
				Sizzle.error( "Syntax error, unrecognized expression: " + name );
			}
		},
		CHILD: function(elem, match){
			var type = match[1], node = elem;
			switch (type) {
				case 'only':
				case 'first':
					while ( (node = node.previousSibling) )	 {
						if ( node.nodeType === 1 ) { 
							return false; 
						}
					}
					if ( type === "first" ) { 
						return true; 
					}
					node = elem;
				case 'last':
					while ( (node = node.nextSibling) )	 {
						if ( node.nodeType === 1 ) { 
							return false; 
						}
					}
					return true;
				case 'nth':
					var first = match[2], last = match[3];

					if ( first === 1 && last === 0 ) {
						return true;
					}
					
					var doneName = match[0],
						parent = elem.parentNode;
	
					if ( parent && (parent.sizcache !== doneName || !elem.nodeIndex) ) {
						var count = 0;
						for ( node = parent.firstChild; node; node = node.nextSibling ) {
							if ( node.nodeType === 1 ) {
								node.nodeIndex = ++count;
							}
						} 
						parent.sizcache = doneName;
					}
					
					var diff = elem.nodeIndex - last;
					if ( first === 0 ) {
						return diff === 0;
					} else {
						return ( diff % first === 0 && diff / first >= 0 );
					}
			}
		},
		ID: function(elem, match){
			return elem.nodeType === 1 && elem.getAttribute("id") === match;
		},
		TAG: function(elem, match){
			return (match === "*" && elem.nodeType === 1) || elem.nodeName.toLowerCase() === match;
		},
		CLASS: function(elem, match){
			return (" " + (elem.className || elem.getAttribute("class")) + " ")
				.indexOf( match ) > -1;
		},
		ATTR: function(elem, match){
			var name = match[1],
				result = Expr.attrHandle[ name ] ?
					Expr.attrHandle[ name ]( elem ) :
					elem[ name ] != null ?
						elem[ name ] :
						elem.getAttribute( name ),
				value = result + "",
				type = match[2],
				check = match[4];

			return result == null ?
				type === "!=" :
				type === "=" ?
				value === check :
				type === "*=" ?
				value.indexOf(check) >= 0 :
				type === "~=" ?
				(" " + value + " ").indexOf(check) >= 0 :
				!check ?
				value && result !== false :
				type === "!=" ?
				value !== check :
				type === "^=" ?
				value.indexOf(check) === 0 :
				type === "$=" ?
				value.substr(value.length - check.length) === check :
				type === "|=" ?
				value === check || value.substr(0, check.length + 1) === check + "-" :
				false;
		},
		POS: function(elem, match, i, array){
			var name = match[2], filter = Expr.setFilters[ name ];

			if ( filter ) {
				return filter( elem, i, match, array );
			}
		}
	}
};

var origPOS = Expr.match.POS;

for ( var type in Expr.match ) {
	Expr.match[ type ] = new RegExp( Expr.match[ type ].source + /(?![^\[]*\])(?![^\(]*\))/.source );
	Expr.leftMatch[ type ] = new RegExp( /(^(?:.|\r|\n)*?)/.source + Expr.match[ type ].source.replace(/\\(\d+)/g, function(all, num){
		return "\\" + (num - 0 + 1);
	}));
}

var makeArray = function(array, results) {
	array = Array.prototype.slice.call( array, 0 );

	if ( results ) {
		results.push.apply( results, array );
		return results;
	}
	
	return array;
};

// Perform a simple check to determine if the browser is capable of
// converting a NodeList to an array using builtin methods.
try {
	Array.prototype.slice.call( document.documentElement.childNodes, 0 );

// Provide a fallback method if it does not work
} catch(e){
	makeArray = function(array, results) {
		var ret = results || [];

		if ( toString.call(array) === "[object Array]" ) {
			Array.prototype.push.apply( ret, array );
		} else {
			if ( typeof array.length === "number" ) {
				for ( var i = 0, l = array.length; i < l; i++ ) {
					ret.push( array[i] );
				}
			} else {
				for ( var i = 0; array[i]; i++ ) {
					ret.push( array[i] );
				}
			}
		}

		return ret;
	};
}

var sortOrder;

if ( document.documentElement.compareDocumentPosition ) {
	sortOrder = function( a, b ) {
		if ( !a.compareDocumentPosition || !b.compareDocumentPosition ) {
			if ( a == b ) {
				hasDuplicate = true;
			}
			return a.compareDocumentPosition ? -1 : 1;
		}

		var ret = a.compareDocumentPosition(b) & 4 ? -1 : a === b ? 0 : 1;
		if ( ret === 0 ) {
			hasDuplicate = true;
		}
		return ret;
	};
} else if ( "sourceIndex" in document.documentElement ) {
	sortOrder = function( a, b ) {
		if ( !a.sourceIndex || !b.sourceIndex ) {
			if ( a == b ) {
				hasDuplicate = true;
			}
			return a.sourceIndex ? -1 : 1;
		}

		var ret = a.sourceIndex - b.sourceIndex;
		if ( ret === 0 ) {
			hasDuplicate = true;
		}
		return ret;
	};
} else if ( document.createRange ) {
	sortOrder = function( a, b ) {
		if ( !a.ownerDocument || !b.ownerDocument ) {
			if ( a == b ) {
				hasDuplicate = true;
			}
			return a.ownerDocument ? -1 : 1;
		}

		var aRange = a.ownerDocument.createRange(), bRange = b.ownerDocument.createRange();
		aRange.setStart(a, 0);
		aRange.setEnd(a, 0);
		bRange.setStart(b, 0);
		bRange.setEnd(b, 0);
		var ret = aRange.compareBoundaryPoints(Range.START_TO_END, bRange);
		if ( ret === 0 ) {
			hasDuplicate = true;
		}
		return ret;
	};
}

// Utility function for retreiving the text value of an array of DOM nodes
function getText( elems ) {
	var ret = "", elem;

	for ( var i = 0; elems[i]; i++ ) {
		elem = elems[i];

		// Get the text from text nodes and CDATA nodes
		if ( elem.nodeType === 3 || elem.nodeType === 4 ) {
			ret += elem.nodeValue;

		// Traverse everything else, except comment nodes
		} else if ( elem.nodeType !== 8 ) {
			ret += getText( elem.childNodes );
		}
	}

	return ret;
}

// Check to see if the browser returns elements by name when
// querying by getElementById (and provide a workaround)
(function(){
	// We're going to inject a fake input element with a specified name
	var form = document.createElement("div"),
		id = "script" + (new Date).getTime();
	form.innerHTML = "<a name='" + id + "'/>";

	// Inject it into the root element, check its status, and remove it quickly
	var root = document.documentElement;
	root.insertBefore( form, root.firstChild );

	// The workaround has to do additional checks after a getElementById
	// Which slows things down for other browsers (hence the branching)
	if ( document.getElementById( id ) ) {
		Expr.find.ID = function(match, context, isXML){
			if ( typeof context.getElementById !== "undefined" && !isXML ) {
				var m = context.getElementById(match[1]);
				return m ? m.id === match[1] || typeof m.getAttributeNode !== "undefined" && m.getAttributeNode("id").nodeValue === match[1] ? [m] : undefined : [];
			}
		};

		Expr.filter.ID = function(elem, match){
			var node = typeof elem.getAttributeNode !== "undefined" && elem.getAttributeNode("id");
			return elem.nodeType === 1 && node && node.nodeValue === match;
		};
	}

	root.removeChild( form );
	root = form = null; // release memory in IE
})();

(function(){
	// Check to see if the browser returns only elements
	// when doing getElementsByTagName("*")

	// Create a fake element
	var div = document.createElement("div");
	div.appendChild( document.createComment("") );

	// Make sure no comments are found
	if ( div.getElementsByTagName("*").length > 0 ) {
		Expr.find.TAG = function(match, context){
			var results = context.getElementsByTagName(match[1]);

			// Filter out possible comments
			if ( match[1] === "*" ) {
				var tmp = [];

				for ( var i = 0; results[i]; i++ ) {
					if ( results[i].nodeType === 1 ) {
						tmp.push( results[i] );
					}
				}

				results = tmp;
			}

			return results;
		};
	}

	// Check to see if an attribute returns normalized href attributes
	div.innerHTML = "<a href='#'></a>";
	if ( div.firstChild && typeof div.firstChild.getAttribute !== "undefined" &&
			div.firstChild.getAttribute("href") !== "#" ) {
		Expr.attrHandle.href = function(elem){
			return elem.getAttribute("href", 2);
		};
	}

	div = null; // release memory in IE
})();

if ( document.querySelectorAll ) {
	(function(){
		var oldSizzle = Sizzle, div = document.createElement("div");
		div.innerHTML = "<p class='TEST'></p>";

		// Safari can't handle uppercase or unicode characters when
		// in quirks mode.
		if ( div.querySelectorAll && div.querySelectorAll(".TEST").length === 0 ) {
			return;
		}
	
		Sizzle = function(query, context, extra, seed){
			context = context || document;

			// Only use querySelectorAll on non-XML documents
			// (ID selectors don't work in non-HTML documents)
			if ( !seed && context.nodeType === 9 && !isXML(context) ) {
				try {
					return makeArray( context.querySelectorAll(query), extra );
				} catch(e){}
			}
		
			return oldSizzle(query, context, extra, seed);
		};

		for ( var prop in oldSizzle ) {
			Sizzle[ prop ] = oldSizzle[ prop ];
		}

		div = null; // release memory in IE
	})();
}

(function(){
	var div = document.createElement("div");

	div.innerHTML = "<div class='test e'></div><div class='test'></div>";

	// Opera can't find a second classname (in 9.6)
	// Also, make sure that getElementsByClassName actually exists
	if ( !div.getElementsByClassName || div.getElementsByClassName("e").length === 0 ) {
		return;
	}

	// Safari caches class attributes, doesn't catch changes (in 3.2)
	div.lastChild.className = "e";

	if ( div.getElementsByClassName("e").length === 1 ) {
		return;
	}
	
	Expr.order.splice(1, 0, "CLASS");
	Expr.find.CLASS = function(match, context, isXML) {
		if ( typeof context.getElementsByClassName !== "undefined" && !isXML ) {
			return context.getElementsByClassName(match[1]);
		}
	};

	div = null; // release memory in IE
})();

function dirNodeCheck( dir, cur, doneName, checkSet, nodeCheck, isXML ) {
	for ( var i = 0, l = checkSet.length; i < l; i++ ) {
		var elem = checkSet[i];
		if ( elem ) {
			elem = elem[dir];
			var match = false;

			while ( elem ) {
				if ( elem.sizcache === doneName ) {
					match = checkSet[elem.sizset];
					break;
				}

				if ( elem.nodeType === 1 && !isXML ){
					elem.sizcache = doneName;
					elem.sizset = i;
				}

				if ( elem.nodeName.toLowerCase() === cur ) {
					match = elem;
					break;
				}

				elem = elem[dir];
			}

			checkSet[i] = match;
		}
	}
}

function dirCheck( dir, cur, doneName, checkSet, nodeCheck, isXML ) {
	for ( var i = 0, l = checkSet.length; i < l; i++ ) {
		var elem = checkSet[i];
		if ( elem ) {
			elem = elem[dir];
			var match = false;

			while ( elem ) {
				if ( elem.sizcache === doneName ) {
					match = checkSet[elem.sizset];
					break;
				}

				if ( elem.nodeType === 1 ) {
					if ( !isXML ) {
						elem.sizcache = doneName;
						elem.sizset = i;
					}
					if ( typeof cur !== "string" ) {
						if ( elem === cur ) {
							match = true;
							break;
						}

					} else if ( Sizzle.filter( cur, [elem] ).length > 0 ) {
						match = elem;
						break;
					}
				}

				elem = elem[dir];
			}

			checkSet[i] = match;
		}
	}
}

var contains = document.compareDocumentPosition ? function(a, b){
	return a.compareDocumentPosition(b) & 16;
} : function(a, b){
	return a !== b && (a.contains ? a.contains(b) : true);
};

var isXML = function(elem){
	// documentElement is verified for cases where it doesn't yet exist
	// (such as loading iframes in IE - #4833) 
	var documentElement = (elem ? elem.ownerDocument || elem : 0).documentElement;
	return documentElement ? documentElement.nodeName !== "HTML" : false;
};

var posProcess = function(selector, context){
	var tmpSet = [], later = "", match,
		root = context.nodeType ? [context] : context;

	// Position selectors must be done after the filter
	// And so must :not(positional) so we move all PSEUDOs to the end
	while ( (match = Expr.match.PSEUDO.exec( selector )) ) {
		later += match[0];
		selector = selector.replace( Expr.match.PSEUDO, "" );
	}

	selector = Expr.relative[selector] ? selector + "*" : selector;

	for ( var i = 0, l = root.length; i < l; i++ ) {
		Sizzle( selector, root[i], tmpSet );
	}

	return Sizzle.filter( later, tmpSet );
};

// EXPOSE
jQuery.find = Sizzle;
jQuery.expr = Sizzle.selectors;
jQuery.expr[":"] = jQuery.expr.filters;
jQuery.unique = Sizzle.uniqueSort;
jQuery.getText = getText;
jQuery.isXMLDoc = isXML;
jQuery.contains = contains;

return;

window.Sizzle = Sizzle;

})();
var runtil = /Until$/,
	rparentsprev = /^(?:parents|prevUntil|prevAll)/,
	// Note: This RegExp should be improved, or likely pulled from Sizzle
	rmultiselector = /,/,
	slice = Array.prototype.slice;

// Implement the identical functionality for filter and not
var winnow = function( elements, qualifier, keep ) {
	if ( jQuery.isFunction( qualifier ) ) {
		return jQuery.grep(elements, function( elem, i ) {
			return !!qualifier.call( elem, i, elem ) === keep;
		});

	} else if ( qualifier.nodeType ) {
		return jQuery.grep(elements, function( elem, i ) {
			return (elem === qualifier) === keep;
		});

	} else if ( typeof qualifier === "string" ) {
		var filtered = jQuery.grep(elements, function( elem ) {
			return elem.nodeType === 1;
		});

		if ( isSimple.test( qualifier ) ) {
			return jQuery.filter(qualifier, filtered, !keep);
		} else {
			qualifier = jQuery.filter( qualifier, filtered );
		}
	}

	return jQuery.grep(elements, function( elem, i ) {
		return (jQuery.inArray( elem, qualifier ) >= 0) === keep;
	});
};

jQuery.fn.extend({
	find: function( selector ) {
		var ret = this.pushStack( "", "find", selector ), length = 0;

		for ( var i = 0, l = this.length; i < l; i++ ) {
			length = ret.length;
			jQuery.find( selector, this[i], ret );

			if ( i > 0 ) {
				// Make sure that the results are unique
				for ( var n = length; n < ret.length; n++ ) {
					for ( var r = 0; r < length; r++ ) {
						if ( ret[r] === ret[n] ) {
							ret.splice(n--, 1);
							break;
						}
					}
				}
			}
		}

		return ret;
	},

	has: function( target ) {
		var targets = jQuery( target );
		return this.filter(function() {
			for ( var i = 0, l = targets.length; i < l; i++ ) {
				if ( jQuery.contains( this, targets[i] ) ) {
					return true;
				}
			}
		});
	},

	not: function( selector ) {
		return this.pushStack( winnow(this, selector, false), "not", selector);
	},

	filter: function( selector ) {
		return this.pushStack( winnow(this, selector, true), "filter", selector );
	},
	
	is: function( selector ) {
		return !!selector && jQuery.filter( selector, this ).length > 0;
	},

	closest: function( selectors, context ) {
		if ( jQuery.isArray( selectors ) ) {
			var ret = [], cur = this[0], match, matches = {}, selector;

			if ( cur && selectors.length ) {
				for ( var i = 0, l = selectors.length; i < l; i++ ) {
					selector = selectors[i];

					if ( !matches[selector] ) {
						matches[selector] = jQuery.expr.match.POS.test( selector ) ? 
							jQuery( selector, context || this.context ) :
							selector;
					}
				}

				while ( cur && cur.ownerDocument && cur !== context ) {
					for ( selector in matches ) {
						match = matches[selector];

						if ( match.jquery ? match.index(cur) > -1 : jQuery(cur).is(match) ) {
							ret.push({ selector: selector, elem: cur });
							delete matches[selector];
						}
					}
					cur = cur.parentNode;
				}
			}

			return ret;
		}

		var pos = jQuery.expr.match.POS.test( selectors ) ? 
			jQuery( selectors, context || this.context ) : null;

		return this.map(function( i, cur ) {
			while ( cur && cur.ownerDocument && cur !== context ) {
				if ( pos ? pos.index(cur) > -1 : jQuery(cur).is(selectors) ) {
					return cur;
				}
				cur = cur.parentNode;
			}
			return null;
		});
	},
	
	// Determine the position of an element within
	// the matched set of elements
	index: function( elem ) {
		if ( !elem || typeof elem === "string" ) {
			return jQuery.inArray( this[0],
				// If it receives a string, the selector is used
				// If it receives nothing, the siblings are used
				elem ? jQuery( elem ) : this.parent().children() );
		}
		// Locate the position of the desired element
		return jQuery.inArray(
			// If it receives a jQuery object, the first element is used
			elem.jquery ? elem[0] : elem, this );
	},

	add: function( selector, context ) {
		var set = typeof selector === "string" ?
				jQuery( selector, context || this.context ) :
				jQuery.makeArray( selector ),
			all = jQuery.merge( this.get(), set );

		return this.pushStack( isDisconnected( set[0] ) || isDisconnected( all[0] ) ?
			all :
			jQuery.unique( all ) );
	},

	andSelf: function() {
		return this.add( this.prevObject );
	}
});

// A painfully simple check to see if an element is disconnected
// from a document (should be improved, where feasible).
function isDisconnected( node ) {
	return !node || !node.parentNode || node.parentNode.nodeType === 11;
}

jQuery.each({
	parent: function( elem ) {
		var parent = elem.parentNode;
		return parent && parent.nodeType !== 11 ? parent : null;
	},
	parents: function( elem ) {
		return jQuery.dir( elem, "parentNode" );
	},
	parentsUntil: function( elem, i, until ) {
		return jQuery.dir( elem, "parentNode", until );
	},
	next: function( elem ) {
		return jQuery.nth( elem, 2, "nextSibling" );
	},
	prev: function( elem ) {
		return jQuery.nth( elem, 2, "previousSibling" );
	},
	nextAll: function( elem ) {
		return jQuery.dir( elem, "nextSibling" );
	},
	prevAll: function( elem ) {
		return jQuery.dir( elem, "previousSibling" );
	},
	nextUntil: function( elem, i, until ) {
		return jQuery.dir( elem, "nextSibling", until );
	},
	prevUntil: function( elem, i, until ) {
		return jQuery.dir( elem, "previousSibling", until );
	},
	siblings: function( elem ) {
		return jQuery.sibling( elem.parentNode.firstChild, elem );
	},
	children: function( elem ) {
		return jQuery.sibling( elem.firstChild );
	},
	contents: function( elem ) {
		return jQuery.nodeName( elem, "iframe" ) ?
			elem.contentDocument || elem.contentWindow.document :
			jQuery.makeArray( elem.childNodes );
	}
}, function( name, fn ) {
	jQuery.fn[ name ] = function( until, selector ) {
		var ret = jQuery.map( this, fn, until );
		
		if ( !runtil.test( name ) ) {
			selector = until;
		}

		if ( selector && typeof selector === "string" ) {
			ret = jQuery.filter( selector, ret );
		}

		ret = this.length > 1 ? jQuery.unique( ret ) : ret;

		if ( (this.length > 1 || rmultiselector.test( selector )) && rparentsprev.test( name ) ) {
			ret = ret.reverse();
		}

		return this.pushStack( ret, name, slice.call(arguments).join(",") );
	};
});

jQuery.extend({
	filter: function( expr, elems, not ) {
		if ( not ) {
			expr = ":not(" + expr + ")";
		}

		return jQuery.find.matches(expr, elems);
	},
	
	dir: function( elem, dir, until ) {
		var matched = [], cur = elem[dir];
		while ( cur && cur.nodeType !== 9 && (until === undefined || cur.nodeType !== 1 || !jQuery( cur ).is( until )) ) {
			if ( cur.nodeType === 1 ) {
				matched.push( cur );
			}
			cur = cur[dir];
		}
		return matched;
	},

	nth: function( cur, result, dir, elem ) {
		result = result || 1;
		var num = 0;

		for ( ; cur; cur = cur[dir] ) {
			if ( cur.nodeType === 1 && ++num === result ) {
				break;
			}
		}

		return cur;
	},

	sibling: function( n, elem ) {
		var r = [];

		for ( ; n; n = n.nextSibling ) {
			if ( n.nodeType === 1 && n !== elem ) {
				r.push( n );
			}
		}

		return r;
	}
});
var rinlinejQuery = / jQuery\d+="(?:\d+|null)"/g,
	rleadingWhitespace = /^\s+/,
	rxhtmlTag = /(<([\w:]+)[^>]*?)\/>/g,
	rselfClosing = /^(?:area|br|col|embed|hr|img|input|link|meta|param)$/i,
	rtagName = /<([\w:]+)/,
	rtbody = /<tbody/i,
	rhtml = /<|&\w+;/,
	rchecked = /checked\s*(?:[^=]|=\s*.checked.)/i,  // checked="checked" or checked (html5)
	fcloseTag = function( all, front, tag ) {
		return rselfClosing.test( tag ) ?
			all :
			front + "></" + tag + ">";
	},
	wrapMap = {
		option: [ 1, "<select multiple='multiple'>", "</select>" ],
		legend: [ 1, "<fieldset>", "</fieldset>" ],
		thead: [ 1, "<table>", "</table>" ],
		tr: [ 2, "<table><tbody>", "</tbody></table>" ],
		td: [ 3, "<table><tbody><tr>", "</tr></tbody></table>" ],
		col: [ 2, "<table><tbody></tbody><colgroup>", "</colgroup></table>" ],
		area: [ 1, "<map>", "</map>" ],
		_default: [ 0, "", "" ]
	};

wrapMap.optgroup = wrapMap.option;
wrapMap.tbody = wrapMap.tfoot = wrapMap.colgroup = wrapMap.caption = wrapMap.thead;
wrapMap.th = wrapMap.td;

// IE can't serialize <link> and <script> tags normally
if ( !jQuery.support.htmlSerialize ) {
	wrapMap._default = [ 1, "div<div>", "</div>" ];
}

jQuery.fn.extend({
	text: function( text ) {
		if ( jQuery.isFunction(text) ) {
			return this.each(function(i) {
				var self = jQuery(this);
				self.text( text.call(this, i, self.text()) );
			});
		}

		if ( typeof text !== "object" && text !== undefined ) {
			return this.empty().append( (this[0] && this[0].ownerDocument || document).createTextNode( text ) );
		}

		return jQuery.getText( this );
	},

	wrapAll: function( html ) {
		if ( jQuery.isFunction( html ) ) {
			return this.each(function(i) {
				jQuery(this).wrapAll( html.call(this, i) );
			});
		}

		if ( this[0] ) {
			// The elements to wrap the target around
			var wrap = jQuery( html, this[0].ownerDocument ).eq(0).clone(true);

			if ( this[0].parentNode ) {
				wrap.insertBefore( this[0] );
			}

			wrap.map(function() {
				var elem = this;

				while ( elem.firstChild && elem.firstChild.nodeType === 1 ) {
					elem = elem.firstChild;
				}

				return elem;
			}).append(this);
		}

		return this;
	},

	wrapInner: function( html ) {
		if ( jQuery.isFunction( html ) ) {
			return this.each(function(i) {
				jQuery(this).wrapInner( html.call(this, i) );
			});
		}

		return this.each(function() {
			var self = jQuery( this ), contents = self.contents();

			if ( contents.length ) {
				contents.wrapAll( html );

			} else {
				self.append( html );
			}
		});
	},

	wrap: function( html ) {
		return this.each(function() {
			jQuery( this ).wrapAll( html );
		});
	},

	unwrap: function() {
		return this.parent().each(function() {
			if ( !jQuery.nodeName( this, "body" ) ) {
				jQuery( this ).replaceWith( this.childNodes );
			}
		}).end();
	},

	append: function() {
		return this.domManip(arguments, true, function( elem ) {
			if ( this.nodeType === 1 ) {
				this.appendChild( elem );
			}
		});
	},

	prepend: function() {
		return this.domManip(arguments, true, function( elem ) {
			if ( this.nodeType === 1 ) {
				this.insertBefore( elem, this.firstChild );
			}
		});
	},

	before: function() {
		if ( this[0] && this[0].parentNode ) {
			return this.domManip(arguments, false, function( elem ) {
				this.parentNode.insertBefore( elem, this );
			});
		} else if ( arguments.length ) {
			var set = jQuery(arguments[0]);
			set.push.apply( set, this.toArray() );
			return this.pushStack( set, "before", arguments );
		}
	},

	after: function() {
		if ( this[0] && this[0].parentNode ) {
			return this.domManip(arguments, false, function( elem ) {
				this.parentNode.insertBefore( elem, this.nextSibling );
			});
		} else if ( arguments.length ) {
			var set = this.pushStack( this, "after", arguments );
			set.push.apply( set, jQuery(arguments[0]).toArray() );
			return set;
		}
	},

	clone: function( events ) {
		// Do the clone
		var ret = this.map(function() {
			if ( !jQuery.support.noCloneEvent && !jQuery.isXMLDoc(this) ) {
				// IE copies events bound via attachEvent when
				// using cloneNode. Calling detachEvent on the
				// clone will also remove the events from the orignal
				// In order to get around this, we use innerHTML.
				// Unfortunately, this means some modifications to
				// attributes in IE that are actually only stored
				// as properties will not be copied (such as the
				// the name attribute on an input).
				var html = this.outerHTML, ownerDocument = this.ownerDocument;
				if ( !html ) {
					var div = ownerDocument.createElement("div");
					div.appendChild( this.cloneNode(true) );
					html = div.innerHTML;
				}

				return jQuery.clean([html.replace(rinlinejQuery, "")
					.replace(rleadingWhitespace, "")], ownerDocument)[0];
			} else {
				return this.cloneNode(true);
			}
		});

		// Copy the events from the original to the clone
		if ( events === true ) {
			cloneCopyEvent( this, ret );
			cloneCopyEvent( this.find("*"), ret.find("*") );
		}

		// Return the cloned set
		return ret;
	},

	html: function( value ) {
		if ( value === undefined ) {
			return this[0] && this[0].nodeType === 1 ?
				this[0].innerHTML.replace(rinlinejQuery, "") :
				null;

		// See if we can take a shortcut and just use innerHTML
		} else if ( typeof value === "string" && !/<script/i.test( value ) &&
			(jQuery.support.leadingWhitespace || !rleadingWhitespace.test( value )) &&
			!wrapMap[ (rtagName.exec( value ) || ["", ""])[1].toLowerCase() ] ) {

			value = value.replace(rxhtmlTag, fcloseTag);

			try {
				for ( var i = 0, l = this.length; i < l; i++ ) {
					// Remove element nodes and prevent memory leaks
					if ( this[i].nodeType === 1 ) {
						jQuery.cleanData( this[i].getElementsByTagName("*") );
						this[i].innerHTML = value;
					}
				}

			// If using innerHTML throws an exception, use the fallback method
			} catch(e) {
				this.empty().append( value );
			}

		} else if ( jQuery.isFunction( value ) ) {
			this.each(function(i){
				var self = jQuery(this), old = self.html();
				self.empty().append(function(){
					return value.call( this, i, old );
				});
			});

		} else {
			this.empty().append( value );
		}

		return this;
	},

	replaceWith: function( value ) {
		if ( this[0] && this[0].parentNode ) {
			// Make sure that the elements are removed from the DOM before they are inserted
			// this can help fix replacing a parent with child elements
			if ( !jQuery.isFunction( value ) ) {
				value = jQuery( value ).detach();

			} else {
				return this.each(function(i) {
					var self = jQuery(this), old = self.html();
					self.replaceWith( value.call( this, i, old ) );
				});
			}

			return this.each(function() {
				var next = this.nextSibling, parent = this.parentNode;

				jQuery(this).remove();

				if ( next ) {
					jQuery(next).before( value );
				} else {
					jQuery(parent).append( value );
				}
			});
		} else {
			return this.pushStack( jQuery(jQuery.isFunction(value) ? value() : value), "replaceWith", value );
		}
	},

	detach: function( selector ) {
		return this.remove( selector, true );
	},

	domManip: function( args, table, callback ) {
		var results, first, value = args[0], scripts = [];

		// We can't cloneNode fragments that contain checked, in WebKit
		if ( !jQuery.support.checkClone && arguments.length === 3 && typeof value === "string" && rchecked.test( value ) ) {
			return this.each(function() {
				jQuery(this).domManip( args, table, callback, true );
			});
		}

		if ( jQuery.isFunction(value) ) {
			return this.each(function(i) {
				var self = jQuery(this);
				args[0] = value.call(this, i, table ? self.html() : undefined);
				self.domManip( args, table, callback );
			});
		}

		if ( this[0] ) {
			// If we're in a fragment, just use that instead of building a new one
			if ( args[0] && args[0].parentNode && args[0].parentNode.nodeType === 11 ) {
				results = { fragment: args[0].parentNode };
			} else {
				results = buildFragment( args, this, scripts );
			}

			first = results.fragment.firstChild;

			if ( first ) {
				table = table && jQuery.nodeName( first, "tr" );

				for ( var i = 0, l = this.length; i < l; i++ ) {
					callback.call(
						table ?
							root(this[i], first) :
							this[i],
						results.cacheable || this.length > 1 || i > 0 ?
							results.fragment.cloneNode(true) :
							results.fragment
					);
				}
			}

			if ( scripts ) {
				jQuery.each( scripts, evalScript );
			}
		}

		return this;

		function root( elem, cur ) {
			return jQuery.nodeName(elem, "table") ?
				(elem.getElementsByTagName("tbody")[0] ||
				elem.appendChild(elem.ownerDocument.createElement("tbody"))) :
				elem;
		}
	}
});

function cloneCopyEvent(orig, ret) {
	var i = 0;

	ret.each(function() {
		if ( this.nodeName !== (orig[i] && orig[i].nodeName) ) {
			return;
		}

		var oldData = jQuery.data( orig[i++] ), curData = jQuery.data( this, oldData ), events = oldData && oldData.events;

		if ( events ) {
			delete curData.handle;
			curData.events = {};

			for ( var type in events ) {
				for ( var handler in events[ type ] ) {
					jQuery.event.add( this, type, events[ type ][ handler ], events[ type ][ handler ].data );
				}
			}
		}
	});
}

function buildFragment( args, nodes, scripts ) {
	var fragment, cacheable, cacheresults, doc;

	// webkit does not clone 'checked' attribute of radio inputs on cloneNode, so don't cache if string has a checked
	if ( args.length === 1 && typeof args[0] === "string" && args[0].length < 512 && args[0].indexOf("<option") < 0 && (jQuery.support.checkClone || !rchecked.test( args[0] )) ) {
		cacheable = true;
		cacheresults = jQuery.fragments[ args[0] ];
		if ( cacheresults ) {
			if ( cacheresults !== 1 ) {
				fragment = cacheresults;
			}
		}
	}

	if ( !fragment ) {
		doc = (nodes && nodes[0] ? nodes[0].ownerDocument || nodes[0] : document);
		fragment = doc.createDocumentFragment();
		jQuery.clean( args, doc, fragment, scripts );
	}

	if ( cacheable ) {
		jQuery.fragments[ args[0] ] = cacheresults ? fragment : 1;
	}

	return { fragment: fragment, cacheable: cacheable };
}

jQuery.fragments = {};

jQuery.each({
	appendTo: "append",
	prependTo: "prepend",
	insertBefore: "before",
	insertAfter: "after",
	replaceAll: "replaceWith"
}, function( name, original ) {
	jQuery.fn[ name ] = function( selector ) {
		var ret = [], insert = jQuery( selector );

		for ( var i = 0, l = insert.length; i < l; i++ ) {
			var elems = (i > 0 ? this.clone(true) : this).get();
			jQuery.fn[ original ].apply( jQuery(insert[i]), elems );
			ret = ret.concat( elems );
		}
		return this.pushStack( ret, name, insert.selector );
	};
});

jQuery.each({
	// keepData is for internal use only--do not document
	remove: function( selector, keepData ) {
		if ( !selector || jQuery.filter( selector, [ this ] ).length ) {
			if ( !keepData && this.nodeType === 1 ) {
				jQuery.cleanData( this.getElementsByTagName("*") );
				jQuery.cleanData( [ this ] );
			}

			if ( this.parentNode ) {
				 this.parentNode.removeChild( this );
			}
		}
	},

	empty: function() {
		// Remove element nodes and prevent memory leaks
		if ( this.nodeType === 1 ) {
			jQuery.cleanData( this.getElementsByTagName("*") );
		}

		// Remove any remaining nodes
		while ( this.firstChild ) {
			this.removeChild( this.firstChild );
		}
	}
}, function( name, fn ) {
	jQuery.fn[ name ] = function() {
		return this.each( fn, arguments );
	};
});

jQuery.extend({
	clean: function( elems, context, fragment, scripts ) {
		context = context || document;

		// !context.createElement fails in IE with an error but returns typeof 'object'
		if ( typeof context.createElement === "undefined" ) {
			context = context.ownerDocument || context[0] && context[0].ownerDocument || document;
		}

		var ret = [];

		jQuery.each(elems, function( i, elem ) {
			if ( typeof elem === "number" ) {
				elem += "";
			}

			if ( !elem ) {
				return;
			}

			// Convert html string into DOM nodes
			if ( typeof elem === "string" && !rhtml.test( elem ) ) {
				elem = context.createTextNode( elem );

			} else if ( typeof elem === "string" ) {
				// Fix "XHTML"-style tags in all browsers
				elem = elem.replace(rxhtmlTag, fcloseTag);

				// Trim whitespace, otherwise indexOf won't work as expected
				var tag = (rtagName.exec( elem ) || ["", ""])[1].toLowerCase(),
					wrap = wrapMap[ tag ] || wrapMap._default,
					depth = wrap[0],
					div = context.createElement("div");

				// Go to html and back, then peel off extra wrappers
				div.innerHTML = wrap[1] + elem + wrap[2];

				// Move to the right depth
				while ( depth-- ) {
					div = div.lastChild;
				}

				// Remove IE's autoinserted <tbody> from table fragments
				if ( !jQuery.support.tbody ) {

					// String was a <table>, *may* have spurious <tbody>
					var hasBody = rtbody.test(elem),
						tbody = tag === "table" && !hasBody ?
							div.firstChild && div.firstChild.childNodes :

							// String was a bare <thead> or <tfoot>
							wrap[1] === "<table>" && !hasBody ?
								div.childNodes :
								[];

					for ( var j = tbody.length - 1; j >= 0 ; --j ) {
						if ( jQuery.nodeName( tbody[ j ], "tbody" ) && !tbody[ j ].childNodes.length ) {
							tbody[ j ].parentNode.removeChild( tbody[ j ] );
						}
					}

				}

				// IE completely kills leading whitespace when innerHTML is used
				if ( !jQuery.support.leadingWhitespace && rleadingWhitespace.test( elem ) ) {
					div.insertBefore( context.createTextNode( rleadingWhitespace.exec(elem)[0] ), div.firstChild );
				}

				elem = jQuery.makeArray( div.childNodes );
			}

			if ( elem.nodeType ) {
				ret.push( elem );
			} else {
				ret = jQuery.merge( ret, elem );
			}

		});

		if ( fragment ) {
			for ( var i = 0; ret[i]; i++ ) {
				if ( scripts && jQuery.nodeName( ret[i], "script" ) && (!ret[i].type || ret[i].type.toLowerCase() === "text/javascript") ) {
					scripts.push( ret[i].parentNode ? ret[i].parentNode.removeChild( ret[i] ) : ret[i] );
				} else {
					if ( ret[i].nodeType === 1 ) {
						ret.splice.apply( ret, [i + 1, 0].concat(jQuery.makeArray(ret[i].getElementsByTagName("script"))) );
					}
					fragment.appendChild( ret[i] );
				}
			}
		}

		return ret;
	},
	
	cleanData: function( elems ) {
		for ( var i = 0, elem, id; (elem = elems[i]) != null; i++ ) {
			jQuery.event.remove( elem );
			jQuery.removeData( elem );
		}
	}
});
// exclude the following css properties to add px
var rexclude = /z-?index|font-?weight|opacity|zoom|line-?height/i,
	ralpha = /alpha\([^)]*\)/,
	ropacity = /opacity=([^)]*)/,
	rfloat = /float/i,
	rdashAlpha = /-([a-z])/ig,
	rupper = /([A-Z])/g,
	rnumpx = /^-?\d+(?:px)?$/i,
	rnum = /^-?\d/,

	cssShow = { position: "absolute", visibility: "hidden", display:"block" },
	cssWidth = [ "Left", "Right" ],
	cssHeight = [ "Top", "Bottom" ],

	// cache check for defaultView.getComputedStyle
	getComputedStyle = document.defaultView && document.defaultView.getComputedStyle,
	// normalize float css property
	styleFloat = jQuery.support.cssFloat ? "cssFloat" : "styleFloat",
	fcamelCase = function( all, letter ) {
		return letter.toUpperCase();
	};

jQuery.fn.css = function( name, value ) {
	return access( this, name, value, true, function( elem, name, value ) {
		if ( value === undefined ) {
			return jQuery.curCSS( elem, name );
		}
		
		if ( typeof value === "number" && !rexclude.test(name) ) {
			value += "px";
		}

		jQuery.style( elem, name, value );
	});
};

jQuery.extend({
	style: function( elem, name, value ) {
		// don't set styles on text and comment nodes
		if ( !elem || elem.nodeType === 3 || elem.nodeType === 8 ) {
			return undefined;
		}

		// ignore negative width and height values #1599
		if ( (name === "width" || name === "height") && parseFloat(value) < 0 ) {
			value = undefined;
		}

		var style = elem.style || elem, set = value !== undefined;

		// IE uses filters for opacity
		if ( !jQuery.support.opacity && name === "opacity" ) {
			if ( set ) {
				// IE has trouble with opacity if it does not have layout
				// Force it by setting the zoom level
				style.zoom = 1;

				// Set the alpha filter to set the opacity
				var opacity = parseInt( value, 10 ) + "" === "NaN" ? "" : "alpha(opacity=" + value * 100 + ")";
				var filter = style.filter || jQuery.curCSS( elem, "filter" ) || "";
				style.filter = ralpha.test(filter) ? filter.replace(ralpha, opacity) : opacity;
			}

			return style.filter && style.filter.indexOf("opacity=") >= 0 ?
				(parseFloat( ropacity.exec(style.filter)[1] ) / 100) + "":
				"";
		}

		// Make sure we're using the right name for getting the float value
		if ( rfloat.test( name ) ) {
			name = styleFloat;
		}

		name = name.replace(rdashAlpha, fcamelCase);

		if ( set ) {
			style[ name ] = value;
		}

		return style[ name ];
	},

	css: function( elem, name, force, extra ) {
		if ( name === "width" || name === "height" ) {
			var val, props = cssShow, which = name === "width" ? cssWidth : cssHeight;

			function getWH() {
				val = name === "width" ? elem.offsetWidth : elem.offsetHeight;

				if ( extra === "border" ) {
					return;
				}

				jQuery.each( which, function() {
					if ( !extra ) {
						val -= parseFloat(jQuery.curCSS( elem, "padding" + this, true)) || 0;
					}

					if ( extra === "margin" ) {
						val += parseFloat(jQuery.curCSS( elem, "margin" + this, true)) || 0;
					} else {
						val -= parseFloat(jQuery.curCSS( elem, "border" + this + "Width", true)) || 0;
					}
				});
			}

			if ( elem.offsetWidth !== 0 ) {
				getWH();
			} else {
				jQuery.swap( elem, props, getWH );
			}

			return Math.max(0, Math.round(val));
		}

		return jQuery.curCSS( elem, name, force );
	},

	curCSS: function( elem, name, force ) {
		var ret, style = elem.style, filter;

		// IE uses filters for opacity
		if ( !jQuery.support.opacity && name === "opacity" && elem.currentStyle ) {
			ret = ropacity.test(elem.currentStyle.filter || "") ?
				(parseFloat(RegExp.$1) / 100) + "" :
				"";

			return ret === "" ?
				"1" :
				ret;
		}

		// Make sure we're using the right name for getting the float value
		if ( rfloat.test( name ) ) {
			name = styleFloat;
		}

		if ( !force && style && style[ name ] ) {
			ret = style[ name ];

		} else if ( getComputedStyle ) {

			// Only "float" is needed here
			if ( rfloat.test( name ) ) {
				name = "float";
			}

			name = name.replace( rupper, "-$1" ).toLowerCase();

			var defaultView = elem.ownerDocument.defaultView;

			if ( !defaultView ) {
				return null;
			}

			var computedStyle = defaultView.getComputedStyle( elem, null );

			if ( computedStyle ) {
				ret = computedStyle.getPropertyValue( name );
			}

			// We should always get a number back from opacity
			if ( name === "opacity" && ret === "" ) {
				ret = "1";
			}

		} else if ( elem.currentStyle ) {
			var camelCase = name.replace(rdashAlpha, fcamelCase);

			ret = elem.currentStyle[ name ] || elem.currentStyle[ camelCase ];

			// From the awesome hack by Dean Edwards
			// http://erik.eae.net/archives/2007/07/27/18.54.15/#comment-102291

			// If we're not dealing with a regular pixel number
			// but a number that has a weird ending, we need to convert it to pixels
			if ( !rnumpx.test( ret ) && rnum.test( ret ) ) {
				// Remember the original values
				var left = style.left, rsLeft = elem.runtimeStyle.left;

				// Put in the new values to get a computed value out
				elem.runtimeStyle.left = elem.currentStyle.left;
				style.left = camelCase === "fontSize" ? "1em" : (ret || 0);
				ret = style.pixelLeft + "px";

				// Revert the changed values
				style.left = left;
				elem.runtimeStyle.left = rsLeft;
			}
		}

		return ret;
	},

	// A method for quickly swapping in/out CSS properties to get correct calculations
	swap: function( elem, options, callback ) {
		var old = {};

		// Remember the old values, and insert the new ones
		for ( var name in options ) {
			old[ name ] = elem.style[ name ];
			elem.style[ name ] = options[ name ];
		}

		callback.call( elem );

		// Revert the old values
		for ( var name in options ) {
			elem.style[ name ] = old[ name ];
		}
	}
});

if ( jQuery.expr && jQuery.expr.filters ) {
	jQuery.expr.filters.hidden = function( elem ) {
		var width = elem.offsetWidth, height = elem.offsetHeight,
			skip = elem.nodeName.toLowerCase() === "tr";

		return width === 0 && height === 0 && !skip ?
			true :
			width > 0 && height > 0 && !skip ?
				false :
				jQuery.curCSS(elem, "display") === "none";
	};

	jQuery.expr.filters.visible = function( elem ) {
		return !jQuery.expr.filters.hidden( elem );
	};
}
var jsc = now(),
	rscript = /<script(.|\s)*?\/script>/gi,
	rselectTextarea = /select|textarea/i,
	rinput = /color|date|datetime|email|hidden|month|number|password|range|search|tel|text|time|url|week/i,
	jsre = /=\?(&|$)/,
	rquery = /\?/,
	rts = /(\?|&)_=.*?(&|$)/,
	rurl = /^(\w+:)?\/\/([^\/?#]+)/,
	r20 = /%20/g;

jQuery.fn.extend({
	// Keep a copy of the old load
	_load: jQuery.fn.load,

	load: function( url, params, callback ) {
		if ( typeof url !== "string" ) {
			return this._load( url );

		// Don't do a request if no elements are being requested
		} else if ( !this.length ) {
			return this;
		}

		var off = url.indexOf(" ");
		if ( off >= 0 ) {
			var selector = url.slice(off, url.length);
			url = url.slice(0, off);
		}

		// Default to a GET request
		var type = "GET";

		// If the second parameter was provided
		if ( params ) {
			// If it's a function
			if ( jQuery.isFunction( params ) ) {
				// We assume that it's the callback
				callback = params;
				params = null;

			// Otherwise, build a param string
			} else if ( typeof params === "object" ) {
				params = jQuery.param( params, jQuery.ajaxSettings.traditional );
				type = "POST";
			}
		}

		var self = this;

		// Request the remote document
		jQuery.ajax({
			url: url,
			type: type,
			dataType: "html",
			data: params,
			complete: function( res, status ) {
				// If successful, inject the HTML into all the matched elements
				if ( status === "success" || status === "notmodified" ) {
					// See if a selector was specified
					self.html( selector ?
						// Create a dummy div to hold the results
						jQuery("<div />")
							// inject the contents of the document in, removing the scripts
							// to avoid any 'Permission Denied' errors in IE
							.append(res.responseText.replace(rscript, ""))

							// Locate the specified elements
							.find(selector) :

						// If not, just inject the full result
						res.responseText );
				}

				if ( callback ) {
					self.each( callback, [res.responseText, status, res] );
				}
			}
		});

		return this;
	},

	serialize: function() {
		return jQuery.param(this.serializeArray());
	},
	serializeArray: function() {
		return this.map(function() {
			return this.elements ? jQuery.makeArray(this.elements) : this;
		})
		.filter(function() {
			return this.name && !this.disabled &&
				(this.checked || rselectTextarea.test(this.nodeName) ||
					rinput.test(this.type));
		})
		.map(function( i, elem ) {
			var val = jQuery(this).val();

			return val == null ?
				null :
				jQuery.isArray(val) ?
					jQuery.map( val, function( val, i ) {
						return { name: elem.name, value: val };
					}) :
					{ name: elem.name, value: val };
		}).get();
	}
});

// Attach a bunch of functions for handling common AJAX events
jQuery.each( "ajaxStart ajaxStop ajaxComplete ajaxError ajaxSuccess ajaxSend".split(" "), function( i, o ) {
	jQuery.fn[o] = function( f ) {
		return this.bind(o, f);
	};
});

jQuery.extend({

	get: function( url, data, callback, type ) {
		// shift arguments if data argument was omited
		if ( jQuery.isFunction( data ) ) {
			type = type || callback;
			callback = data;
			data = null;
		}

		return jQuery.ajax({
			type: "GET",
			url: url,
			data: data,
			success: callback,
			dataType: type
		});
	},

	getScript: function( url, callback ) {
		return jQuery.get(url, null, callback, "script");
	},

	getJSON: function( url, data, callback ) {
		return jQuery.get(url, data, callback, "json");
	},

	post: function( url, data, callback, type ) {
		// shift arguments if data argument was omited
		if ( jQuery.isFunction( data ) ) {
			type = type || callback;
			callback = data;
			data = {};
		}

		return jQuery.ajax({
			type: "POST",
			url: url,
			data: data,
			success: callback,
			dataType: type
		});
	},

	ajaxSetup: function( settings ) {
		jQuery.extend( jQuery.ajaxSettings, settings );
	},

	ajaxSettings: {
		url: location.href,
		global: true,
		type: "GET",
		contentType: "application/x-www-form-urlencoded",
		processData: true,
		async: true,
		/*
		timeout: 0,
		data: null,
		username: null,
		password: null,
		traditional: false,
		*/
		// Create the request object; Microsoft failed to properly
		// implement the XMLHttpRequest in IE7 (can't request local files),
		// so we use the ActiveXObject when it is available
		// This function can be overriden by calling jQuery.ajaxSetup
		xhr: window.XMLHttpRequest && (window.location.protocol !== "file:" || !window.ActiveXObject) ?
			function() {
				return new window.XMLHttpRequest();
			} :
			function() {
				try {
					return new window.ActiveXObject("Microsoft.XMLHTTP");
				} catch(e) {}
			},
		accepts: {
			xml: "application/xml, text/xml",
			html: "text/html",
			script: "text/javascript, application/javascript",
			json: "application/json, text/javascript",
			text: "text/plain",
			_default: "*/*"
		}
	},

	// Last-Modified header cache for next request
	lastModified: {},
	etag: {},

	ajax: function( origSettings ) {
		var s = jQuery.extend(true, {}, jQuery.ajaxSettings, origSettings);
		
		var jsonp, status, data,
			callbackContext = origSettings && origSettings.context || s,
			type = s.type.toUpperCase();

		// convert data if not already a string
		if ( s.data && s.processData && typeof s.data !== "string" ) {
			s.data = jQuery.param( s.data, s.traditional );
		}

		// Handle JSONP Parameter Callbacks
		if ( s.dataType === "jsonp" ) {
			if ( type === "GET" ) {
				if ( !jsre.test( s.url ) ) {
					s.url += (rquery.test( s.url ) ? "&" : "?") + (s.jsonp || "callback") + "=?";
				}
			} else if ( !s.data || !jsre.test(s.data) ) {
				s.data = (s.data ? s.data + "&" : "") + (s.jsonp || "callback") + "=?";
			}
			s.dataType = "json";
		}

		// Build temporary JSONP function
		if ( s.dataType === "json" && (s.data && jsre.test(s.data) || jsre.test(s.url)) ) {
			jsonp = s.jsonpCallback || ("jsonp" + jsc++);

			// Replace the =? sequence both in the query string and the data
			if ( s.data ) {
				s.data = (s.data + "").replace(jsre, "=" + jsonp + "$1");
			}

			s.url = s.url.replace(jsre, "=" + jsonp + "$1");

			// We need to make sure
			// that a JSONP style response is executed properly
			s.dataType = "script";

			// Handle JSONP-style loading
			window[ jsonp ] = window[ jsonp ] || function( tmp ) {
				data = tmp;
				success();
				complete();
				// Garbage collect
				window[ jsonp ] = undefined;

				try {
					delete window[ jsonp ];
				} catch(e) {}

				if ( head ) {
					head.removeChild( script );
				}
			};
		}

		if ( s.dataType === "script" && s.cache === null ) {
			s.cache = false;
		}

		if ( s.cache === false && type === "GET" ) {
			var ts = now();

			// try replacing _= if it is there
			var ret = s.url.replace(rts, "$1_=" + ts + "$2");

			// if nothing was replaced, add timestamp to the end
			s.url = ret + ((ret === s.url) ? (rquery.test(s.url) ? "&" : "?") + "_=" + ts : "");
		}

		// If data is available, append data to url for get requests
		if ( s.data && type === "GET" ) {
			s.url += (rquery.test(s.url) ? "&" : "?") + s.data;
		}

		// Watch for a new set of requests
		if ( s.global && ! jQuery.active++ ) {
			jQuery.event.trigger( "ajaxStart" );
		}

		// Matches an absolute URL, and saves the domain
		var parts = rurl.exec( s.url ),
			remote = parts && (parts[1] && parts[1] !== location.protocol || parts[2] !== location.host);

		// If we're requesting a remote document
		// and trying to load JSON or Script with a GET
		if ( s.dataType === "script" && type === "GET" && remote ) {
			var head = document.getElementsByTagName("head")[0] || document.documentElement;
			var script = document.createElement("script");
			script.src = s.url;
			if ( s.scriptCharset ) {
				script.charset = s.scriptCharset;
			}

			// Handle Script loading
			if ( !jsonp ) {
				var done = false;

				// Attach handlers for all browsers
				script.onload = script.onreadystatechange = function() {
					if ( !done && (!this.readyState ||
							this.readyState === "loaded" || this.readyState === "complete") ) {
						done = true;
						success();
						complete();

						// Handle memory leak in IE
						script.onload = script.onreadystatechange = null;
						if ( head && script.parentNode ) {
							head.removeChild( script );
						}
					}
				};
			}

			// Use insertBefore instead of appendChild  to circumvent an IE6 bug.
			// This arises when a base node is used (#2709 and #4378).
			head.insertBefore( script, head.firstChild );

			// We handle everything using the script element injection
			return undefined;
		}

		var requestDone = false;

		// Create the request object
		var xhr = s.xhr();

		if ( !xhr ) {
			return;
		}

		// Open the socket
		// Passing null username, generates a login popup on Opera (#2865)
		if ( s.username ) {
			xhr.open(type, s.url, s.async, s.username, s.password);
		} else {
			xhr.open(type, s.url, s.async);
		}

		// Need an extra try/catch for cross domain requests in Firefox 3
		try {
			// Set the correct header, if data is being sent
			if ( s.data || origSettings && origSettings.contentType ) {
				xhr.setRequestHeader("Content-Type", s.contentType);
			}

			// Set the If-Modified-Since and/or If-None-Match header, if in ifModified mode.
			if ( s.ifModified ) {
				if ( jQuery.lastModified[s.url] ) {
					xhr.setRequestHeader("If-Modified-Since", jQuery.lastModified[s.url]);
				}

				if ( jQuery.etag[s.url] ) {
					xhr.setRequestHeader("If-None-Match", jQuery.etag[s.url]);
				}
			}

			// Set header so the called script knows that it's an XMLHttpRequest
			// Only send the header if it's not a remote XHR
			if ( !remote ) {
				xhr.setRequestHeader("X-Requested-With", "XMLHttpRequest");
			}

			// Set the Accepts header for the server, depending on the dataType
			xhr.setRequestHeader("Accept", s.dataType && s.accepts[ s.dataType ] ?
				s.accepts[ s.dataType ] + ", */*" :
				s.accepts._default );
		} catch(e) {}

		// Allow custom headers/mimetypes and early abort
		if ( s.beforeSend && s.beforeSend.call(callbackContext, xhr, s) === false ) {
			// Handle the global AJAX counter
			if ( s.global && ! --jQuery.active ) {
				jQuery.event.trigger( "ajaxStop" );
			}

			// close opended socket
			xhr.abort();
			return false;
		}

		if ( s.global ) {
			trigger("ajaxSend", [xhr, s]);
		}

		// Wait for a response to come back
		var onreadystatechange = xhr.onreadystatechange = function( isTimeout ) {
			// The request was aborted
			if ( !xhr || xhr.readyState === 0 || isTimeout === "abort" ) {
				// Opera doesn't call onreadystatechange before this point
				// so we simulate the call
				if ( !requestDone ) {
					complete();
				}

				requestDone = true;
				if ( xhr ) {
					xhr.onreadystatechange = jQuery.noop;
				}

			// The transfer is complete and the data is available, or the request timed out
			} else if ( !requestDone && xhr && (xhr.readyState === 4 || isTimeout === "timeout") ) {
				requestDone = true;
				xhr.onreadystatechange = jQuery.noop;

				status = isTimeout === "timeout" ?
					"timeout" :
					!jQuery.httpSuccess( xhr ) ?
						"error" :
						s.ifModified && jQuery.httpNotModified( xhr, s.url ) ?
							"notmodified" :
							"success";

				var errMsg;

				if ( status === "success" ) {
					// Watch for, and catch, XML document parse errors
					try {
						// process the data (runs the xml through httpData regardless of callback)
						data = jQuery.httpData( xhr, s.dataType, s );
					} catch(err) {
						status = "parsererror";
						errMsg = err;
					}
				}

				// Make sure that the request was successful or notmodified
				if ( status === "success" || status === "notmodified" ) {
					// JSONP handles its own success callback
					if ( !jsonp ) {
						success();
					}
				} else {
					jQuery.handleError(s, xhr, status, errMsg);
				}

				// Fire the complete handlers
				complete();

				if ( isTimeout === "timeout" ) {
					xhr.abort();
				}

				// Stop memory leaks
				if ( s.async ) {
					xhr = null;
				}
			}
		};

		// Override the abort handler, if we can (IE doesn't allow it, but that's OK)
		// Opera doesn't fire onreadystatechange at all on abort
		try {
			var oldAbort = xhr.abort;
			xhr.abort = function() {
				if ( xhr ) {
					oldAbort.call( xhr );
				}

				onreadystatechange( "abort" );
			};
		} catch(e) { }

		// Timeout checker
		if ( s.async && s.timeout > 0 ) {
			setTimeout(function() {
				// Check to see if the request is still happening
				if ( xhr && !requestDone ) {
					onreadystatechange( "timeout" );
				}
			}, s.timeout);
		}

		// Send the data
		try {
			xhr.send( type === "POST" || type === "PUT" || type === "DELETE" ? s.data : null );
		} catch(e) {
			jQuery.handleError(s, xhr, null, e);
			// Fire the complete handlers
			complete();
		}

		// firefox 1.5 doesn't fire statechange for sync requests
		if ( !s.async ) {
			onreadystatechange();
		}

		function success() {
			// If a local callback was specified, fire it and pass it the data
			if ( s.success ) {
				s.success.call( callbackContext, data, status, xhr );
			}

			// Fire the global callback
			if ( s.global ) {
				trigger( "ajaxSuccess", [xhr, s] );
			}
		}

		function complete() {
			// Process result
			if ( s.complete ) {
				s.complete.call( callbackContext, xhr, status);
			}

			// The request was completed
			if ( s.global ) {
				trigger( "ajaxComplete", [xhr, s] );
			}

			// Handle the global AJAX counter
			if ( s.global && ! --jQuery.active ) {
				jQuery.event.trigger( "ajaxStop" );
			}
		}
		
		function trigger(type, args) {
			(s.context ? jQuery(s.context) : jQuery.event).trigger(type, args);
		}

		// return XMLHttpRequest to allow aborting the request etc.
		return xhr;
	},

	handleError: function( s, xhr, status, e ) {
		// If a local callback was specified, fire it
		if ( s.error ) {
			s.error.call( s.context || s, xhr, status, e );
		}

		// Fire the global callback
		if ( s.global ) {
			(s.context ? jQuery(s.context) : jQuery.event).trigger( "ajaxError", [xhr, s, e] );
		}
	},

	// Counter for holding the number of active queries
	active: 0,

	// Determines if an XMLHttpRequest was successful or not
	httpSuccess: function( xhr ) {
		try {
			// IE error sometimes returns 1223 when it should be 204 so treat it as success, see #1450
			return !xhr.status && location.protocol === "file:" ||
				// Opera returns 0 when status is 304
				( xhr.status >= 200 && xhr.status < 300 ) ||
				xhr.status === 304 || xhr.status === 1223 || xhr.status === 0;
		} catch(e) {}

		return false;
	},

	// Determines if an XMLHttpRequest returns NotModified
	httpNotModified: function( xhr, url ) {
		var lastModified = xhr.getResponseHeader("Last-Modified"),
			etag = xhr.getResponseHeader("Etag");

		if ( lastModified ) {
			jQuery.lastModified[url] = lastModified;
		}

		if ( etag ) {
			jQuery.etag[url] = etag;
		}

		// Opera returns 0 when status is 304
		return xhr.status === 304 || xhr.status === 0;
	},

	httpData: function( xhr, type, s ) {
		var ct = xhr.getResponseHeader("content-type") || "",
			xml = type === "xml" || !type && ct.indexOf("xml") >= 0,
			data = xml ? xhr.responseXML : xhr.responseText;

		if ( xml && data.documentElement.nodeName === "parsererror" ) {
			jQuery.error( "parsererror" );
		}

		// Allow a pre-filtering function to sanitize the response
		// s is checked to keep backwards compatibility
		if ( s && s.dataFilter ) {
			data = s.dataFilter( data, type );
		}

		// The filter can actually parse the response
		if ( typeof data === "string" ) {
			// Get the JavaScript object, if JSON is used.
			if ( type === "json" || !type && ct.indexOf("json") >= 0 ) {
				data = jQuery.parseJSON( data );

			// If the type is "script", eval it in global context
			} else if ( type === "script" || !type && ct.indexOf("javascript") >= 0 ) {
				jQuery.globalEval( data );
			}
		}

		return data;
	},

	// Serialize an array of form elements or a set of
	// key/values into a query string
	param: function( a, traditional ) {
		var s = [];
		
		// Set traditional to true for jQuery <= 1.3.2 behavior.
		if ( traditional === undefined ) {
			traditional = jQuery.ajaxSettings.traditional;
		}
		
		// If an array was passed in, assume that it is an array of form elements.
		if ( jQuery.isArray(a) || a.jquery ) {
			// Serialize the form elements
			jQuery.each( a, function() {
				add( this.name, this.value );
			});
			
		} else {
			// If traditional, encode the "old" way (the way 1.3.2 or older
			// did it), otherwise encode params recursively.
			for ( var prefix in a ) {
				buildParams( prefix, a[prefix] );
			}
		}

		// Return the resulting serialization
		return s.join("&").replace(r20, "+");

		function buildParams( prefix, obj ) {
			if ( jQuery.isArray(obj) ) {
				// Serialize array item.
				jQuery.each( obj, function( i, v ) {
					if ( traditional ) {
						// Treat each array item as a scalar.
						add( prefix, v );
					} else {
						// If array item is non-scalar (array or object), encode its
						// numeric index to resolve deserialization ambiguity issues.
						// Note that rack (as of 1.0.0) can't currently deserialize
						// nested arrays properly, and attempting to do so may cause
						// a server error. Possible fixes are to modify rack's
						// deserialization algorithm or to provide an option or flag
						// to force array serialization to be shallow.
						buildParams( prefix + "[" + ( typeof v === "object" || jQuery.isArray(v) ? i : "" ) + "]", v );
					}
				});
					
			} else if ( !traditional && obj != null && typeof obj === "object" ) {
				// Serialize object item.
				jQuery.each( obj, function( k, v ) {
					buildParams( prefix + "[" + k + "]", v );
				});
					
			} else {
				// Serialize scalar item.
				add( prefix, obj );
			}
		}

		function add( key, value ) {
			// If value is a function, invoke it and return its value
			value = jQuery.isFunction(value) ? value() : value;
			s[ s.length ] = encodeURIComponent(key) + "=" + encodeURIComponent(value);
		}
	}
});
var elemdisplay = {},
	rfxtypes = /toggle|show|hide/,
	rfxnum = /^([+-]=)?([\d+-.]+)(.*)$/,
	timerId,
	fxAttrs = [
		// height animations
		[ "height", "marginTop", "marginBottom", "paddingTop", "paddingBottom" ],
		// width animations
		[ "width", "marginLeft", "marginRight", "paddingLeft", "paddingRight" ],
		// opacity animations
		[ "opacity" ]
	];

jQuery.fn.extend({
	show: function( speed, callback ) {
		if ( speed || speed === 0) {
			return this.animate( genFx("show", 3), speed, callback);

		} else {
			for ( var i = 0, l = this.length; i < l; i++ ) {
				var old = jQuery.data(this[i], "olddisplay");

				this[i].style.display = old || "";

				if ( jQuery.css(this[i], "display") === "none" ) {
					var nodeName = this[i].nodeName, display;

					if ( elemdisplay[ nodeName ] ) {
						display = elemdisplay[ nodeName ];

					} else {
						var elem = jQuery("<" + nodeName + " />").appendTo("body");

						display = elem.css("display");

						if ( display === "none" ) {
							display = "block";
						}

						elem.remove();

						elemdisplay[ nodeName ] = display;
					}

					jQuery.data(this[i], "olddisplay", display);
				}
			}

			// Set the display of the elements in a second loop
			// to avoid the constant reflow
			for ( var j = 0, k = this.length; j < k; j++ ) {
				this[j].style.display = jQuery.data(this[j], "olddisplay") || "";
			}

			return this;
		}
	},

	hide: function( speed, callback ) {
		if ( speed || speed === 0 ) {
			return this.animate( genFx("hide", 3), speed, callback);

		} else {
			for ( var i = 0, l = this.length; i < l; i++ ) {
				var old = jQuery.data(this[i], "olddisplay");
				if ( !old && old !== "none" ) {
					jQuery.data(this[i], "olddisplay", jQuery.css(this[i], "display"));
				}
			}

			// Set the display of the elements in a second loop
			// to avoid the constant reflow
			for ( var j = 0, k = this.length; j < k; j++ ) {
				this[j].style.display = "none";
			}

			return this;
		}
	},

	// Save the old toggle function
	_toggle: jQuery.fn.toggle,

	toggle: function( fn, fn2 ) {
		var bool = typeof fn === "boolean";

		if ( jQuery.isFunction(fn) && jQuery.isFunction(fn2) ) {
			this._toggle.apply( this, arguments );

		} else if ( fn == null || bool ) {
			this.each(function() {
				var state = bool ? fn : jQuery(this).is(":hidden");
				jQuery(this)[ state ? "show" : "hide" ]();
			});

		} else {
			this.animate(genFx("toggle", 3), fn, fn2);
		}

		return this;
	},

	fadeTo: function( speed, to, callback ) {
		return this.filter(":hidden").css("opacity", 0).show().end()
					.animate({opacity: to}, speed, callback);
	},

	animate: function( prop, speed, easing, callback ) {
		var optall = jQuery.speed(speed, easing, callback);

		if ( jQuery.isEmptyObject( prop ) ) {
			return this.each( optall.complete );
		}

		return this[ optall.queue === false ? "each" : "queue" ](function() {
			var opt = jQuery.extend({}, optall), p,
				hidden = this.nodeType === 1 && jQuery(this).is(":hidden"),
				self = this;

			for ( p in prop ) {
				var name = p.replace(rdashAlpha, fcamelCase);

				if ( p !== name ) {
					prop[ name ] = prop[ p ];
					delete prop[ p ];
					p = name;
				}

				if ( prop[p] === "hide" && hidden || prop[p] === "show" && !hidden ) {
					return opt.complete.call(this);
				}

				if ( ( p === "height" || p === "width" ) && this.style ) {
					// Store display property
					opt.display = jQuery.css(this, "display");

					// Make sure that nothing sneaks out
					opt.overflow = this.style.overflow;
				}

				if ( jQuery.isArray( prop[p] ) ) {
					// Create (if needed) and add to specialEasing
					(opt.specialEasing = opt.specialEasing || {})[p] = prop[p][1];
					prop[p] = prop[p][0];
				}
			}

			if ( opt.overflow != null ) {
				this.style.overflow = "hidden";
			}

			opt.curAnim = jQuery.extend({}, prop);

			jQuery.each( prop, function( name, val ) {
				var e = new jQuery.fx( self, opt, name );

				if ( rfxtypes.test(val) ) {
					e[ val === "toggle" ? hidden ? "show" : "hide" : val ]( prop );

				} else {
					var parts = rfxnum.exec(val),
						start = e.cur(true) || 0;

					if ( parts ) {
						var end = parseFloat( parts[2] ),
							unit = parts[3] || "px";

						// We need to compute starting value
						if ( unit !== "px" ) {
							self.style[ name ] = (end || 1) + unit;
							start = ((end || 1) / e.cur(true)) * start;
							self.style[ name ] = start + unit;
						}

						// If a +=/-= token was provided, we're doing a relative animation
						if ( parts[1] ) {
							end = ((parts[1] === "-=" ? -1 : 1) * end) + start;
						}

						e.custom( start, end, unit );

					} else {
						e.custom( start, val, "" );
					}
				}
			});

			// For JS strict compliance
			return true;
		});
	},

	stop: function( clearQueue, gotoEnd ) {
		var timers = jQuery.timers;

		if ( clearQueue ) {
			this.queue([]);
		}

		this.each(function() {
			// go in reverse order so anything added to the queue during the loop is ignored
			for ( var i = timers.length - 1; i >= 0; i-- ) {
				if ( timers[i].elem === this ) {
					if (gotoEnd) {
						// force the next step to be the last
						timers[i](true);
					}

					timers.splice(i, 1);
				}
			}
		});

		// start the next in the queue if the last step wasn't forced
		if ( !gotoEnd ) {
			this.dequeue();
		}

		return this;
	}

});

// Generate shortcuts for custom animations
jQuery.each({
	slideDown: genFx("show", 1),
	slideUp: genFx("hide", 1),
	slideToggle: genFx("toggle", 1),
	fadeIn: { opacity: "show" },
	fadeOut: { opacity: "hide" }
}, function( name, props ) {
	jQuery.fn[ name ] = function( speed, callback ) {
		return this.animate( props, speed, callback );
	};
});

jQuery.extend({
	speed: function( speed, easing, fn ) {
		var opt = speed && typeof speed === "object" ? speed : {
			complete: fn || !fn && easing ||
				jQuery.isFunction( speed ) && speed,
			duration: speed,
			easing: fn && easing || easing && !jQuery.isFunction(easing) && easing
		};

		opt.duration = jQuery.fx.off ? 0 : typeof opt.duration === "number" ? opt.duration :
			jQuery.fx.speeds[opt.duration] || jQuery.fx.speeds._default;

		// Queueing
		opt.old = opt.complete;
		opt.complete = function() {
			if ( opt.queue !== false ) {
				jQuery(this).dequeue();
			}
			if ( jQuery.isFunction( opt.old ) ) {
				opt.old.call( this );
			}
		};

		return opt;
	},

	easing: {
		linear: function( p, n, firstNum, diff ) {
			return firstNum + diff * p;
		},
		swing: function( p, n, firstNum, diff ) {
			return ((-Math.cos(p*Math.PI)/2) + 0.5) * diff + firstNum;
		}
	},

	timers: [],

	fx: function( elem, options, prop ) {
		this.options = options;
		this.elem = elem;
		this.prop = prop;

		if ( !options.orig ) {
			options.orig = {};
		}
	}

});

jQuery.fx.prototype = {
	// Simple function for setting a style value
	update: function() {
		if ( this.options.step ) {
			this.options.step.call( this.elem, this.now, this );
		}

		(jQuery.fx.step[this.prop] || jQuery.fx.step._default)( this );

		// Set display property to block for height/width animations
		if ( ( this.prop === "height" || this.prop === "width" ) && this.elem.style ) {
			this.elem.style.display = "block";
		}
	},

	// Get the current size
	cur: function( force ) {
		if ( this.elem[this.prop] != null && (!this.elem.style || this.elem.style[this.prop] == null) ) {
			return this.elem[ this.prop ];
		}

		var r = parseFloat(jQuery.css(this.elem, this.prop, force));
		return r && r > -10000 ? r : parseFloat(jQuery.curCSS(this.elem, this.prop)) || 0;
	},

	// Start an animation from one number to another
	custom: function( from, to, unit ) {
		this.startTime = now();
		this.start = from;
		this.end = to;
		this.unit = unit || this.unit || "px";
		this.now = this.start;
		this.pos = this.state = 0;

		var self = this;
		function t( gotoEnd ) {
			return self.step(gotoEnd);
		}

		t.elem = this.elem;

		if ( t() && jQuery.timers.push(t) && !timerId ) {
			timerId = setInterval(jQuery.fx.tick, 13);
		}
	},

	// Simple 'show' function
	show: function() {
		// Remember where we started, so that we can go back to it later
		this.options.orig[this.prop] = jQuery.style( this.elem, this.prop );
		this.options.show = true;

		// Begin the animation
		// Make sure that we start at a small width/height to avoid any
		// flash of content
		this.custom(this.prop === "width" || this.prop === "height" ? 1 : 0, this.cur());

		// Start by showing the element
		jQuery( this.elem ).show();
	},

	// Simple 'hide' function
	hide: function() {
		// Remember where we started, so that we can go back to it later
		this.options.orig[this.prop] = jQuery.style( this.elem, this.prop );
		this.options.hide = true;

		// Begin the animation
		this.custom(this.cur(), 0);
	},

	// Each step of an animation
	step: function( gotoEnd ) {
		var t = now(), done = true;

		if ( gotoEnd || t >= this.options.duration + this.startTime ) {
			this.now = this.end;
			this.pos = this.state = 1;
			this.update();

			this.options.curAnim[ this.prop ] = true;

			for ( var i in this.options.curAnim ) {
				if ( this.options.curAnim[i] !== true ) {
					done = false;
				}
			}

			if ( done ) {
				if ( this.options.display != null ) {
					// Reset the overflow
					this.elem.style.overflow = this.options.overflow;

					// Reset the display
					var old = jQuery.data(this.elem, "olddisplay");
					this.elem.style.display = old ? old : this.options.display;

					if ( jQuery.css(this.elem, "display") === "none" ) {
						this.elem.style.display = "block";
					}
				}

				// Hide the element if the "hide" operation was done
				if ( this.options.hide ) {
					jQuery(this.elem).hide();
				}

				// Reset the properties, if the item has been hidden or shown
				if ( this.options.hide || this.options.show ) {
					for ( var p in this.options.curAnim ) {
						jQuery.style(this.elem, p, this.options.orig[p]);
					}
				}

				// Execute the complete function
				this.options.complete.call( this.elem );
			}

			return false;

		} else {
			var n = t - this.startTime;
			this.state = n / this.options.duration;

			// Perform the easing function, defaults to swing
			var specialEasing = this.options.specialEasing && this.options.specialEasing[this.prop];
			var defaultEasing = this.options.easing || (jQuery.easing.swing ? "swing" : "linear");
			this.pos = jQuery.easing[specialEasing || defaultEasing](this.state, n, 0, 1, this.options.duration);
			this.now = this.start + ((this.end - this.start) * this.pos);

			// Perform the next step of the animation
			this.update();
		}

		return true;
	}
};

jQuery.extend( jQuery.fx, {
	tick: function() {
		var timers = jQuery.timers;

		for ( var i = 0; i < timers.length; i++ ) {
			if ( !timers[i]() ) {
				timers.splice(i--, 1);
			}
		}

		if ( !timers.length ) {
			jQuery.fx.stop();
		}
	},
		
	stop: function() {
		clearInterval( timerId );
		timerId = null;
	},
	
	speeds: {
		slow: 600,
 		fast: 200,
 		// Default speed
 		_default: 400
	},

	step: {
		opacity: function( fx ) {
			jQuery.style(fx.elem, "opacity", fx.now);
		},

		_default: function( fx ) {
			if ( fx.elem.style && fx.elem.style[ fx.prop ] != null ) {
				fx.elem.style[ fx.prop ] = (fx.prop === "width" || fx.prop === "height" ? Math.max(0, fx.now) : fx.now) + fx.unit;
			} else {
				fx.elem[ fx.prop ] = fx.now;
			}
		}
	}
});

if ( jQuery.expr && jQuery.expr.filters ) {
	jQuery.expr.filters.animated = function( elem ) {
		return jQuery.grep(jQuery.timers, function( fn ) {
			return elem === fn.elem;
		}).length;
	};
}

function genFx( type, num ) {
	var obj = {};

	jQuery.each( fxAttrs.concat.apply([], fxAttrs.slice(0,num)), function() {
		obj[ this ] = type;
	});

	return obj;
}
if ( "getBoundingClientRect" in document.documentElement ) {
	jQuery.fn.offset = function( options ) {
		var elem = this[0];

		if ( options ) { 
			return this.each(function( i ) {
				jQuery.offset.setOffset( this, options, i );
			});
		}

		if ( !elem || !elem.ownerDocument ) {
			return null;
		}

		if ( elem === elem.ownerDocument.body ) {
			return jQuery.offset.bodyOffset( elem );
		}

		var box = elem.getBoundingClientRect(), doc = elem.ownerDocument, body = doc.body, docElem = doc.documentElement,
			clientTop = docElem.clientTop || body.clientTop || 0, clientLeft = docElem.clientLeft || body.clientLeft || 0,
			top  = box.top  + (self.pageYOffset || jQuery.support.boxModel && docElem.scrollTop  || body.scrollTop ) - clientTop,
			left = box.left + (self.pageXOffset || jQuery.support.boxModel && docElem.scrollLeft || body.scrollLeft) - clientLeft;

		return { top: top, left: left };
	};

} else {
	jQuery.fn.offset = function( options ) {
		var elem = this[0];

		if ( options ) { 
			return this.each(function( i ) {
				jQuery.offset.setOffset( this, options, i );
			});
		}

		if ( !elem || !elem.ownerDocument ) {
			return null;
		}

		if ( elem === elem.ownerDocument.body ) {
			return jQuery.offset.bodyOffset( elem );
		}

		jQuery.offset.initialize();

		var offsetParent = elem.offsetParent, prevOffsetParent = elem,
			doc = elem.ownerDocument, computedStyle, docElem = doc.documentElement,
			body = doc.body, defaultView = doc.defaultView,
			prevComputedStyle = defaultView ? defaultView.getComputedStyle( elem, null ) : elem.currentStyle,
			top = elem.offsetTop, left = elem.offsetLeft;

		while ( (elem = elem.parentNode) && elem !== body && elem !== docElem ) {
			if ( jQuery.offset.supportsFixedPosition && prevComputedStyle.position === "fixed" ) {
				break;
			}

			computedStyle = defaultView ? defaultView.getComputedStyle(elem, null) : elem.currentStyle;
			top  -= elem.scrollTop;
			left -= elem.scrollLeft;

			if ( elem === offsetParent ) {
				top  += elem.offsetTop;
				left += elem.offsetLeft;

				if ( jQuery.offset.doesNotAddBorder && !(jQuery.offset.doesAddBorderForTableAndCells && /^t(able|d|h)$/i.test(elem.nodeName)) ) {
					top  += parseFloat( computedStyle.borderTopWidth  ) || 0;
					left += parseFloat( computedStyle.borderLeftWidth ) || 0;
				}

				prevOffsetParent = offsetParent, offsetParent = elem.offsetParent;
			}

			if ( jQuery.offset.subtractsBorderForOverflowNotVisible && computedStyle.overflow !== "visible" ) {
				top  += parseFloat( computedStyle.borderTopWidth  ) || 0;
				left += parseFloat( computedStyle.borderLeftWidth ) || 0;
			}

			prevComputedStyle = computedStyle;
		}

		if ( prevComputedStyle.position === "relative" || prevComputedStyle.position === "static" ) {
			top  += body.offsetTop;
			left += body.offsetLeft;
		}

		if ( jQuery.offset.supportsFixedPosition && prevComputedStyle.position === "fixed" ) {
			top  += Math.max( docElem.scrollTop, body.scrollTop );
			left += Math.max( docElem.scrollLeft, body.scrollLeft );
		}

		return { top: top, left: left };
	};
}

jQuery.offset = {
	initialize: function() {
		var body = document.body, container = document.createElement("div"), innerDiv, checkDiv, table, td, bodyMarginTop = parseFloat( jQuery.curCSS(body, "marginTop", true) ) || 0,
			html = "<div style='position:absolute;top:0;left:0;margin:0;border:5px solid #000;padding:0;width:1px;height:1px;'><div></div></div><table style='position:absolute;top:0;left:0;margin:0;border:5px solid #000;padding:0;width:1px;height:1px;' cellpadding='0' cellspacing='0'><tr><td></td></tr></table>";

		jQuery.extend( container.style, { position: "absolute", top: 0, left: 0, margin: 0, border: 0, width: "1px", height: "1px", visibility: "hidden" } );

		container.innerHTML = html;
		body.insertBefore( container, body.firstChild );
		innerDiv = container.firstChild;
		checkDiv = innerDiv.firstChild;
		td = innerDiv.nextSibling.firstChild.firstChild;

		this.doesNotAddBorder = (checkDiv.offsetTop !== 5);
		this.doesAddBorderForTableAndCells = (td.offsetTop === 5);

		checkDiv.style.position = "fixed", checkDiv.style.top = "20px";
		// safari subtracts parent border width here which is 5px
		this.supportsFixedPosition = (checkDiv.offsetTop === 20 || checkDiv.offsetTop === 15);
		checkDiv.style.position = checkDiv.style.top = "";

		innerDiv.style.overflow = "hidden", innerDiv.style.position = "relative";
		this.subtractsBorderForOverflowNotVisible = (checkDiv.offsetTop === -5);

		this.doesNotIncludeMarginInBodyOffset = (body.offsetTop !== bodyMarginTop);

		body.removeChild( container );
		body = container = innerDiv = checkDiv = table = td = null;
		jQuery.offset.initialize = jQuery.noop;
	},

	bodyOffset: function( body ) {
		var top = body.offsetTop, left = body.offsetLeft;

		jQuery.offset.initialize();

		if ( jQuery.offset.doesNotIncludeMarginInBodyOffset ) {
			top  += parseFloat( jQuery.curCSS(body, "marginTop",  true) ) || 0;
			left += parseFloat( jQuery.curCSS(body, "marginLeft", true) ) || 0;
		}

		return { top: top, left: left };
	},
	
	setOffset: function( elem, options, i ) {
		// set position first, in-case top/left are set even on static elem
		if ( /static/.test( jQuery.curCSS( elem, "position" ) ) ) {
			elem.style.position = "relative";
		}
		var curElem   = jQuery( elem ),
			curOffset = curElem.offset(),
			curTop    = parseInt( jQuery.curCSS( elem, "top",  true ), 10 ) || 0,
			curLeft   = parseInt( jQuery.curCSS( elem, "left", true ), 10 ) || 0;

		if ( jQuery.isFunction( options ) ) {
			options = options.call( elem, i, curOffset );
		}

		var props = {
			top:  (options.top  - curOffset.top)  + curTop,
			left: (options.left - curOffset.left) + curLeft
		};
		
		if ( "using" in options ) {
			options.using.call( elem, props );
		} else {
			curElem.css( props );
		}
	}
};


jQuery.fn.extend({
	position: function() {
		if ( !this[0] ) {
			return null;
		}

		var elem = this[0],

		// Get *real* offsetParent
		offsetParent = this.offsetParent(),

		// Get correct offsets
		offset       = this.offset(),
		parentOffset = /^body|html$/i.test(offsetParent[0].nodeName) ? { top: 0, left: 0 } : offsetParent.offset();

		// Subtract element margins
		// note: when an element has margin: auto the offsetLeft and marginLeft
		// are the same in Safari causing offset.left to incorrectly be 0
		offset.top  -= parseFloat( jQuery.curCSS(elem, "marginTop",  true) ) || 0;
		offset.left -= parseFloat( jQuery.curCSS(elem, "marginLeft", true) ) || 0;

		// Add offsetParent borders
		parentOffset.top  += parseFloat( jQuery.curCSS(offsetParent[0], "borderTopWidth",  true) ) || 0;
		parentOffset.left += parseFloat( jQuery.curCSS(offsetParent[0], "borderLeftWidth", true) ) || 0;

		// Subtract the two offsets
		return {
			top:  offset.top  - parentOffset.top,
			left: offset.left - parentOffset.left
		};
	},

	offsetParent: function() {
		return this.map(function() {
			var offsetParent = this.offsetParent || document.body;
			while ( offsetParent && (!/^body|html$/i.test(offsetParent.nodeName) && jQuery.css(offsetParent, "position") === "static") ) {
				offsetParent = offsetParent.offsetParent;
			}
			return offsetParent;
		});
	}
});


// Create scrollLeft and scrollTop methods
jQuery.each( ["Left", "Top"], function( i, name ) {
	var method = "scroll" + name;

	jQuery.fn[ method ] = function(val) {
		var elem = this[0], win;
		
		if ( !elem ) {
			return null;
		}

		if ( val !== undefined ) {
			// Set the scroll offset
			return this.each(function() {
				win = getWindow( this );

				if ( win ) {
					win.scrollTo(
						!i ? val : jQuery(win).scrollLeft(),
						 i ? val : jQuery(win).scrollTop()
					);

				} else {
					this[ method ] = val;
				}
			});
		} else {
			win = getWindow( elem );

			// Return the scroll offset
			return win ? ("pageXOffset" in win) ? win[ i ? "pageYOffset" : "pageXOffset" ] :
				jQuery.support.boxModel && win.document.documentElement[ method ] ||
					win.document.body[ method ] :
				elem[ method ];
		}
	};
});

function getWindow( elem ) {
	return ("scrollTo" in elem && elem.document) ?
		elem :
		elem.nodeType === 9 ?
			elem.defaultView || elem.parentWindow :
			false;
}
// Create innerHeight, innerWidth, outerHeight and outerWidth methods
jQuery.each([ "Height", "Width" ], function( i, name ) {

	var type = name.toLowerCase();

	// innerHeight and innerWidth
	jQuery.fn["inner" + name] = function() {
		return this[0] ?
			jQuery.css( this[0], type, false, "padding" ) :
			null;
	};

	// outerHeight and outerWidth
	jQuery.fn["outer" + name] = function( margin ) {
		return this[0] ?
			jQuery.css( this[0], type, false, margin ? "margin" : "border" ) :
			null;
	};

	jQuery.fn[ type ] = function( size ) {
		// Get window width or height
		var elem = this[0];
		if ( !elem ) {
			return size == null ? null : this;
		}
		
		if ( jQuery.isFunction( size ) ) {
			return this.each(function( i ) {
				var self = jQuery( this );
				self[ type ]( size.call( this, i, self[ type ]() ) );
			});
		}

		return ("scrollTo" in elem && elem.document) ? // does it walk and quack like a window?
			// Everyone else use document.documentElement or document.body depending on Quirks vs Standards mode
			elem.document.compatMode === "CSS1Compat" && elem.document.documentElement[ "client" + name ] ||
			elem.document.body[ "client" + name ] :

			// Get document width or height
			(elem.nodeType === 9) ? // is it a document
				// Either scroll[Width/Height] or offset[Width/Height], whichever is greater
				Math.max(
					elem.documentElement["client" + name],
					elem.body["scroll" + name], elem.documentElement["scroll" + name],
					elem.body["offset" + name], elem.documentElement["offset" + name]
				) :

				// Get or set width or height on the element
				size === undefined ?
					// Get width or height on the element
					jQuery.css( elem, type ) :

					// Set the width or height on the element (default to pixels if value is unitless)
					this.css( type, typeof size === "string" ? size : size + "px" );
	};

});

exports = module.exports = jQuery;
exports.jQuery = jQuery;

})(window);
;});
/* >>>>>>>>>> BEGIN source/lib/qunit.js */
tiki.module('core_test:qunit',function(require,exports,module,tiki){// ==========================================================================
// Project:   CoreTest Unit Testing Library
// Copyright: ©2010 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================

"use exports module test ok equals expect start stop same run raises";

var Ct = require('core'),
    utils = require('utils');

    
// internal function called to finish a phase
function _finishPhase() {
  var st = Ct.qUnitStatus, pr = st.promise;

  // invoke final handler...
  if (st.final) st.final();
  
  // cleanup phase and resolve promise to move on
  st.promise    = st.final = null;
  st.isFinished = true;
  st.isRunning  = true;
  if (pr) pr.resolve();
  
}

// starts a new module.  This is basically the same as CoreTest
exports.module = function(moduleName, opts) {
  Ct.module(moduleName);
  
  var setup = opts && opts.setup,
      teardown = opts && opts.teardown;
      
  Ct.setup(function(t) {
    
    var pr = utils.Promise.create() ;
    
    var st = {
      test: t,
      promise: pr, // changes as we step through each phase
      isFinished: false, // reset each phase
      expect:  -1, // tests in teardown
      assertions: 0,
      isRunning: true // used by start/stop
    };
    Ct.qUnitStatus = st;

    if (setup) setup();

    st.isFinished = true;
    if (st.isRunning) _finishPhase();
    
    return pr; //make async
  });

  Ct.teardown(function(t) {
    var st = Ct.qUnitStatus,
        pr = utils.Promise.create();
        
    st.promise = pr ;
    st.isFinished = false;
    st.isRunning = true;
    
    // final code to run after teardown handler 
    st.final = function() {
      if (st.expects>=0) {
        t.equal(st.assertions, st.expects, 'expected assertions');
      }
      Ct.qUnitStatus = null ;
    };

    if (teardown) teardown();
    st.isFinished = true;
    if (st.isRunning) _finishPhase();
    
    return pr; // make async
  });
  
};


// adds a test.  We actually wrap an async test here so we can implement 
// start, stop, and expect
exports.test  = function(testName, handler) {
  
  if (!Ct.defaultPlan) exports.module('default');
  
  Ct.test(testName, function(t) {
    var st = Ct.qUnitStatus,
        pr = utils.Promise.create();
        
    st.promise = pr;
    st.isFinished = false;
    st.isRunning  = true;
    
    handler();
    st.isFinished = true;
    if (st.isRunning) _finishPhase();

    return pr; // make this async
  });
  
};

// stops a test from finishing.  must be matched with a call to start() later
exports.stop = function() {
  if (!Ct.qUnitStatus) throw "cannot call stop() outside of a unit test";
  Ct.qUnitStatus.isRunning = false ;
};

// starts a test that was stopped.  Must be called after a call to start()
exports.start = function() {
  var st = Ct.qUnitStatus;

  if (!st) throw "cannot call stop() outside of a unit test";
  if (!st.isRunning && st.isFinished) _finishPhase();
  else st.isRunning = true;
};

// set the expected assertions for the test
exports.expect = function(cnt) {
  if (!Ct.qUnitStatus) throw "cannot call expect() outside of a unit test";
  Ct.qUnitStatus.expects = cnt;
};

// ok assertion
exports.ok = function(value, message) {
  if (!Ct.qUnitStatus) throw "cannot call ok() outside of a unit test";
  if (message === undefined) message = 'ok';
  Ct.qUnitStatus.assertions++;
  Ct.qUnitStatus.test.ok(value, message);
};

// equals assertion
exports.equals = function(actual, expected, message) {
  if (!Ct.qUnitStatus) throw "cannot call equals() outside of a unit test";
  if (message === undefined) message = 'ok';
  Ct.qUnitStatus.assertions++;
  Ct.qUnitStatus.test.equal(actual, expected, message);
};

// same assertion
exports.same = function(actual, expected, message) {
  if (!Ct.qUnitStatus) throw "cannot call same() outside of a unit test";
  if (message === undefined) message = 'same';
  Ct.qUnitStatus.assertions++;
  Ct.qUnitStatus.test.deepEqual(actual, expected, message);
};

exports.raises = function(handler, err, message) {
  if (!Ct.qUnitStatus) throw "cannot call raises() outside of a unit test";
  if (err === true) {
    err = message;
    message = undefined;
  }
  Ct.qUnitStatus.assertions++;
  Ct.qUnitStatus.test.throws(handler, err, message);
};

exports.run = function() {
  Ct.run();
};


;});
/* >>>>>>>>>> BEGIN source/lib/spec.js */
tiki.module('core_test:spec',function(require,exports,module,tiki){// ==========================================================================
// Project:   CoreTest Unit Testing Library
// Copyright: ©2010 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================
/*globals describe before after it val */
"use exports describe before after it val";

/**
  @file @private
  
  UNFINISHED -
  
  Implements a BDD-style specification API.  Example:
  
  {{{
    describe('some part of the app', function() {
    
      before(function() {
        this.foo = 'bar';
      });
      
      after(function() {
        this.foo = null;
      });
      
      it('should do something', function() {
        val(this.foo).shouldBe('bar');
      });
      
      describe('a nested part of the app', function() {
        before(function() {
          // also sets this.foo = 'bar' before this
          this.bar = 10;
        });
        
        it('should have foo and bar', function() {
          val(this.foo).shouldBe('bar');
          val(this.bar).greaterThan(5);
        });
      });
      
    });
  }}}
*/

var Ct    =  require('core'),
    utils = require('utils'),
    valueOf;

// WARNING: This code is experimental and unfinished.  Do not use it. 

var state = {
  beforeHandlers: [],
  afterHandlers: [],
  mod: null
};

function _cancel(v, pr) {
  pr.cancel();
}

function _resolve() {
  
}

/**
  Describes a new context.  In CoreTest terms this will setup a new module 
  on the current plan.
*/
exports.describe = function(desc, handler, context) {

  var plan, mod;
  
  // make a place for handlers on stack
  state.beforeHandlers.push(null);
  state.afterHandlers.push(null);

  // push on a new module or create one
  if (!state.mod) {
    plan = new Ct.Plan('spec');
    mod = state.mod = plan.module(desc);
  } else mod = state.mod = state.mod.module(desc); // nest
  
  handler.call(context); // setup the description

  // extract a list of all handlers currently nested
  var beforeHandlers, afterHandlers, idx, len;
  len = state.beforeHandlers.length;
  beforeHandlers = [];
  for(idx=0;idx<len;idx++) {
    handler = state.beforeHandlers[idx];
    if (handler) beforeHandlers.push(handler);
  }

  len = state.afterHandlers.length;
  afterHandlers = [];
  for(idx=0;idx<len;idx++) {
    handler = state.afterHandlers[idx];
    if (handler) afterHandlers.push(handler);
  }
  

  if (beforeHandlers.length>0) {
    mod.setup(function(test, setupPr) {
      var first = utils.Promise.create(),
          pr    = first,
          len   = beforeHandlers.length,
          idx;
          
      for(idx=0;idx<len;idx++) {
        pr = pr.then(test, _resolve(beforeHandlers[idx]), _cancel);
      }
    }, true);
  }
  
  // remove before/after handlers from stack
  state.beforeHandlers.pop();
  state.afterHandlers.pop();

  // we created a plan if this is a top level describe.  If so, schedule it
  if (plan) Ct.run(plan);
};

describe("some foobar", function() {

  before.async(function(test, pr) {
    
  });
  
  it("should do one thing or another", function() {
    val('foo').shouldBe('foo');
  });

});



// root handler for the tdd model.  returns a new assertion object
valueOf = function(val) {
  var ret = utils.beget(valueOf.fn);
  ret.val = val;
  return ret ;
};

// object meaning 'empty' - make an array so it is a unique instance
exports.EMPTY = Ct.EMPTY = ['EMPTY'];

// comparison functions go here
valueOf.fn = {

  inverted: false,
  
  explain: function(pass, verb, expected) {
    var actual = Ct.dump(this.val);
    expected   = Ct.dump(actual);

    if (this.inverted) {
      pass = !pass;
      verb = 'not ' + verb;
    }
    var msg = utils.fmt('%@ should %@ %@', expected, verb, actual);
    
    Ct.assertion(pass, msg);
    return this;
  },
  
  // linker - valueOf('foo').should().be('foo');
  
  shouldBe: function(val) {
    this.inverted = false;
    if (arguments.length===0) return this;
    else return this._shouldBe(val);
  },
  
  _shouldBe: function(val) {
    var v = this.val;
    if (val===Ct.EMPTY) v = v && (v.length!==undefined) && v.length===0;
    else if (val===true) v = !!v;
    else if (val===false) v = !v;
    else v = (v === val);

    return this.explain(v, 'be', val);
  },

  shouldNotBe: function(val) {
    this.inverted = true;
    if (arguments.length===0) return this;
    else return this._shouldBe(val);
  },

  shouldBeTrue: function() {
    return this._shouldBe(true);
  },

  shouldBeFalse: function() {
    return this._shouldBe(false);
  },
  
  shouldBeEmpty: function() {
    return this._shouldBe(Ct.EMPTY);
  },
  
  shouldBeNull: function() {
    return this.shouldBe(null);
  },
  
  shouldNotBeNull: function() {
    return this.shouldNotBe(null);
  },
  
  shouldBeUndefined: function() {
    return this.shouldBe(undefined);
  },
  
  shouldNotBeUndefined: function() {
    return this.shouldNotBe(undefined);
  },
  
  shouldBeSame: function(val) {
    
  },
  
  shouldNotBeSame: function(val) {
    
  },
  
  shouldHave: function(len) {
    
  },
  
  shouldInclude: function(item) {
    this.inverted = false;
    return this.include(item);
  },
  
  shouldNotInclude: function(item) {
    this.inverted = true;
    return this.include(item);
  },
  
  include: function(item) {
    
  },
  
  shouldMatch: function(regex) {
    
  },
  
  shouldNotMatch: function(regex) {
    
  },
  
  shouldFail: function() {
    
  }
  
};



exports = module.exports = valueOf;
exports.valueOf = valueOf;;});
/* >>>>>>>>>> BEGIN source/lib/system/dump.js */
tiki.module('core_test:system/dump',function(require,exports,module,tiki){// ==========================================================================
// Project:   SproutCore Costello - Property Observing Library
// Copyright: ©2006-2009 Sprout Systems, Inc. and contributors.
//            Portions ©2008-2009 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================

var CoreTest = require('core');

/**
 * jsDump
 * Copyright (c) 2008 Ariel Flesler - aflesler(at)gmail(dot)com | http://flesler.blogspot.com
 * Licensed under BSD (http://www.opensource.org/licenses/bsd-license.php)
 * Date: 5/15/2008
 * @projectDescription Advanced and extensible data dumping for Javascript.
 * @version 1.0.0
 * @author Ariel Flesler
 * @link {http://flesler.blogspot.com/2008/05/jsdump-pretty-dump-of-any-javascript.html}
 */
var reName, jsDump;

function quote( str ){
  return '"' + str.toString().replace(/"/g, '\\"') + '"';
}

function literal( o ){
  return o + '';  
}

function join( pre, arr, post ){
  var s     = jsDump.separator(),
      base  = jsDump.indent(),
      inner = jsDump.indent(1);
      
  if( arr.join )  arr = arr.join( ',' + s + inner );
  if( !arr ) return pre + post;
  
  return [ pre, inner + arr, base + post ].join(s);
}

function array( arr ){
  var i = arr.length, ret = [];
  ret.length=i;         
  this.up();
  while( i-- ) ret[i] = this._parse( arr[i] );        
  this.down();
  return join( '[', ret, ']' );
}

reName = /^function (\w+)/;

jsDump = CoreTest.jsDump = {

  parse: function(obj, type) {
    if (obj && obj.toString) {
      var toString = obj.toString;
      if ((toString !== Object.prototype.toString) && (toString !== Array.toString)) return obj.toString();
    }
    if (obj && obj.inspect) return obj.inspect();
    
    this.seen = [];
    var ret = this._parse(obj, type);
    this.seen = null;
    return ret ;
  },
  
  //type is used mostly internally, you can fix a (custom)type in advance
  _parse: function( obj, type ) {
    
    
    var parser = this.parsers[ type || this.typeOf(obj) ];
    type = typeof parser;     

    // avoid recursive loops
    if ((parser === this.parsers.object) && (this.seen.indexOf(obj)>=0)) {
      return '(recursive)';
    }
    this.seen.push(obj);
    
    return type == 'function' ? parser.call( this, obj ) :
         type == 'string' ? parser :
         this.parsers.error;
  },
  typeOf:function( obj ){
    var type = typeof obj,
      f = 'function';//we'll use it 3 times, save it
      
    if (obj && (obj.isObject || obj.isClass)) return 'scobj';
    return type != 'object' && type != f ? type :
      !obj ? 'null' :
      obj.exec ? 'regexp' :// some browsers (FF) consider regexps functions
      obj.getHours ? 'date' :
      obj.scrollBy ?  'window' :
      obj.nodeName == '#document' ? 'document' :
      obj.nodeName ? 'node' :
      obj.item ? 'nodelist' : // Safari reports nodelists as functions
      obj.callee ? 'arguments' :
      obj.call || obj.constructor != Array && //an array would also fall on this hack
        (obj+'').indexOf(f) != -1 ? f : //IE reports functions like alert, as objects
      'length' in obj ? 'array' :
      type;
  },
  separator:function(){
    return this.multiline ? this.HTML ? '<br />' : '\n' : this.HTML ? '&nbsp;' : ' ';
  },
  indent:function( extra ){// extra can be a number, shortcut for increasing-calling-decreasing
    if( !this.multiline ) return '';
    
    var chr = this.indentChar;
    if( this.HTML ) chr = chr.replace(/\t/g,'   ').replace(/ /g,'&nbsp;');
    
    var ret = [];
    ret.length = this._depth_ + (extra||0);
    return ret.join(chr);
  },
  
  up:function( a ){
    this._depth_ += a || 1;
  },
  down:function( a ){
    this._depth_ -= a || 1;
  },
  setParser:function( name, parser ){
    this.parsers[name] = parser;
  },
  // The next 3 are exposed so you can use them
  quote:quote, 
  literal:literal,
  join:join,
  //
  _depth_: 1,
  // This is the list of parsers, to modify them, use jsDump.setParser
  parsers:{
    window: '[Window]',
    document: '[Document]',
    error:'[ERROR]', //when no parser is found, shouldn't happen
    unknown: '[Unknown]',
    'null':'null',
    'undefined':'undefined',
    'function':function( fn ){
      var ret = 'function',
        name = 'name' in fn ? fn.name : (reName.exec(fn)||[])[1];//functions never have name in IE
      if( name ) ret += ' ' + name;
      ret += '(';
      
      ret = [ ret, this._parse( fn, 'functionArgs' ), '){'].join('');
      return join( ret, this._parse(fn,'functionCode'), '}' );
    },
    array: array,
    nodelist: array,
    'arguments': array,
    scobj: function(obj) { return obj.toString(); },
    object:function( map ){
      
      var ret = [ ];
      this.up();
      for( var key in map ) {
        ret.push( this._parse(key,'key') + ': ' + this._parse(map[key]) );
      }
      this.down();
      return join( '{', ret, '}' );
    },
    node:function( node ){
      var open = this.HTML ? '&lt;' : '<',
        close = this.HTML ? '&gt;' : '>';
        
      var tag = node.nodeName.toLowerCase(),
        ret = open + tag;
        
      for( var a in this.DOMAttrs ){
        var val = node[this.DOMAttrs[a]];
        if( val ) {
          ret += ' ' + a + '=' + this._parse( val, 'attribute' );
        }
      }
      return ret + close + open + '/' + tag + close;
    },
    functionArgs:function( fn ){//function calls it internally, it's the arguments part of the function
      var l = fn.length;
      if( !l ) return '';       
      
      var args = [];
      args.length = l;
      while( l-- ) args[l] = String.fromCharCode(97+l);//97 is 'a'
      return ' ' + args.join(', ') + ' ';
    },
    key:quote, //object calls it internally, the key part of an item in a map
    functionCode:'[code]', //function calls it internally, it's the content of the function
    attribute:quote, //node calls it internally, it's an html attribute value
    string:quote,
    date:quote,
    regexp:literal, //regex
    number:literal,
    'boolean':literal
  },
  DOMAttrs:{//attributes to dump from nodes, name=>realName
    id:'id',
    name:'name',
    'class':'className'
  },
  HTML:false,//if true, entities are escaped ( <, >, \t, space and \n )
  indentChar:'   ',//indentation unit
  multiline:true //if true, items in a collection, are separated by a \n, else just a space.
};

CoreTest.dump = function dump(obj,type) {
  return CoreTest.jsDump.parse(obj, type);
};
;});
/* >>>>>>>>>> BEGIN source/lib/system/equiv.js */
tiki.module('core_test:system/equiv',function(require,exports,module,tiki){// ==========================================================================
// Project:   SproutCore Costello - Property Observing Library
// Copyright: ©2006-2009 Sprout Systems, Inc. and contributors.
//            Portions ©2008-2009 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================

var Ct = require('core');

var _equiv; // internal loop function

/** 
  Tests for deep equality of any JavaScript type and structure without 
  unexpected results.  You can extend the behavior of equiv() by patching 
  properties on the equiv() function.  
  
  equiv.typeOf() is a function that will return the equivalence for a 
  particular object.  You can support new types by overriding this method to
  provide your own custom types.
  
  equiv.fn is a hash that contains equivalence tests for various types as 
  returned by equiv.typeOf().  You can change an existing implementation by
  replacing it or add support for new custom types by adding them here and 
  overriding equiv.typeOf().
  
  Note that this code cannot detect cyclical references.  If you include 
  cyclical references in your data structure equiv() may go into an infinite
  loop.
  
  @param {Object} object1
    the first object to compare
    
  @param {Object} object2
    the second object to compare
    
  @param {Object...} object3
    (optional) zero or more additional objects to also compare
    
  @returns {Boolean} true if all passed objects have deep equivalence
*/
Ct.equiv = function(object1, object2, object3) {
  var ret = true, len = arguments.length, t1, idx;
  
  t1 = Ct.equiv.typeOf(object1);
  if (len<2) len = 2; // always look at two args even if one is undefined
  for(idx=1;ret && (idx<len);idx++) ret = _equiv(t1, object1, arguments[idx]);
  return ret ;
};

_equiv = function _equiv(aType, a, b) {
  var func;
  
  // handle some fast fail cases
  if (a === b) return true; 
  if (aType !== Ct.equiv.typeOf(b)) return false;

  // look up comparison function
  func = Ct.equiv.fn[aType];
  if (!func) throw "SC.equiv() does not have comparison for type " + aType;
  return func(a,b);
};

function isArgs (object) {
  return Object.prototype.toString.call(object) == '[object Arguments]';
}

Ct.equiv.typeOf = function typeOf(o) {
  if (typeof o === "string") return "string";
  else if (typeof o === "boolean") return "boolean";
  else if (typeof o === "number") return isNaN(o) ? "nan" : "number";
  else if (typeof o === "undefined") return "undefined";

  // consider: typeof null === object
  else if (o === null) return "null";

  // consider: typeof [] === object
  else if (o instanceof Array) return "array";
  
  // consider: typeof new Date() === object
  else if (o instanceof Date) return "date";

  // consider: /./ instanceof Object;
  //           /./ instanceof RegExp;
  //          typeof /./ === "function"; // => false in IE and Opera,
  //                                          true in FF and Safari
  else if (o instanceof RegExp) return "regexp";

  else if (o instanceof Number) return "number";
  
  else if (o instanceof String) return "string";
  
  else if (o instanceof Boolean) return "boolean";
  
  else if (typeof o === "object") return isArgs(o) ? "arguments": "object";

  else if (o instanceof Function) return "function";
  else return "unknown";
};

// for string, boolean, number and null
function useBasicEquality(b, a) { 
  return a == b;
}

var pHasOwnProperty = Object.prototype.hasOwnProperty;

// for objects and arrays - not arguments
function useObjectEquiv(a, b) {
  
  // let objects implement their own equivalence test.  use it if available
  if (b && b.isEqual && ('function' === typeof b.isEqual)) {
    return b.isEqual(a);
  }
  
  // handle a few fast cases
  if (a.prototype !== b.prototype) return false;
  if (a.constructor !== b.constructor) return false;
  
  // first collect all keys.  Don't used Object.keys() in case we are on a
  // platform without it.
  var keys = [], key, idx, len, ptype;
  for(key in a) {
    if (!pHasOwnProperty.call(a, key)) continue;
    if (!pHasOwnProperty.call(b, key)) return false; // b does not have a.key
    keys.push(key);
  }

  // next, make sure b does not have extra keys
  len = 0 ;
  for(key in b) {
    if (!pHasOwnProperty.call(b,key)) continue;
    len++;
  }
  
  if (len !== keys.length) return false; 

  // has all the same keys, now test each individual keys
  len = keys.length;
  for(idx=0;idx<len;idx++) {
    key = keys[idx];
    ptype = Ct.equiv.typeOf(a[key]);
    
    // methods of objects should generally be ignored as long as the other
    // object also has a method there.  This is because we might end up with
    // two instances - both with methods added that are equivalent but 
    // different instances and therefore would fail a normal test.
    if (ptype === 'function') {
      if (Ct.equiv.typeOf(b[key]) !== 'function') return false;
    } else if (!_equiv(ptype, a[key], b[key])) return false;
  }
  
  return true;
}

var callers = []; // stack to decide between skip/abort functions

Ct.equiv.fn = {
  
  // these primitives will use a truthy == comparison instead of strict === 
  // because we assume typeOf() has already filtered out any objects that 
  // shouldn't be massaged into the same type.
  //
  "string": useBasicEquality,
  "boolean": useBasicEquality,
  "number": useBasicEquality,
  "null": useBasicEquality,
  "undefined": useBasicEquality,

  "nan": function (a) {
      return isNaN(a);
  },

  "date": function (a, b) {
    return a.getTime() === b.getTime();
  },

  "regexp": function (a,b) {
    return a.source === b.source && // the regex itself
           a.global === b.global && // and its modifers (gmi) ...
           a.ignoreCase === b.ignoreCase &&
           a.multiline === b.multiline;
  },

  "function": function (a, b) {
    return a === b ;
  },

  "arguments": function(a,b) {
    if (a.length !== b.length) return false;
    var ret = true, len = a.length, idx;
    for(idx=0;ret && (idx<len);idx++) ret = _equiv(a[idx], b[idx]);  
    return ret ;
  },
  
  "array": useObjectEquiv,
  "object": useObjectEquiv
};

;});
/* >>>>>>>>>> BEGIN source/lib/system/ext.js */
tiki.module('core_test:system/ext',function(require,exports,module,tiki){// ==========================================================================
// Project:   SproutCore Unit Testing Library
// Copyright: ©2006-2009 Sprout Systems, Inc. and contributors.
//            Portions ©2008-2009 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================

// These utility methods are included from the SproutCore's foundation to 
// make it easier write unit tests.  They only install themselves if a method
// has not already been defined.

if (!String.prototype.camelize) {
  String.prototype.camelize = function camelize() {
    var ret = this.replace(SC.STRING_TITLEIZE_REGEXP, 
      function(str,separater,character) { 
        return (character) ? character.toUpperCase() : '' ;
      }) ;
    var first = ret.charAt(0), lower = first.toLowerCase() ;
    return (first !== lower) ? (lower + ret.slice(1)) : ret ;
  };
}

if (!String.prototype.trim) {
  String.prototype.trim = function trim() {
    return this.replace(/^\s+|\s+$/g,"");
  } ;
}

if (!String.prototype.fmt) {
  String.prototype.fmt = function fmt() {
    // first, replace any ORDERED replacements.
    var args = arguments;
    var idx  = 0; // the current index for non-numerical replacements
    return this.replace(/%@([0-9]+)?/g, function(s, argIndex) {
      argIndex = (argIndex) ? parseInt(argIndex,0)-1 : idx++ ;
      s =args[argIndex];
      return ((s===null) ? '(null)' : (s===undefined) ? '' : s).toString(); 
    }) ;
  } ;
}

if (!Array.prototype.uniq) {
  Array.prototype.uniq = function uniq() {
    var ret = [], len = this.length, item, idx ;
    for(idx=0;idx<len;idx++) {
      item = this[idx];
      if (ret.indexOf(item) < 0) ret.push(item);
    }
    return ret ;
  };
}

if (!String.prototype.w) {
  String.prototype.w = function w() { 
    var ary = [], ary2 = this.split(' '), len = ary2.length ;
    for (var idx=0; idx<len; ++idx) {
      var str = ary2[idx] ;
      if (str.length !== 0) ary.push(str) ; // skip empty strings
    }
    return ary ;
  };
};});
/* >>>>>>>>>> BEGIN source/lib/system/module.js */
tiki.module('core_test:system/module',function(require,exports,module,tiki){// ==========================================================================
// Project:   CoreTest Unit Testing Library
// Copyright: ©2010 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================


var utils = require('utils'),
    Ct = require('core');

require('system/test'); // add Ct.Test

/**
  Describes a single test module.  A test module contains an array of tests 
  which can be scheduled in the overall plan.  It may also contain a setup
  and teardown function to run before/after each test.
*/
Ct.Module = utils.extend({
  
  name: null,
  
  init: function(name) {
    this.name = name;
    this.tasks = [] ; // empty set of tests belonging to this module
  },
  
  /**
    Returns the module's parent when used as a getter, or updates the module
    parent when used as a setter.  A parent may be a Ct.Plan or another 
    module.
    
    @param {Ct.Plan|Ct.Module} newParent
      (Optional) new parent
      
    @returns {Ct.Plan|Ct.Module} parent as getter, receiver as setter
  */
  parent: function(newParent) {
    if (arguments.length===0) return this._parent;
    this._parent = newParent;
    return this;
  },
  
  /**
    The plan that owns the current module.  Computed by asking the module 
    parent.
    
    @returns {Ct.Plan} plan
  */
  plan: function() {
    var parent = this.parent();
    return parent ? parent.plan() : null;
  },
  
  /**
    Set or get the current setup handler for the module.
  */
  setup: function(handler) {
    if (arguments.length===0) return this._setup;
    this._setup = handler;
    return this;
  },
  
  /**
    Set or get the current teardown handler for the module
  */
  teardown: function(handler) {
    if (arguments.length===0) return this._teardown;
    this._teardown = handler;
    return this;
  },
  
  /**
    Add a new test to the module.  If you pass just a function, this will
    create a new Ct.Test object to contain it.
  */
  test: function(testName, handler) {
    if (typeof testName === utils.T_STRING) {
      testName = new Ct.Test(testName, handler);
    }
    testName.module(this);
    this.tasks.push(testName);
  },
  
  /**
    Add a new submodule to the current module.  The returned module instance
    should be used to schedule additional tests.
    
    @param {String|Ct.Module} moduleName
      A module object to add to the current task to a human readable 
      description of the module name to create a new module
      
    @param {Hash} opts
      (Optional) hash containing a default setup/teardown method
      
    @returns {Ct.Module} the new module
  */
  module: function(moduleName, opts) {
    var mod;
    if ('string' === typeof moduleName) {
      mod = new Ct.Module(moduleName);
      if (opts && opts.setup) mod.setup(opts.setup);
      if (opts && opts.teardown) mod.teardown(opts.teardown);
    }
    
    mod.parent(this);
    this.tasks.push(mod); // add to task list
    return mod;
  },
  
  /**
    Returns an array containing the ordered list of modules.  Used by the 
    tests to send to the logger.
    
    @returns {Array} array of module names
  */
  moduleContext: function() {
    var par = this.parent(),
        ret = (par && par.moduleContext) ? par.moduleContext() : null;
    if (!ret) ret = [];
    ret.push(this.name);
    return ret ;
  },
  
  /**
    Schedules all the tests in the current module by attaching to the passed
    promise.  Returns a new promise that should form the room of the plan
    schedule.
  */
  schedule: function(pr, filter) {
    
    // skip if a filter is applied and this module is not included
    if (filter && !filter[this.name]) return pr; 
    else if (filter) filter = filter[this.name];

    var tasks = this.tasks,
        len   = tasks.length,
        idx;
    for(idx=0;idx<len;idx++) pr = tasks[idx].schedule(pr, filter);
    return pr;
  }
  
});
;});
/* >>>>>>>>>> BEGIN source/lib/system/plan.js */
tiki.module('core_test:system/plan',function(require,exports,module,tiki){// ==========================================================================
// Project:   CoreTest Unit Testing Library
// Copyright: ©2010 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================


var utils = require('utils'),
    Promise = require('promise', 'tiki'),
    Ct = require('core');
    
require('system/module'); // add Ct.Module

/**
  Describes a test plan.  A plan contains modules and tests.  To run it, call
  Ct.run(plan), which will generate a schedule to execute based on the 
  promises.
  
  @extends Object
*/
Ct.Plan = utils.extend({

  init: function(name) {
    this.name = name;
    this.modules = [];
  },

  /**
    Always returns receiver.  Used by modules to lookup the plan.
    
    @returns {Ct.Plan} receiver
  */
  plan: function() {
     return this;
  },
  
  // current module being described
  currentModule: null,

  /**
    Get or set the logger for the plan.  If you do not set a logger, then 
    a plan will use the logger defined at Ct.logger or create a new 
    Ct.DefaultLogger instead.
    
    @param {Ct.DefaultLogger} newLogger
      (Optional) new logger.  Instance must subclass Ct.DefaultLogger
      
    @returns {Ct.DefaultLogger|Ct.Plan} logger as getter, receiver as setter
  */
  logger: function(newLogger) {
    var ret, DefaultLogger;
    
    if (arguments.length>0) {
      this._logger = newLogger;
      return this;
      
    } else {
      ret = this._logger;
      if (!ret) ret = this._logger = (Ct.logger || Ct.defaultLogger);
      return ret ; 
    }
  },
  
  /**
    Add a new module to the plan.  Additional calls to setup(), teardown(),
    and test() will be passed along to this module until you call module()
    again to start a new module.
    
    @param {String} moduleName
      Human readable description of the new module
      
    @param {Hash} opts 
      (Optional) hash containing a setup and/or teardown property to be set
      as a the standard setup/teardown handlers
      
    @returns {Ct.Module|Ct.Plan} current module as getter, plan as setter
  */
  module: function(moduleName, opts) {
    var mod ;

    if (arguments.length === 0) return this._module();
    
    if (utils.T_STRING === typeof moduleName) {
      mod = new Ct.Module(moduleName);
    } else mod = moduleName ;
    
    mod.parent(this);
    this.currentModule = mod;
    this.modules.push(mod);

    if (opts && opts.setup) this.setup(opts.setup);
    if (opts && opts.teardown) this.teardown(opts.teardown);
    return this.currentModule ;
  },
  
  
  _module: function() {
    if (!this.currentModule) this.module('default');
    return this.currentModule;
  },

  /**
    Configure the setup handler for the current module.  If no module is 
    currently defined, a new module will be opened named "default" and the 
    setup handler will be added to it.
    
    @param {Function} handler 
      The setup handler
      
    @returns {Ct.Plan} receiver
  */
  setup: function(handler) {
    this._module().setup(handler);
    return this;
  },

  /**
    Configure the teardown handler for the current module.  If no module is 
    currently defined, a new module will be opened named "default" and the 
    setup handler will be added to it.
    
    @param {Function} handler 
      The teardown handler
      
    @returns {Ct.Plan} receiver
  */
  teardown: function(handler) {
    this._module().teardown(handler);
    return this;
  },

  /**
    Add a test to the current module.  If no module is currently set, starts 
    a new module named "defaulted" and adds the test to it.
    
    @param {String} testName
      Human readable description of the test
      
    @param {Function} handler 
      The test function
      
    @returns {Ct.Plan} receiver
  */
  test: function(testName, handler) {
    this._module().test(testName, handler);
    return this;
  },
  
  /**
    Runs the plan outside of the common CoreTest schedule.  Normally you will
    not use this method; it is really intended for testing the CoreTest API 
    itself.
    
    Instead you should always use CoreTest.run(plan) to schedule the plan as
    part of the global CoreTest schedule.
    
    @param {Hash} filter
      (Optional) nested hash of module names and tests to filter
      
    @returns {Promise} the final promise for the plan schedule 
  */
  run: function(filter) {
    var pr  = utils.Promise.create(),
        ret = this.schedule(pr, filter);
        
    // cleanup memory when finished
    ret.then(function() { pr.destroy(); });
    pr.resolve();

    return ret;
  },

  /**
    Adds the current plan to the schedule represented by the promise, 
    optionally filtering tests according to the passed nested hash.  Normally
    you do not call this method directly.  It is called instead when you run
    the plan.
    
    @param {Promise} pr
      Initial schedule promise
      
    @param {Hash} filter
      (Optional) nested hash of module names and tests to filter
      
    @returns {Promise} new schedule promise to replace the one passed in
  */
  schedule: function(pr, filter) {
    
    var modules = this.modules,
        len   = modules.length,
        idx;
        
    if (len<=0) return pr; // nothing to do
    
    pr = pr.then(this, this._begin, this._begin);
    for(idx=0;idx<len;idx++) pr = modules[idx].schedule(pr, filter);
    pr = pr.then(this, this._end, this._end);
    
    return pr;
  },
  
  _begin: function() {
    Ct.runningPlan = this;
    this.logger().planDidBegin(this.name);
  },
  
  _end: function() {
    this.logger().planDidEnd(this.name);
    Ct.runningPlan = null;
  }
  
});
;});
/* >>>>>>>>>> BEGIN source/lib/system/stub.js */
tiki.module('core_test:system/stub',function(require,exports,module,tiki){// ==========================================================================
// Project:   SproutCore Unit Testing Library
// Copyright: ©2006-2009 Sprout Systems, Inc. and contributors.
//            Portions ©2008-2009 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================
/*globals CoreTest */

var CoreTest = require('core');


/**
  Returns a stub function that records any passed arguments and a call
  count.  You can pass no parameters, a single function or a hash.  
  
  If you pass no parameters, then this simply returns a function that does 
  nothing but record being called.  
  
  If you pass a function, then the function will execute when the method is
  called, allowing you to stub in some fake behavior.
  
  If you pass a hash, you can supply any properties you want attached to the
  stub function.  The two most useful are "action", which is the function 
  that will execute when the stub runs (as if you just passed a function), 
  and "expect" which should evaluate the stub results.
  
  In your unit test you can verify the stub by calling stub.expect(X), 
  where X is the number of times you expect the function to be called.  If
  you implement your own test function, you can actually pass whatever you
  want.
  
  Calling stub.reset() will reset the record on the stub for further 
  testing.

  @param {String} name the name of the stub to use for logging
  @param {Function|Hash} func the function or hash
  @returns {Function} stub function
*/
CoreTest.stub = function(name, func) {  

  // normalize param
  var attrs = {};
  if (typeof func === "function") {
    attrs.action = func;
  } else if (typeof func === "object") {
    attrs = func ;
  }

  // create basic stub
  var ret = function() {
    ret.callCount++;
    
    // get arguments into independent array and save in history
    var args = [], loc = arguments.length;
    while(--loc >= 0) args[loc] = arguments[loc];
    args.unshift(this); // save context
    ret.history.push(args);
    
    return ret.action.apply(this, arguments);
  };
  ret.callCount = 0 ;
  ret.history = [];
  ret.stubName = name ;

  // copy attrs
  var key;
  for(key in attrs) {
    if (!attrs.hasOwnProperty(key)) continue ;
    ret[key] = attrs[key];
  }

  // add on defaults
  if (!ret.reset) {
    ret.reset = function() {
      this.callCount = 0;
      this.history = [];
    };
  }
  
  if (!ret.action) {
    ret.action = function() { return this; };
  }
  
  if (!ret.expect) {
    ret.expect = function(callCount) {
      if (callCount === YES) {
        ok(this.callCount > 0, utils.fmt("%@ should be called at least once", this.stubName));
      } else {
        if (callCount === NO) callCount = 0;
        equals(this.callCount, callCount, utils.fmt("%@ should be called X times", this.stubName));
      }
    };
  }
  
  return ret ;
};

;});
/* >>>>>>>>>> BEGIN source/lib/system/suite.js */
tiki.module('core_test:system/suite',function(require,exports,module,tiki){// ==========================================================================
// Project:   SproutCore Costello - Property Observing Library
// Copyright: ©2006-2009 Sprout Systems, Inc. and contributors.
//            Portions ©2008-2009 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================

var CoreTest = require('core'),
    utils    = require('utils');

require('system/plan'); 

/** @class

  A test Suite defines a group of reusable unit tests that can be added to a 
  test plan at any time by calling the generate() method.  Suites are most
  useful for defining groups of tests that validate compliance with a mixin.
  You can then generate customized versions of the test suite for different
  types of objects to ensure that both the mixin and the object implementing
  the mixin use the API properly.
  
  h1. Using a Suite
  
  To use a Suite, call the generate() method on the suite inside on of your
  unit test files.  This will generate new modules and tests in the suite
  and add them to your test plan.
  
  Usually you will need to customize the suite to apply to a specific object.
  You can supply these customizations through an attribute hash passed to the
  generate() method.  See the documentation on the specific test suite for
  information on the kind of customizations you may need to provide.
  
  h2. Example
  
  {{{
    // generates the SC.ArrayTestSuite tests for a built-in array.
    SC.ArrayTests.generate('Array', {
      newObject: function() { return []; }
    });
  }}}
  
  h1. Defining a Suite
  
  To define a test suite, simply call the extend() method, passing any 
  attributs you want to define on the stuie along with this method.  You can
  then add functions that will define the test suite with the define() method.
  
  Functions you pass to define will have an instance of the test suite passed
  as their first parameter when invoked.

  h2. Example 
  
  {{{
    SC.ArrayTests = CoreTest.Suite.create("Verify SC.Array compliance", {
      
      // override to generate a new object that implements SC.Array
      newObject: function() { return null; }
    });
    
    SC.ArrayTests.define(function(T) {
      T.module("length tests");
      
      test("new length", function() {
        equals(T.object.get('length'), 0, 'array length');
      });
      
    });
  }}}
  
  @since SproutCore 1.0
  
*/
CoreTest.Suite = /** @scope CoreTest.Suite.prototype */ {

  /**
    Call this method to define a new test suite.  Pass one or more hashes of
    properties you want added to the new suite.  
    
    @param {Hash} attrs one or more attribute hashes
    @returns {CoreTest.Suite} subclass of suite.
  */
  create: function(desc, attrs) {
    var len = arguments.length,
        ret = utils.beget(this),
        idx;
        
    // copy any attributes
    for(idx=1;idx<len;idx++) utils.mixin(ret, arguments[idx]);
    
    if (desc) ret.basedesc = desc;
    
    // clone so that new definitions will be kept separate
    ret.definitions = ret.definitions.slice();
    
    return ret ;
  },

  /**
    Generate a new test suite instance, adding the suite definitions to the 
    current test plan.  Pass a description of the test suite as well as one or
    more attribute hashes to apply to the test plan.
    
    The description you add will be prefixed in front of the 'desc' property
    on the test plan itself.
    
    @param {String} desc suite description
    @param {Hash} attrs one or more attribute hashes
    @returns {CoreTest.Suite} suite instance
  */
  generate: function(desc, attrs) {
    var len = arguments.length,
        ret = utils.beget(this),
        idx, defs;
        
    // apply attributes - skip first argument b/c it is a string
    for(idx=1;idx<len;idx++) utils.mixin(ret, arguments[idx]);    
    ret.subdesc = desc ;
    
    // invoke definitions
    defs = ret.definitions ;
    len = defs.length;
    for(idx=0;idx<len;idx++) defs[idx].call(ret, ret);
    
    return ret ;
  },
  
  /**
    Adds the passed function to the array of definitions that will be invoked
    when the suite is generated.
    
    The passed function should expect to have the TestSuite instance passed
    as the first and only parameter.  The function should actually define 
    a module and tests, which will be added to the test suite.
    
    @param {Function} func definition function
    @returns {CoreTest.Suite} receiver
  */
  define: function(func) {
    this.definitions.push(func);
    return this ;
  },
  
  /** 
    Definition functions.  These are invoked in order when  you generate a 
    suite to add unit tests and modules to the test plan.
  */
  definitions: [],
  
  /**
    Generates a module description by merging the based description, sub 
    description and the passed description.  This is usually used inside of 
    a suite definition function.
    
    @param {String} str detailed description for this module
    @returns {String} generated description
  */
  desc: function(str) {
    return this.basedesc.fmt(this.subdesc, str);
  },
  
  /**
    The base description string.  This should accept two formatting options,
    a sub description and a detailed description.  This is the description
    set when you call extend()
  */
  basedesc: "%@ > %@",
  
  /**
    Default setup method for use with modules.  This method will call the
    newObject() method and set its return value on the object property of 
    the receiver.
  */
  setup: function() {
    this.object = this.newObject();
  },
  
  /**
    Default teardown method for use with modules.  This method will call the
    destroyObejct() method, passing the current object property on the 
    receiver.  It will also clear the object property.
  */
  teardown: function() {
    if (this.object) this.destroyObject(this.object);
    this.object = null;
  },
  
  /**
    Default method to create a new object instance.  You will probably want
    to override this method when you generate() a suite with a function that
    can generate the type of object you want to test.
    
    @returns {Object} generated object
  */
  newObject: function() { return null; },
  
  /**
    Default method to destroy a generated object instance after a test has 
    completed.  If you override newObject() you can also overried this method
    to cleanup the object you just created.
    
    Default method does nothing.
  */
  destroyObject: function(obj) { 
    // do nothing.
  },
  
  /**
    Generates a default module with the description you provide.  This is 
    a convenience function for use inside of a definition function.  You could
    do the same thing by calling:
    
    {{{
      var T = this ;
      module(T.desc(description), {
        setup: function() { T.setup(); },
        teardown: function() { T.teardown(); }
      }
    }}}
    
    @param {String} desc detailed descrition
    @returns {CoreTest.Suite} receiver
  */
  module: function(desc) {
    var T = this ;
    CoreTest.module(T.desc(desc), {
      setup: function() { T.setup(); },
      teardown: function() { T.teardown(); }
    });
  }
  
};
;});
/* >>>>>>>>>> BEGIN source/lib/system/test.js */
tiki.module('core_test:system/test',function(require,exports,module,tiki){// ==========================================================================
// Project:   CoreTest Unit Testing Library
// Copyright: ©2010 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================


var utils       = require('utils'),
    Ct          = require('core'),
    Promise     = require('promise', 'tiki'),
    Assertable  = require('assert').xCoreTestAssertable;

require('system/dump'); // add Ct.dump
 
/**
  Describes a single test.  To run a test must be scheduled as part of a 
  module.  If the test is async then the handler will be passed a promise 
  which is much resolve or cancel for the test run to continue.
*/
Ct.Test = utils.extend(Assertable, {
  
  /**
    Human readable test description.  This will be used when logging test
    results.
    
    @param {String}
  */
  name: null,

  /**
    true if this test should be run asynchronously.  All tests have a promise 
    associated with them that must resolve when they finish execution to 
    cause the next step to begin.  When you run a test asynchronously, you
    must resolve this promise yourself.  Otherwise it will resolve 
    automatically when your test handler finishes executing.
    
    Defaults to false.
    
    @property {Boolean}  
  */
  async: false,

  /**
    Set an alternate context to run the handler.  You can use this to control
    the value of "this" when your handler runs.  The setup and teardown 
    functions for a test will also have this same context when run.
    
    If no context is set, the test object itself will be used as the value of
    "this".
    
    @property {Object}
  */
  context: null,

  /**
    The current test execution mode.  Normally this value will remain 
    Ct.PLANNING_MODE, but when the test is actually running it will be one of
    Ct.SETUP_MODE, Ct.TEARDOWN_MODE, or Ct.TEST_MODE.  This is mostly used by
    the logger API to determine how a test result should be logged.
  */
  mode: Ct.PLANNING_MODE,

  /**
    Returns the plan that this test current belongs to.  Computed from the
    plan's module.
    
    @returns {Ct.Plan}
  */
  plan: function() {
    var mod = this.module();
    return mod ? mod.plan() : null;
  },

  /**
    Returns the module the test belongs to.  You can also set the module by
    passing it as a parameter to this method.  If you set the module returns
    the receiver.
    
    @param {Ct.Module} newModule
      (Optional) new module to set for this test
      
    @returns {Ct.Module|Ct.Test} module as getter, receiver as setter
  */
  module: function(newModule) {
    if (arguments.length===0) return this._module;    
    this._module = newModule;
    return this;
  },

  /**
    Returns the current logger for the test.  This is actually computted from
    the current plan.
    
    @returns {Ct.DefaultLogger} CoreTest logger instance
  */
  logger: function() {
    return this.plan().logger();  
  },
  

  // ..........................................................
  // PRIMITIVE API
  // 

  /**
    Set the number of expected assertions for the current phase.
  */
  expect: function(assertCount) {
    if (arguments.length === 0) return this._expectedAsserts;
    this._expectedAsserts = assertCount;
    return this;
  },
  
  /** 
    Primitive to assert a given test.  This will invoke the proper callback on
    the test plan logger as long as you are actually running a test.  
    
    @param {Boolean} pass
      truthy value indicating whether the assertion should pass or fail
    
    @param {String} message
      (Optional) log message describing the assertion
      
    @param {Object} actual
      (Optinal) actual value to log [no comparison will be performed]

    @param {Object} expected
      (Optional) expected value to log [no comparison will be performed]
    
    @returns {Ct.Test} receiver
  */
  assert: function(pass, message, actual, expected) {
    
    if (this.mode === Ct.PLANNING_MODE) {
      throw utils.fmt("Cannot assert while test is not running (%@)", this);
    }
    
    var logger   = this.logger(),
        plan     = this.plan(),
        raises   = plan ? plan.throwsOnFailure : false,
        showVars = arguments.length > 2,
        str;

    if (!pass) this.didFail = true; 
    this._actualAsserts++;
    
    if (showVars) {
      actual = Ct.dump(actual);
      expected = Ct.dump(expected);
    }

    if (pass) {
      str = showVars ? '%@ (actual = %@, expected = %@)' : '%@';
      logger.pass(this, utils.fmt(str, message, actual, expected));

    } else if (raises) {
      throw new Ct.AssertionError(actual, expected, message);

    } else {
      str = showVars ? '%@ (actual = %@, expected = %@)' : '%@';
      logger.fail(this, utils.fmt(str, message, actual, expected));
    }
    
    return this;
  },
  
  /**
    Primitive to log an error messages.  This is invoked if the test 
    encounters an unexpected exception while running.
  
    @param {String|Error} message message or exception to log
    @returns {Ct.Test} receiver
  */
  error: function(message) {
    if (this.mode === Ct.PLANNING_MODE) {
      throw utils.fmt("Cannot assert error while test is not running (%@)", this);
    }

    this.didFail = true ;
    this.logger().error(this, message);
    return this;
  },

  /**
    Adds some info to the logger.  Not a test - just some output.
    
    @param {String} the message to log
    @returns {Ct.Test} receiver
  */
  info: function(message) {
    if (this.mode === Ct.PLANNING_MODE) {
      throw utils.fmt("Cannot log info while test is not running (%@)", this);
    }
    this.logger().info(this, message);
    return this;
  },
  
  /**
    Schedules the current test to run after the passed promise resolves.  This
    will invoke a setup/teardown function, returning the final promise that 
    will resolve when the test as finished running.
    
    Normally you will not call this directly; it will be called by the owner
    module during scheduling.
    
    @param {Promise} pr the promise
    @param {Hash} filter 
      (Optional) hash of test names and a boolean value to select which to run
      
    @returns {Promise} the next promise for the schedule
  */
  schedule: function(pr, filter) {
    
    // skip if filter is included and this test is not part of it
    if (filter && !filter[this.name]) return pr;
    
    var mod     = this.module(),
        testId  = mod.name + ':' + this.name,
        ret     = Promise.create(testId), // resolves when test has finished
        setupPr = Promise.create(testId+'::setup'),
        testPr  = Promise.create(testId+'::test'),
        teardownPr = Promise.create(testId+'::teardown'),
        setupHandler, teardownHandler, testHandler, breakPr;
        
    setupHandler = this._phase(Ct.SETUP_MODE, mod.setup(), setupPr);
    testHandler = this._phase(Ct.TEST_MODE, this.handler, testPr);
    teardownHandler = this._phase(Ct.TEARDOWN_MODE, mod.teardown(), teardownPr);

    // force take a break between each test to avoid timer overruns
    breakPr = Promise.create();
    pr.then(function() { setTimeout(function() { breakPr.resolve(); }, 1); });
    
    breakPr.then(this, setupHandler);
    setupPr.then(this, testHandler, teardownHandler);
    testPr.then(this, teardownHandler, teardownHandler);
    teardownPr.then(ret, ret.resolve, ret.resolve);
    return ret ;
  },
  
  // ..........................................................
  // PRIVATE METHODS
  // 

  /**
    @constructor
    
    Set the test name and (optionally) handler and async value.  A test 
    must have a handler to actually run.
    
    @param {String} name
      Human readable description of the test
      
    @param {Function} handler
      A handler to invoke when running the test
    
    @returns {void}
  */
  init: function(name, handler) {
    this.name = this.testName = name; // testName is needed for logging
    if (arguments.length>1) {
      this.handler = handler;
    }
    
    this.tests = [] ; // empty set of tests belonging to this module
  },
  
  toString: function() {
    var mod = this.module();
    mod = (mod && mod.name) ? mod.name : '(unknown module)';
    return utils.fmt("Ct.Test<%@:%@>", mod, this.name);  
  },
  
  /** @private
    Generate a handler function to invoke a particlar phase of the test 
    (setup, teardown or the test itself)
  */
  _phase: function(mode, handler, pr) {
    var test = this;
    return function() {

      // handle cleanup
      var mod = this.module(), hpr;
      
      if (mod && mod.moduleContext) mod = mod.moduleContext();
      this.moduleNames = mod || [];

      this.mode = mode;
      this.didFail = false ; 
      this._actualAsserts = 0 ;
      this._expectedAsserts = -1;
      Ct.currentTest = this;
      
      try { 
        if (handler) hpr = handler.call(test.context || test, test);
      } catch(e) {
        this.error(e);
        this.didFail = true;
        pr.cancel();
      }

      // treat async if a promise was returned.  schedule followups
      if (hpr && hpr.then) {
        this._hookupAsync(hpr, pr);
      } else {
        if (!this.didFail) this._verifyAsserts(); 
        this.mode = Ct.PLANNING_MODE;
        Ct.currentTest = null;
        if (this.didFail) pr.cancel(); else pr.resolve();
      }
    };
  },
  
  _bind: function(f) {
    var target = this;
    return function() {
      return f.apply(target, arguments);
    };
  },
  
  // listen for success/cancel handlers on promise.  Note that this must use
  // only the CommonJS promise API since we don't know the source of the 
  // promise that will be returned.
  _hookupAsync: function(hpr, pr) {
    var ok = this._bind(function() {
      this._verifyAsserts();
      this.mode = Ct.PLANNING_MODE ;
      Ct.currentTest = null;
      pr.resolve();
    });

    var cancel = this._bind(function() {
      this.mode = Ct.PLANNING_MODE;
      Ct.currentTest = null;
      pr.cancel();
    });
    
    hpr.then(ok, cancel);
  },
  
  // if test is configured to expect some assertions, test for them.  If the
  // test has already failed for some reason, don't bother since it will 
  // fail anyway
  _verifyAsserts: function() {
    if (this.didFail) return; // nothing to do
    if (this._expectedAsserts>=0) {
      this.equal(this._actualAsserts, this._expectedAsserts, 'expected assertions');
    }
  }
  
});
;});
/* >>>>>>>>>> BEGIN source/lib/test.js */
tiki.module('core_test:test',function(require,exports,module,tiki){// ==========================================================================
// Project:   CoreTest Unit Testing Library
// Copyright: ©2010 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================

var Ct    = require('core'),
    utils = require('utils'),
    trimmer = (/^test\s*/) ;

require('system/plan'); // add Ct.Plan
    
/** 
  @file 
  
  Implements the CommonJS API to run a suite of tests.  This looks on the 
  passed object hash and builds a "plan" which can be run to the standard 
  logger.  Looks for a property called "logger" to select the logger.  
  
  Otherwise will use the default logger defined on CoreTest.
*/

// iterates through a single hash and adds any tests. recursively called for
// any nested subgroups.  subgroups are treated as modules
function addTests(mod, tests) {
  
  var key, value ;
  for(key in tests) {

    // limit processing to spec...
    if ((key==='test') || (key.indexOf('test')!==0)) continue;
    
    value = tests[key];
    switch(utils.typeOf(value)) {
      case utils.T_HASH:
        addTests(mod.module(key), value);
        break;
        
      case utils.T_FUNCTION:
        mod.test(key, value);
        break;
        
      default:
        throw utils.fmt("test '%@' must be a function or hash (was %@)", key, utils.typeOf(value));
    }
  }
}

exports.run = function(tests) {
  
  var plan = new Ct.Plan(tests.name || 'unnamed plan');
  if (tests.logger) plan.logger(tests.logger);
 
  // add tests to a default module
  addTests(plan.module(), tests);
  
  Ct.run(plan);
};
;});
/* >>>>>>>>>> BEGIN source/lib/utils.js */
tiki.module('core_test:utils',function(require,exports,module,tiki){// ==========================================================================
// Project:   SproutCore Unit Testing Library
// Copyright: ©2006-2009 Sprout Systems, Inc. and contributors.
//            Portions ©2008-2009 Apple Inc. All rights reserved.
// License:   Licened under MIT license (see license.js)
// ==========================================================================

// Borrowed from tiki:index.  We don't want to depend on tiki for any actual
// code so that this can be used in server-side code

var tiki = require('tiki'); // get tiki package
exports = module.exports = tiki.beget(tiki); // borrow tiki API

/** Borrowed from SproutCore Runtime Core */
exports.fmt = function fmt(str) {
  // first, replace any ORDERED replacements.
  var args = arguments;
  var idx  = 1; // the current index for non-numerical replacements
  return str.replace(/%@([0-9]+)?/g, function(s, argIndex) {
    argIndex = (argIndex) ? parseInt(argIndex,0) : idx++ ;
    s =args[argIndex];
    return ((s===null) ? '(null)' : (s===undefined) ? '' : s).toString(); 
  }) ;
};

exports.Promise = require('promise', 'tiki');;});
; tiki.script('core_test:en/7cad3ee1925b727665f4dadcb6cc4ca4cd6b94d2/javascript.js');