/* vim:set ts=2 sw=2 sts=2 et: */
/* ***** BEGIN LICENSE BLOCK *****
 * Version: MPL 1.1/GPL 2.0/LGPL 2.1
 *
 * The contents of this file are subject to the Mozilla Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with
 * the License. You may obtain a copy of the License at
 * http://www.mozilla.org/MPL/
 *
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License
 * for the specific language governing rights and limitations under the
 * License.
 *
 * The Original Code is Jetpack.
 *
 * The Initial Developer of the Original Code is
 * the Mozilla Foundation.
 * Portions created by the Initial Developer are Copyright (C) 2011
 * the Initial Developer. All Rights Reserved.
 *
 * Contributor(s):
 *   Irakli Gozalishvili <gozala@mozilla.com> (Original author)
 *
 * Alternatively, the contents of this file may be used under the terms of
 * either the GNU General Public License Version 2 or later (the "GPL"), or
 * the GNU Lesser General Public License Version 2.1 or later (the "LGPL"),
 * in which case the provisions of the GPL or the LGPL are applicable instead
 * of those above. If you wish to allow use of your version of this file only
 * under the terms of either the GPL or the LGPL, and not to allow others to
 * use your version of this file under the terms of the MPL, indicate your
 * decision by deleting the provisions above and replace them with the notice
 * and other provisions required by the GPL or the LGPL. If you do not delete
 * the provisions above, a recipient may use your version of this file under
 * the terms of any one of the MPL, the GPL or the LGPL.
 *
 * ***** END LICENSE BLOCK ***** */

"use strict";

const { EventEmitterTrait: EventEmitter } = require("../events");
const { DOMEventAssembler } = require("../events/assembler");
const { Trait } = require("../light-traits");
const { getActiveTab, getTabs, getTabContainers } = require("./utils");
const { windowIterator, isBrowser } = require("../window-utils");
const windowObserver = require("../windows/observer");

const EVENTS = {
  "TabOpen": "open",
  "TabClose": "close",
  "TabSelect": "select",
  "TabMove": "move",
  "TabPinned": "pin",
  "TabUnpinned": "unpin"
};


// Event emitter objects used to register listeners and emit events on them
// when they occur.
const observer = Trait.compose(DOMEventAssembler, EventEmitter).create({
  /**
   * Method is implemented by `EventEmitter` and is used just for emitting
   * events on registered listeners.
   */
  _emit: Trait.required,
  /**
   * Events that are supported and emitted by the module.
   */
  supportedEventsTypes: Object.keys(EVENTS),
  /**
   * Function handles all the supported events on all the windows that are
   * observed. Method is used to proxy events to the listeners registered on
   * this event emitter.
   * @param {Event} event
   *    Keyboard event being emitted.
   */
  handleEvent: function handleEvent(event) {
    this._emit(EVENTS[event.type], event.target, event);
  }
});

// Currently gecko does not dispatches any event on the previously selected
// tab before / after "TabSelect" is dispatched. In order to work around this
// limitation we keep track of selected tab and emit "deactivate" event with
// that before emitting "activate" on selected tab.
var selectedTab = null;
function onTabSelect(tab) {
  if (selectedTab !== tab) {
    if (selectedTab) observer._emit("deactivate", selectedTab);
    if (tab) observer._emit("activate", selectedTab = tab);
  }
};
observer.on("select", onTabSelect);

// We also observe opening / closing windows in order to add / remove it's
// containers to the observed list.
function onWindowOpen(chromeWindow) {
  if (!isBrowser(chromeWindow)) return; // Ignore if it's not a browser window.
  getTabContainers(chromeWindow).forEach(function (container) {
    observer.observe(container);
  });
}
windowObserver.on("open", onWindowOpen);

function onWindowClose(chromeWindow) {
  if (!isBrowser(chromeWindow)) return; // Ignore if it's not a browser window.
  getTabContainers(chromeWindow).forEach(function (container) {
    observer.ignore(container);
  });
}
windowObserver.on("close", onWindowClose);


// Currently gecko does not dispatches "TabSelect" events when different
// window gets activated. To work around this limitation we emulate "select"
// event for this case.
windowObserver.on("activate", function onWindowActivate(chromeWindow) {
  if (!isBrowser(chromeWindow)) return; // Ignore if it's not a browser window.
  observer._emit("select", getActiveTab(chromeWindow));
});

// We should synchronize state, since probably we already have at least one
// window open.
for each (let window in windowIterator()) onWindowOpen(window);

// Getting rid of all listeners when add-on is unloaded.
require("unload").when(function() { observer._events = {} });

module.exports = observer;
