/* vim: set ts=2 et sw=2 tw=80: */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/. */

#ifndef rsaencrypt_bb2048_vectors_h__
#define rsaencrypt_bb2048_vectors_h__

#include <memory>
#include "testvectors_base/test-structs.h"

/* 2048 bit key from Hubert's Bleichenbacher tests */
static const std::vector<uint8_t> kRsaBb2048{
    0x30, 0x82, 0x04, 0xbd, 0x02, 0x01, 0x01, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x04, 0xa7, 0x30, 0x82, 0x04, 0xa3, 0x02, 0x01, 0x00, 0x02, 0x82, 0x01,
    0x01, 0x00, 0xc8, 0xcc, 0x83, 0x97, 0x14, 0x09, 0x8d, 0xa5, 0x6c, 0xaa,
    0x23, 0x64, 0x0f, 0x93, 0xdc, 0x89, 0x97, 0xc1, 0x63, 0x72, 0x96, 0x8f,
    0xc1, 0xb0, 0xc6, 0xdf, 0x51, 0x13, 0xc1, 0xc9, 0x4e, 0x8b, 0x21, 0xe4,
    0x8a, 0xd2, 0x29, 0x7e, 0x65, 0x41, 0x90, 0x11, 0xb4, 0xe6, 0xd8, 0xf5,
    0xe7, 0x3b, 0x1b, 0x78, 0xb2, 0x57, 0x40, 0x03, 0x21, 0xd1, 0xef, 0x6b,
    0x60, 0x2d, 0x4e, 0xc8, 0xce, 0x8d, 0x14, 0x1c, 0x94, 0x90, 0x5e, 0xb4,
    0xad, 0x30, 0x66, 0x39, 0xa4, 0x92, 0x06, 0x53, 0x4b, 0x6e, 0x7f, 0x26,
    0x07, 0x42, 0x3e, 0x97, 0xdf, 0xfd, 0x13, 0x3c, 0x88, 0xd7, 0x21, 0x39,
    0x9d, 0xef, 0xbc, 0x7e, 0x96, 0xcc, 0xdc, 0xbd, 0x7f, 0x3a, 0xae, 0x1f,
    0xe8, 0x92, 0x71, 0x2b, 0xfb, 0x49, 0x29, 0x81, 0x7d, 0x51, 0x16, 0x66,
    0x44, 0x0a, 0x1f, 0xac, 0xb7, 0xa2, 0x08, 0xf5, 0xea, 0x16, 0x59, 0x10,
    0xad, 0xd8, 0xa3, 0xf2, 0xd4, 0x97, 0x20, 0x23, 0x60, 0xcc, 0xb6, 0x32,
    0x02, 0x4f, 0x0d, 0x07, 0x16, 0x9c, 0x19, 0x18, 0xf3, 0x16, 0xf7, 0x94,
    0xb1, 0x43, 0xae, 0xf5, 0x4e, 0xc8, 0x75, 0x22, 0xa4, 0xc0, 0x29, 0x78,
    0xf9, 0x68, 0x99, 0x80, 0xbf, 0xfb, 0xf6, 0x49, 0xc3, 0x07, 0xe8, 0x18,
    0x19, 0xbf, 0xf8, 0x84, 0x09, 0x63, 0x8d, 0x48, 0xbd, 0x94, 0xbe, 0x15,
    0x2b, 0x59, 0xff, 0x64, 0x9f, 0xa0, 0xbd, 0x62, 0x9d, 0x0f, 0xfa, 0x18,
    0x13, 0xc3, 0xab, 0xf4, 0xb5, 0x6b, 0xd3, 0xc2, 0xea, 0x54, 0x65, 0xdf,
    0xfa, 0x14, 0x58, 0x92, 0x92, 0xa9, 0xd8, 0xa2, 0x4a, 0xd2, 0x6b, 0xe7,
    0xee, 0x05, 0x10, 0x74, 0x1b, 0x63, 0x82, 0xd4, 0x3c, 0x83, 0xd5, 0xbf,
    0xa4, 0x0a, 0x46, 0x61, 0x3d, 0x06, 0x2b, 0xe4, 0x45, 0x51, 0x7d, 0xbc,
    0xaf, 0x0c, 0xb4, 0xe1, 0xa7, 0x69, 0x02, 0x03, 0x01, 0x00, 0x01, 0x02,
    0x82, 0x01, 0x00, 0x14, 0x55, 0x01, 0x0e, 0x0f, 0x2d, 0x58, 0x76, 0x63,
    0xa6, 0x66, 0xa6, 0xff, 0x1c, 0xcd, 0xbb, 0xf0, 0xed, 0xd8, 0x10, 0x06,
    0x46, 0xd0, 0x2a, 0x02, 0x39, 0x22, 0x90, 0x89, 0x92, 0xc4, 0xad, 0x39,
    0xe5, 0x56, 0x59, 0x29, 0x72, 0x6e, 0xf6, 0x50, 0x8c, 0x3a, 0x71, 0x15,
    0x8e, 0xf0, 0xb6, 0xff, 0x75, 0x1d, 0x39, 0xd0, 0x75, 0x80, 0xbb, 0x2d,
    0x2f, 0x06, 0x32, 0x10, 0x44, 0x2d, 0x06, 0x03, 0xff, 0x50, 0xdb, 0xbd,
    0x7b, 0x35, 0xfe, 0x2c, 0x9b, 0xb1, 0x9a, 0x47, 0xa1, 0xaf, 0x85, 0xa4,
    0xc2, 0x49, 0x01, 0xe0, 0x2c, 0xa8, 0xb5, 0x8b, 0x79, 0x19, 0xb2, 0x0e,
    0xdf, 0x32, 0xaa, 0xcf, 0xbf, 0x51, 0xad, 0xb4, 0xbc, 0x4b, 0x61, 0xb9,
    0xb7, 0xe9, 0x68, 0xca, 0xa4, 0xd5, 0x70, 0xf7, 0x0e, 0xf1, 0x8d, 0x80,
    0x63, 0x22, 0x88, 0x93, 0xe4, 0x7d, 0x43, 0x9e, 0xfc, 0xa7, 0x93, 0x25,
    0x9b, 0xcf, 0x2c, 0xd1, 0x08, 0xa3, 0xd8, 0x68, 0x8c, 0xdf, 0x07, 0x8e,
    0x7a, 0xc7, 0x99, 0x96, 0x9f, 0x23, 0x39, 0xd2, 0xc1, 0xf5, 0x22, 0xb9,
    0x69, 0x68, 0x46, 0x29, 0xa9, 0x33, 0xba, 0xae, 0xc2, 0x68, 0x16, 0x25,
    0xea, 0xb8, 0x4f, 0x4e, 0x56, 0xf4, 0x44, 0x7e, 0x9d, 0x88, 0xfb, 0x9a,
    0x19, 0x9c, 0xf7, 0x10, 0x23, 0xe0, 0xe2, 0x57, 0xb1, 0x44, 0x41, 0xb3,
    0x3c, 0x84, 0xd3, 0xbc, 0x67, 0xca, 0x80, 0x31, 0xd2, 0x61, 0x26, 0x18,
    0x10, 0x3a, 0x7a, 0x0a, 0x40, 0x84, 0x42, 0x62, 0xf7, 0x5d, 0x88, 0x90,
    0xcd, 0x61, 0x6e, 0x51, 0xf9, 0x03, 0x54, 0x88, 0xfd, 0x6e, 0x09, 0x9d,
    0xe8, 0xff, 0x6d, 0x65, 0xa4, 0xff, 0x11, 0x82, 0x54, 0x80, 0x7c, 0x9f,
    0x58, 0xd2, 0xfb, 0xba, 0x8b, 0xa1, 0x51, 0xdc, 0x8c, 0x68, 0xbe, 0x34,
    0x9c, 0x97, 0x7a, 0x20, 0x4e, 0x04, 0xc1, 0x02, 0x81, 0x81, 0x00, 0xf8,
    0xf5, 0xad, 0x6b, 0xa8, 0x28, 0x93, 0x1b, 0xea, 0x45, 0x9b, 0x8a, 0x3f,
    0x6d, 0xc0, 0x41, 0xd2, 0x34, 0x82, 0x40, 0x9c, 0x25, 0x71, 0xe9, 0x63,
    0xf3, 0x1f, 0x74, 0x86, 0x02, 0xa2, 0x56, 0x37, 0x1b, 0x38, 0x83, 0xed,
    0x45, 0x9e, 0xcf, 0x97, 0x05, 0x26, 0x45, 0x9e, 0xdd, 0x16, 0xe0, 0x55,
    0x22, 0xf5, 0xa4, 0x5d, 0x94, 0x75, 0x1b, 0x2e, 0xc2, 0xda, 0xf2, 0x72,
    0xc7, 0xf8, 0x81, 0x6a, 0x52, 0xc0, 0x0d, 0x18, 0x08, 0x01, 0x71, 0x63,
    0x4d, 0xa8, 0x99, 0xd7, 0x97, 0x32, 0x22, 0xf5, 0x1b, 0x93, 0x76, 0x30,
    0x54, 0x86, 0x96, 0xa9, 0xf7, 0xd8, 0xc2, 0x4a, 0x59, 0x49, 0x7c, 0x1e,
    0xfc, 0xd4, 0x55, 0xcf, 0xb9, 0x7e, 0xe8, 0x6d, 0x2b, 0x6d, 0x34, 0x97,
    0x2b, 0x33, 0x2f, 0xda, 0x30, 0x3f, 0x04, 0x99, 0x9b, 0x4e, 0xb6, 0xb5,
    0xcc, 0x0b, 0xb3, 0x3e, 0x77, 0x61, 0xdd, 0x02, 0x81, 0x81, 0x00, 0xce,
    0x7a, 0x2e, 0x3b, 0x49, 0xa9, 0x0b, 0x96, 0x33, 0x0a, 0x12, 0xdc, 0x68,
    0x2b, 0xdf, 0xbd, 0xfb, 0xae, 0x8d, 0xd6, 0xdc, 0x03, 0xb6, 0x14, 0x7a,
    0xef, 0xbd, 0x57, 0x57, 0x43, 0xf0, 0xf6, 0xda, 0x4d, 0x86, 0x23, 0x50,
    0x61, 0xb7, 0x1a, 0xfd, 0x9c, 0xad, 0x2d, 0x34, 0x02, 0x5e, 0x56, 0xac,
    0x86, 0xb0, 0xf7, 0x74, 0x3e, 0xb3, 0x5e, 0x1a, 0xcb, 0xca, 0x23, 0x78,
    0x95, 0x42, 0x44, 0x65, 0xb7, 0x06, 0xed, 0x22, 0x17, 0x5e, 0x57, 0x18,
    0xc8, 0xc7, 0x0b, 0x67, 0x03, 0xea, 0x8f, 0x6b, 0x51, 0x0f, 0x94, 0x5b,
    0xe4, 0x8e, 0x5a, 0x36, 0xbb, 0x3c, 0x3c, 0x91, 0x73, 0x2b, 0x58, 0x9d,
    0xfc, 0x05, 0xd7, 0x2d, 0x80, 0x90, 0x31, 0x94, 0x45, 0x2b, 0xda, 0x21,
    0x34, 0x86, 0x47, 0xec, 0x72, 0x94, 0x3f, 0x11, 0xa8, 0x46, 0xe6, 0x2f,
    0xae, 0xbe, 0x8e, 0xb5, 0x36, 0xb0, 0xfd, 0x02, 0x81, 0x80, 0x76, 0xfe,
    0x15, 0xf1, 0x8a, 0xe2, 0x39, 0xcd, 0xf1, 0xdf, 0x6b, 0x44, 0x5c, 0xa4,
    0xbc, 0x6b, 0xb9, 0x68, 0xd7, 0x88, 0xc2, 0x19, 0x33, 0xa4, 0xf5, 0xdc,
    0xd2, 0x80, 0x03, 0x3d, 0x67, 0x12, 0x06, 0x2c, 0xc0, 0x8a, 0x6d, 0xf2,
    0x04, 0xc1, 0xfb, 0xd0, 0xbe, 0x46, 0x30, 0x74, 0x43, 0xe6, 0xdd, 0x4a,
    0x64, 0x56, 0x37, 0x54, 0x29, 0xd4, 0xe0, 0x38, 0xca, 0x25, 0x6f, 0xaf,
    0x1c, 0x9b, 0xde, 0x91, 0xc6, 0xb1, 0x7b, 0x76, 0xf8, 0x19, 0x95, 0xf9,
    0x1c, 0x48, 0xcb, 0xbe, 0xbc, 0x7b, 0xf0, 0xe3, 0x49, 0x4c, 0x08, 0x35,
    0x9e, 0x4e, 0x8c, 0xd6, 0xa5, 0x87, 0xd7, 0xb9, 0x6d, 0x62, 0x21, 0xfd,
    0x7e, 0x0f, 0xb5, 0xc5, 0x57, 0x5f, 0x08, 0x2e, 0xe5, 0x77, 0x69, 0x79,
    0x80, 0x71, 0xb2, 0xbb, 0xb4, 0xa3, 0x22, 0x38, 0x15, 0x1b, 0x47, 0x31,
    0x4b, 0xb6, 0x54, 0x79, 0x03, 0x11, 0x02, 0x81, 0x81, 0x00, 0x99, 0x88,
    0x48, 0xb0, 0x55, 0x49, 0x9a, 0x10, 0x09, 0xcb, 0xc7, 0xd2, 0x94, 0xb3,
    0x6b, 0x1f, 0xfd, 0xf2, 0x02, 0x0e, 0x6e, 0x73, 0x64, 0x05, 0x3e, 0x94,
    0xde, 0x1a, 0x00, 0x0d, 0xc9, 0x34, 0x05, 0x87, 0xf7, 0xe2, 0x72, 0x76,
    0xf6, 0x8c, 0xdf, 0x60, 0x8d, 0x75, 0x3b, 0x63, 0x37, 0x7b, 0x03, 0xb6,
    0xf4, 0x08, 0x4d, 0x2c, 0x02, 0x7c, 0x4b, 0x38, 0x96, 0x0a, 0x62, 0x33,
    0xba, 0x9e, 0xd9, 0x73, 0x8b, 0x76, 0xf1, 0x0e, 0xa7, 0x5b, 0xe4, 0x56,
    0x07, 0x8b, 0xf7, 0x01, 0xf6, 0x7c, 0xc6, 0xb3, 0xf3, 0xfd, 0xc1, 0x86,
    0xe6, 0x43, 0x36, 0xc7, 0x6b, 0x37, 0x2e, 0x80, 0x91, 0x0e, 0xc8, 0x0b,
    0x0a, 0xdc, 0xc2, 0x3d, 0x02, 0xfb, 0x9a, 0xe1, 0x04, 0x86, 0xa2, 0x82,
    0x48, 0x07, 0x5b, 0x4e, 0xa7, 0xe5, 0x6d, 0xdf, 0xcf, 0x38, 0x82, 0xe4,
    0x51, 0x56, 0x14, 0x71, 0xa2, 0x91, 0x02, 0x81, 0x80, 0x64, 0x3b, 0xf7,
    0x46, 0x42, 0x9f, 0x7d, 0x83, 0x66, 0x7a, 0x06, 0x53, 0x02, 0x13, 0x47,
    0xef, 0xbf, 0xc0, 0x5e, 0x63, 0x51, 0xf8, 0x21, 0xa9, 0xde, 0xbb, 0x60,
    0xe0, 0xec, 0xcd, 0xe5, 0x00, 0x5a, 0xd9, 0xe9, 0xec, 0x31, 0xe5, 0x58,
    0xf7, 0xe9, 0x2c, 0x29, 0x32, 0x8e, 0x74, 0x56, 0x9d, 0x7c, 0xef, 0x7c,
    0x74, 0xca, 0xbc, 0x2b, 0x35, 0x5e, 0xd4, 0x01, 0xa1, 0xa0, 0x91, 0x4b,
    0x4e, 0x3c, 0xbb, 0x06, 0x48, 0x4e, 0x58, 0x19, 0x60, 0x51, 0x16, 0x9e,
    0xd1, 0x4c, 0xaa, 0x2e, 0xfa, 0x6e, 0xa0, 0x44, 0xe0, 0x54, 0xd2, 0x61,
    0x44, 0xcc, 0x16, 0x29, 0xc5, 0x50, 0x10, 0x55, 0x8a, 0x04, 0xe1, 0x33,
    0xf4, 0x4b, 0x7c, 0x24, 0x4d, 0xac, 0x25, 0xbf, 0x91, 0x3c, 0x57, 0xb8,
    0x90, 0xee, 0x49, 0xf5, 0x48, 0x25, 0x9c, 0xd6, 0x34, 0x04, 0xfe, 0xf6,
    0x85, 0x9d, 0xcf, 0x97, 0x5a};

// Hubert's Bleichenbacher vectors.
const std::vector<RsaDecryptTestVector> kRsaBb2048Vectors = {
    //  test 1 positive test.
    {0,
     // lorem ipsum dolor sit amet
     {0x6c, 0x6f, 0x72, 0x65, 0x6d, 0x20, 0x69, 0x70, 0x73,
      0x75, 0x6d, 0x20, 0x64, 0x6f, 0x6c, 0x6f, 0x72, 0x20,
      0x73, 0x69, 0x74, 0x20, 0x61, 0x6d, 0x65, 0x74},
     {0x8b, 0xfe, 0x26, 0x4e, 0x85, 0xd3, 0xbd, 0xea, 0xa6, 0xb8, 0x85, 0x1b,
      0x8e, 0x3b, 0x95, 0x6e, 0xe3, 0xd2, 0x26, 0xfd, 0x3f, 0x69, 0x06, 0x3a,
      0x86, 0x88, 0x01, 0x73, 0xa2, 0x73, 0xd9, 0xf2, 0x83, 0xb2, 0xee, 0xbd,
      0xd1, 0xed, 0x35, 0xf7, 0xe0, 0x2d, 0x91, 0xc5, 0x71, 0x98, 0x1b, 0x67,
      0x37, 0xd5, 0x32, 0x0b, 0xd8, 0x39, 0x6b, 0x0f, 0x3a, 0xd5, 0xb0, 0x19,
      0xda, 0xec, 0x1b, 0x0a, 0xab, 0x3c, 0xbb, 0xc0, 0x26, 0x39, 0x5f, 0x4f,
      0xd1, 0x4f, 0x13, 0x67, 0x3f, 0x2d, 0xfc, 0x81, 0xf9, 0xb6, 0x60, 0xec,
      0x26, 0xac, 0x38, 0x1e, 0x6d, 0xb3, 0x29, 0x9b, 0x4e, 0x46, 0x0b, 0x43,
      0xfa, 0xb9, 0x95, 0x5d, 0xf2, 0xb3, 0xcf, 0xaa, 0x20, 0xe9, 0x00, 0xe1,
      0x9c, 0x85, 0x62, 0x38, 0xfd, 0x37, 0x18, 0x99, 0xc2, 0xbf, 0x2c, 0xe8,
      0xc8, 0x68, 0xb7, 0x67, 0x54, 0xe5, 0xdb, 0x3b, 0x03, 0x65, 0x33, 0xfd,
      0x60, 0x37, 0x46, 0xbe, 0x13, 0xc1, 0x0d, 0x4e, 0x3e, 0x60, 0x22, 0xeb,
      0xc9, 0x05, 0xd2, 0x0c, 0x2a, 0x7f, 0x32, 0xb2, 0x15, 0xa4, 0xcd, 0x53,
      0xb3, 0xf4, 0x4c, 0xa1, 0xc3, 0x27, 0xd2, 0xc2, 0xb6, 0x51, 0x14, 0x58,
      0x21, 0xc0, 0x83, 0x96, 0xc8, 0x90, 0x71, 0xf6, 0x65, 0x34, 0x9c, 0x25,
      0xe4, 0x4d, 0x27, 0x33, 0xcd, 0x93, 0x05, 0x98, 0x5c, 0xee, 0xf6, 0x43,
      0x0c, 0x3c, 0xf5, 0x7a, 0xf5, 0xfa, 0x22, 0x40, 0x89, 0x22, 0x12, 0x18,
      0xfa, 0x34, 0x73, 0x7c, 0x79, 0xc4, 0x46, 0xd2, 0x8a, 0x94, 0xc4, 0x1c,
      0x96, 0xe4, 0xe9, 0x2a, 0xc5, 0x3f, 0xbc, 0xf3, 0x84, 0xde, 0xa8, 0x41,
      0x9e, 0xa0, 0x89, 0xf8, 0x78, 0x44, 0x45, 0xa4, 0x92, 0xc8, 0x12, 0xeb,
      0x0d, 0x40, 0x94, 0x67, 0xf7, 0x5a, 0xfd, 0x7d, 0x4d, 0x10, 0x78, 0x88,
      0x62, 0x05, 0xa0, 0x66},
     {},
     false,
     true},

    // Invalid Empty Message
    {0,
     {},
     {0x20, 0xaa, 0xa8, 0xad, 0xbb, 0xc5, 0x93, 0xa9, 0x24, 0xba, 0x1c, 0x5c,
      0x79, 0x90, 0xb5, 0xc2, 0x24, 0x2a, 0xe4, 0xb9, 0x9d, 0x0f, 0xe6, 0x36,
      0xa1, 0x9a, 0x4c, 0xf7, 0x54, 0xed, 0xbc, 0xee, 0x77, 0x4e, 0x47, 0x2f,
      0xe0, 0x28, 0x16, 0x0e, 0xd4, 0x26, 0x34, 0xf8, 0x86, 0x49, 0x00, 0xcb,
      0x51, 0x40, 0x06, 0xda, 0x64, 0x2c, 0xae, 0x6a, 0xe8, 0xc7, 0xd0, 0x87,
      0xca, 0xeb, 0xcf, 0xa6, 0xda, 0xd1, 0x55, 0x13, 0x01, 0xe1, 0x30, 0x34,
      0x49, 0x89, 0xa1, 0xd4, 0x62, 0xd4, 0x16, 0x45, 0x05, 0xf6, 0x39, 0x39,
      0x33, 0x45, 0x0c, 0x67, 0xbc, 0x6d, 0x39, 0xd8, 0xf5, 0x16, 0x09, 0x07,
      0xca, 0xbc, 0x25, 0x1b, 0x73, 0x79, 0x25, 0xa1, 0xcf, 0x21, 0xe5, 0xc6,
      0xaa, 0x57, 0x81, 0xb7, 0x76, 0x9f, 0x6a, 0x2a, 0x58, 0x3d, 0x97, 0xcc,
      0xe0, 0x08, 0xc0, 0xf8, 0xb6, 0xad, 0xd5, 0xf0, 0xb2, 0xbd, 0x80, 0xbe,
      0xe6, 0x02, 0x37, 0xaa, 0x39, 0xbb, 0x20, 0x71, 0x9f, 0xe7, 0x57, 0x49,
      0xf4, 0xbc, 0x4e, 0x42, 0x46, 0x6e, 0xf5, 0xa8, 0x61, 0xae, 0x3a, 0x92,
      0x39, 0x5c, 0x7d, 0x85, 0x8d, 0x43, 0x0b, 0xfe, 0x38, 0x04, 0x0f, 0x44,
      0x5e, 0xa9, 0x3f, 0xa2, 0x95, 0x8b, 0x50, 0x35, 0x39, 0x80, 0x0f, 0xfa,
      0x5c, 0xe5, 0xf8, 0xcf, 0x51, 0xfa, 0x81, 0x71, 0xa9, 0x1f, 0x36, 0xcb,
      0x4f, 0x45, 0x75, 0xe8, 0xde, 0x6b, 0x4d, 0x3f, 0x09, 0x6e, 0xe1, 0x40,
      0xb9, 0x38, 0xfd, 0x2f, 0x50, 0xee, 0x13, 0xf0, 0xd0, 0x50, 0x22, 0x2e,
      0x2a, 0x72, 0xb0, 0xa3, 0x06, 0x9f, 0xf3, 0xa6, 0x73, 0x8e, 0x82, 0xc8,
      0x70, 0x90, 0xca, 0xa5, 0xae, 0xd4, 0xfc, 0xbe, 0x88, 0x2c, 0x49, 0x64,
      0x6a, 0xa2, 0x50, 0xb9, 0x8f, 0x12, 0xf8, 0x3c, 0x8d, 0x52, 0x81, 0x13,
      0x61, 0x4a, 0x29, 0xe7},
     {},
     false,
     true},

    // Invalid Max Nessage
    {0,
     {0x22, 0xd8, 0x50, 0x13, 0x7b, 0x9e, 0xeb, 0xe0, 0x92, 0xb2, 0x4f, 0x60,
      0x2d, 0xc5, 0xbb, 0x79, 0x18, 0xc1, 0x6b, 0xd8, 0x9d, 0xdb, 0xf2, 0x04,
      0x67, 0xb1, 0x19, 0xd2, 0x05, 0xf9, 0xc2, 0xe4, 0xbd, 0x7d, 0x25, 0x92,
      0xcf, 0x1e, 0x53, 0x21, 0x06, 0xe0, 0xf3, 0x35, 0x57, 0x56, 0x59, 0x23,
      0xc7, 0x3a, 0x02, 0xd4, 0xf0, 0x9c, 0x0c, 0x22, 0xbe, 0xa8, 0x91, 0x48,
      0x18, 0x3e, 0x60, 0x31, 0x7f, 0x70, 0x28, 0xb3, 0xaa, 0x1f, 0x26, 0x1f,
      0x91, 0xc9, 0x79, 0x39, 0x31, 0x01, 0xd7, 0xe1, 0x5f, 0x40, 0x67, 0xe6,
      0x39, 0x79, 0xb3, 0x27, 0x51, 0x65, 0x8e, 0xf7, 0x69, 0x61, 0x0f, 0xe9,
      0x7c, 0xf9, 0xce, 0xf3, 0x27, 0x8b, 0x31, 0x17, 0xd3, 0x84, 0x05, 0x1c,
      0x3b, 0x1d, 0x82, 0xc2, 0x51, 0xc2, 0x30, 0x54, 0x18, 0xc8, 0xf6, 0x84,
      0x05, 0x30, 0xe6, 0x31, 0xaa, 0xd6, 0x3e, 0x70, 0xe2, 0x0e, 0x02, 0x5b,
      0xcd, 0x8e, 0xfb, 0x54, 0xc9, 0x2e, 0xc6, 0xd3, 0xb1, 0x06, 0xa2, 0xf8,
      0xe6, 0x4e, 0xef, 0xf7, 0xd3, 0x84, 0x95, 0xb0, 0xfc, 0x50, 0xc9, 0x71,
      0x38, 0xaf, 0x4b, 0x1c, 0x0a, 0x67, 0xa1, 0xc4, 0xe2, 0x7b, 0x07, 0x7b,
      0x84, 0x39, 0x33, 0x2e, 0xdf, 0xa8, 0x60, 0x8d, 0xfe, 0xae, 0x65, 0x3c,
      0xd6, 0xa6, 0x28, 0xac, 0x55, 0x03, 0x95, 0xf7, 0xe7, 0x43, 0x90, 0xe4,
      0x2c, 0x11, 0x68, 0x22, 0x34, 0x87, 0x09, 0x25, 0xee, 0xaa, 0x1f, 0xa7,
      0x1b, 0x76, 0xcf, 0x1f, 0x2e, 0xe3, 0xbd, 0xa6, 0x9f, 0x67, 0x17, 0x03,
      0x3f, 0xf8, 0xb7, 0xc9, 0x5c, 0x97, 0x99, 0xe7, 0xa3, 0xbe, 0xa5, 0xe7,
      0xe4, 0xa1, 0xc3, 0x59, 0x77, 0x2f, 0xb6, 0xb1, 0xc6, 0xe6, 0xc5, 0x16,
      0x66, 0x1d, 0xfe, 0x30, 0xc3},
     {0x48, 0xcc, 0xea, 0xb1, 0x0f, 0x39, 0xa4, 0xdb, 0x32, 0xf6, 0x00, 0x74,
      0xfe, 0xea, 0x47, 0x3c, 0xbc, 0xdb, 0x7a, 0xcc, 0xf9, 0x2e, 0x15, 0x04,
      0x17, 0xf7, 0x6b, 0x44, 0x75, 0x6b, 0x19, 0x0e, 0x84, 0x3e, 0x79, 0xec,
      0x12, 0xaa, 0x85, 0x08, 0x3a, 0x21, 0xf5, 0x43, 0x7e, 0x7b, 0xad, 0x0a,
      0x60, 0x48, 0x2e, 0x60, 0x11, 0x98, 0xf9, 0xd8, 0x69, 0x23, 0x23, 0x9c,
      0x87, 0x86, 0xee, 0x72, 0x82, 0x85, 0xaf, 0xd0, 0x93, 0x7f, 0x7d, 0xde,
      0x12, 0x71, 0x7f, 0x28, 0x38, 0x98, 0x43, 0xd7, 0x37, 0x59, 0x12, 0xb0,
      0x7b, 0x99, 0x1f, 0x4f, 0xdb, 0x01, 0x90, 0xfc, 0xed, 0x8b, 0xa6, 0x65,
      0x31, 0x43, 0x67, 0xe8, 0xc5, 0xf9, 0xd2, 0x98, 0x1d, 0x0f, 0x51, 0x28,
      0xfe, 0xeb, 0x46, 0xcb, 0x50, 0xfc, 0x23, 0x7e, 0x64, 0x43, 0x8a, 0x86,
      0xdf, 0x19, 0x8d, 0xd0, 0x20, 0x93, 0x64, 0xae, 0x3a, 0x84, 0x2d, 0x77,
      0x53, 0x2b, 0x66, 0xb7, 0xef, 0x26, 0x3b, 0x83, 0xb1, 0x54, 0x1e, 0xd6,
      0x71, 0xb1, 0x20, 0xdf, 0xd6, 0x60, 0x46, 0x2e, 0x21, 0x07, 0xa4, 0xee,
      0x7b, 0x96, 0x4e, 0x73, 0x4a, 0x7b, 0xd6, 0x8d, 0x90, 0xdd, 0xa6, 0x17,
      0x70, 0x65, 0x8a, 0x3c, 0x24, 0x29, 0x48, 0x53, 0x2d, 0xa3, 0x26, 0x48,
      0x68, 0x7e, 0x03, 0x18, 0x28, 0x64, 0x73, 0xf6, 0x75, 0xb4, 0x12, 0xd6,
      0x46, 0x8f, 0x01, 0x3f, 0x14, 0xd7, 0x60, 0xa3, 0x58, 0xdf, 0xca, 0xd3,
      0xcd, 0xa2, 0xaf, 0xee, 0xc5, 0xe2, 0x68, 0xa3, 0x7d, 0x25, 0x0c, 0x37,
      0xf7, 0x22, 0xf4, 0x68, 0xa7, 0x0d, 0xfd, 0x92, 0xd7, 0x29, 0x4c, 0x3c,
      0x1e, 0xe1, 0xe7, 0xf8, 0x84, 0x3b, 0x7d, 0x16, 0xf9, 0xf3, 0x7e, 0xf3,
      0x57, 0x48, 0xc3, 0xae, 0x93, 0xaa, 0x15, 0x5c, 0xdc, 0xdf, 0xeb, 0x4e,
      0x78, 0x56, 0x73, 0x03},
     {},
     false,
     true},

    // invalid the last value from the PRF is 246, which is longer than the max
    // allowed length: 245, so it needs to select second to last: 2
    {0,
     {0x0f, 0x9b},
     {0x14, 0x39, 0xe0, 0x8c, 0x3f, 0x84, 0xc1, 0xa7, 0xfe, 0xc7, 0x4c, 0xe0,
      0x76, 0x14, 0xb2, 0x0e, 0x01, 0xf6, 0xfa, 0x4e, 0x8c, 0x2a, 0x6c, 0xff,
      0xdc, 0x35, 0x20, 0xd8, 0x88, 0x9e, 0x5d, 0x9a, 0x95, 0x0c, 0x64, 0x25,
      0x79, 0x8f, 0x85, 0xd4, 0xbe, 0x38, 0xd3, 0x00, 0xea, 0x56, 0x95, 0xf1,
      0x3e, 0xcd, 0x4c, 0xb3, 0x89, 0xd1, 0xff, 0x5b, 0x82, 0x48, 0x4b, 0x49,
      0x4d, 0x62, 0x80, 0xab, 0x7f, 0xa7, 0x8e, 0x64, 0x59, 0x33, 0x98, 0x1c,
      0xb9, 0x34, 0xcc, 0xe8, 0xbf, 0xcd, 0x11, 0x4c, 0xc0, 0xe6, 0x81, 0x1e,
      0xef, 0xa4, 0x7a, 0xae, 0x20, 0xaf, 0x63, 0x8a, 0x1c, 0xd1, 0x63, 0xd2,
      0xd3, 0x36, 0x61, 0x86, 0xd0, 0xa0, 0x7d, 0xf0, 0xc8, 0x1f, 0x6c, 0x9f,
      0x31, 0x71, 0xcf, 0x35, 0x61, 0x47, 0x2e, 0x98, 0xa6, 0x00, 0x6b, 0xf7,
      0x5d, 0xdb, 0x45, 0x7b, 0xed, 0x03, 0x6d, 0xcc, 0xe1, 0x99, 0x36, 0x9d,
      0xe7, 0xd9, 0x4e, 0xf2, 0xc6, 0x8e, 0x84, 0x67, 0xee, 0x06, 0x04, 0xee,
      0xa2, 0xb3, 0x00, 0x94, 0x79, 0x16, 0x2a, 0x78, 0x91, 0xba, 0x5c, 0x40,
      0xca, 0xb1, 0x7f, 0x49, 0xe1, 0xc4, 0x38, 0xcb, 0x6e, 0xae, 0xa4, 0xf7,
      0x6c, 0xe2, 0x3c, 0xce, 0x0e, 0x48, 0x3f, 0xf0, 0xe9, 0x6f, 0xa7, 0x90,
      0xea, 0x15, 0xbe, 0x67, 0x67, 0x18, 0x14, 0x34, 0x2d, 0x0a, 0x23, 0xf4,
      0xa2, 0x02, 0x62, 0xb6, 0x18, 0x2e, 0x72, 0xf3, 0xa6, 0x7c, 0xd2, 0x89,
      0x71, 0x15, 0x03, 0xc8, 0x55, 0x16, 0xa9, 0xed, 0x22, 0x54, 0x22, 0xf9,
      0x8b, 0x11, 0x6f, 0x1a, 0xb0, 0x80, 0xa8, 0x0a, 0xbd, 0x6f, 0x02, 0x16,
      0xdf, 0x88, 0xd8, 0xcf, 0xd6, 0x7c, 0x13, 0x92, 0x43, 0xbe, 0x8d, 0xd7,
      0x85, 0x02, 0xa7, 0xaa, 0xf6, 0xbc, 0x99, 0xd7, 0xda, 0x71, 0xbc, 0xdf,
      0x62, 0x7e, 0x73, 0x54},
     {},
     false,
     true},

    // Invalid: the last three numbers from prf are: 2, 247, 255, so we need to
    // pick 2, the third one from the end
    {0,
     {0x4f, 0x02},
     {0x16, 0x90, 0xeb, 0xcc, 0xee, 0xce, 0x2c, 0xe0, 0x24, 0xf3, 0x82, 0xe4,
      0x67, 0xcf, 0x85, 0x10, 0xe7, 0x45, 0x14, 0x12, 0x09, 0x37, 0x97, 0x85,
      0x76, 0xca, 0xf6, 0x84, 0xd4, 0xa0, 0x2a, 0xd5, 0x69, 0xe8, 0xd7, 0x6c,
      0xbe, 0x36, 0x5a, 0x06, 0x0e, 0x00, 0x77, 0x9d, 0xe2, 0xf0, 0x86, 0x5c,
      0xcf, 0x0d, 0x92, 0x3d, 0xe3, 0xb4, 0x78, 0x3a, 0x4e, 0x2c, 0x74, 0xf4,
      0x22, 0xe2, 0xf3, 0x26, 0x08, 0x6c, 0x39, 0x0b, 0x65, 0x8b, 0xa4, 0x7f,
      0x31, 0xab, 0x01, 0x3a, 0xa8, 0x0f, 0x46, 0x8c, 0x71, 0x25, 0x6e, 0x5f,
      0xa5, 0x67, 0x9b, 0x24, 0xe8, 0x3c, 0xd8, 0x2c, 0x3d, 0x1e, 0x05, 0xe3,
      0x98, 0x20, 0x81, 0x55, 0xde, 0x22, 0x12, 0x99, 0x3c, 0xd2, 0xb8, 0xba,
      0xb6, 0x98, 0x7c, 0xf4, 0xcc, 0x12, 0x93, 0xf1, 0x99, 0x09, 0x21, 0x94,
      0x39, 0xd7, 0x41, 0x27, 0x54, 0x5e, 0x9e, 0xd8, 0xa7, 0x06, 0x96, 0x1b,
      0x8e, 0xe2, 0x11, 0x9f, 0x6b, 0xfa, 0xca, 0xfb, 0xef, 0x91, 0xb7, 0x5a,
      0x78, 0x9b, 0xa6, 0x5b, 0x8b, 0x83, 0x3b, 0xc6, 0x14, 0x9c, 0xf4, 0x9b,
      0x5c, 0x4d, 0x2c, 0x63, 0x59, 0xf6, 0x28, 0x08, 0x65, 0x9b, 0xa6, 0x54,
      0x1e, 0x1c, 0xd2, 0x4b, 0xf7, 0xf7, 0x41, 0x04, 0x86, 0xb5, 0x10, 0x3f,
      0x6c, 0x0e, 0xa2, 0x93, 0x34, 0xea, 0x6f, 0x49, 0x75, 0xb1, 0x73, 0x87,
      0x47, 0x4f, 0xe9, 0x20, 0x71, 0x0e, 0xa6, 0x15, 0x68, 0xd7, 0xb7, 0xc0,
      0xa7, 0x91, 0x6a, 0xcf, 0x21, 0x66, 0x5a, 0xd5, 0xa3, 0x1c, 0x4e, 0xab,
      0xcd, 0xe4, 0x4f, 0x8f, 0xb6, 0x12, 0x0d, 0x84, 0x57, 0xaf, 0xa1, 0xf3,
      0xc8, 0x5d, 0x51, 0x7c, 0xda, 0x36, 0x4a, 0xf6, 0x20, 0x11, 0x3a, 0xe5,
      0xa3, 0xc5, 0x2a, 0x04, 0x88, 0x21, 0x73, 0x19, 0x22, 0x73, 0x73, 0x07,
      0xf7, 0x7a, 0x10, 0x81},
     {},
     false,
     true},

    // ciphertext that generates a fake 11 byte plaintext, but decrypts
    // to real 11 byte long plaintext
    {0,
     // lorem ipsum
     {0x6c, 0x6f, 0x72, 0x65, 0x6d, 0x20, 0x69, 0x70, 0x73, 0x75, 0x6d},
     {0x62, 0x13, 0x63, 0x45, 0x93, 0x33, 0x2c, 0x48, 0x5c, 0xef, 0x78, 0x3e,
      0xa2, 0x84, 0x6e, 0x3d, 0x6e, 0x8b, 0x0e, 0x00, 0x5c, 0xd8, 0x29, 0x3e,
      0xae, 0xbb, 0xaa, 0x50, 0x79, 0x71, 0x2f, 0xd6, 0x81, 0x57, 0x9b, 0xdf,
      0xbb, 0xda, 0x13, 0x8a, 0xe4, 0xd9, 0xd9, 0x52, 0x91, 0x7a, 0x03, 0xc9,
      0x23, 0x98, 0xec, 0x0c, 0xb2, 0xbb, 0x0c, 0x6b, 0x5a, 0x8d, 0x55, 0x06,
      0x1f, 0xed, 0x0d, 0x0d, 0x8d, 0x72, 0x47, 0x35, 0x63, 0x15, 0x26, 0x48,
      0xcf, 0xe6, 0x40, 0xb3, 0x35, 0xdc, 0x95, 0x33, 0x1c, 0x21, 0xcb, 0x13,
      0x3a, 0x91, 0x79, 0x0f, 0xa9, 0x3a, 0xe4, 0x44, 0x97, 0xc1, 0x28, 0x70,
      0x89, 0x70, 0xd2, 0xbe, 0xeb, 0x77, 0xe8, 0x72, 0x1b, 0x06, 0x1b, 0x1c,
      0x44, 0x03, 0x41, 0x43, 0x73, 0x4a, 0x77, 0xbe, 0x82, 0x20, 0x87, 0x74,
      0x15, 0xa6, 0xdb, 0xa0, 0x73, 0xc3, 0x87, 0x16, 0x05, 0x38, 0x05, 0x42,
      0xa9, 0xf2, 0x52, 0x52, 0xa4, 0xba, 0xbe, 0x83, 0x31, 0xcd, 0xd5, 0x3c,
      0xf8, 0x28, 0x42, 0x3f, 0x3c, 0xc7, 0x0b, 0x56, 0x06, 0x24, 0xd0, 0x58,
      0x1f, 0xb1, 0x26, 0xb2, 0xed, 0x4f, 0x4e, 0xd3, 0x58, 0xf0, 0xeb, 0x80,
      0x65, 0xcf, 0x17, 0x63, 0x99, 0xac, 0x1a, 0x84, 0x6a, 0x31, 0x05, 0x5f,
      0x9a, 0xe8, 0xc9, 0xc2, 0x4a, 0x1b, 0xa0, 0x50, 0xbc, 0x20, 0x84, 0x21,
      0x25, 0xbc, 0x17, 0x53, 0x15, 0x8f, 0x80, 0x65, 0xf3, 0xad, 0xb9, 0xcc,
      0x16, 0xbf, 0xdf, 0x83, 0x81, 0x6b, 0xdf, 0x38, 0xb6, 0x24, 0xf1, 0x20,
      0x22, 0xc5, 0xa6, 0xfb, 0xfe, 0x29, 0xbc, 0x91, 0x54, 0x2b, 0xe8, 0xc0,
      0x20, 0x8a, 0x77, 0x0b, 0xcd, 0x67, 0x7d, 0xc5, 0x97, 0xf5, 0x55, 0x7d,
      0xc2, 0xce, 0x28, 0xa1, 0x1b, 0xf3, 0xe3, 0x85, 0x7f, 0x15, 0x87, 0x17,
      0xa3, 0x3f, 0x65, 0x92},
     {},
     false,
     true},

    // ciphertext that starts with a null byte, decrypts to real 11 byte
    // long plaintext
    {0,
     // lorem ipsum
     {0x6c, 0x6f, 0x72, 0x65, 0x6d, 0x20, 0x69, 0x70, 0x73, 0x75, 0x6d},
     {0x00, 0xa2, 0xe8, 0xf1, 0x14, 0xea, 0x8d, 0x05, 0xd1, 0x2d, 0xc8, 0x43,
      0xe3, 0xcc, 0x3b, 0x2e, 0xdc, 0x82, 0x29, 0xff, 0x2a, 0x02, 0x8b, 0xda,
      0x29, 0xba, 0x9d, 0x55, 0xe3, 0xcd, 0x02, 0x91, 0x19, 0x02, 0xfe, 0xf1,
      0xf4, 0x2a, 0x07, 0x5b, 0xf0, 0x5e, 0x80, 0x16, 0xe8, 0x56, 0x72, 0x13,
      0xd6, 0xf2, 0x60, 0xfa, 0x49, 0xe3, 0x60, 0x77, 0x9d, 0xd8, 0x1a, 0xee,
      0xa3, 0xe0, 0x4c, 0x2c, 0xb5, 0x67, 0xe0, 0xd7, 0x2b, 0x98, 0xbf, 0x75,
      0x40, 0x14, 0x56, 0x1b, 0x75, 0x11, 0xe0, 0x83, 0xd2, 0x0e, 0x0b, 0xfb,
      0x9c, 0xd2, 0x3f, 0x8a, 0x0d, 0x3c, 0x88, 0x90, 0x0c, 0x49, 0xd2, 0xfc,
      0xd5, 0x84, 0x3f, 0xf0, 0x76, 0x56, 0x07, 0xb2, 0x02, 0x6f, 0x28, 0x20,
      0x2a, 0x87, 0xaa, 0x94, 0x67, 0x8a, 0xed, 0x22, 0xa0, 0xc2, 0x07, 0x24,
      0x54, 0x13, 0x94, 0xcd, 0x8f, 0x44, 0xe3, 0x73, 0xeb, 0xa1, 0xd2, 0xba,
      0xe9, 0x8f, 0x51, 0x6c, 0x1e, 0x2b, 0xa3, 0xd8, 0x68, 0x52, 0xd0, 0x64,
      0xf8, 0x56, 0xb1, 0xda, 0xf2, 0x47, 0x95, 0xe7, 0x67, 0xa2, 0xb9, 0x03,
      0x96, 0xe5, 0x07, 0x43, 0xe3, 0x15, 0x06, 0x64, 0xaf, 0xab, 0x13, 0x1f,
      0xe4, 0x0e, 0xa4, 0x05, 0xdc, 0xf5, 0x72, 0xdd, 0x10, 0x79, 0xaf, 0x1d,
      0x3f, 0x03, 0x92, 0xcc, 0xad, 0xcc, 0xa0, 0xa1, 0x27, 0x40, 0xdb, 0xb2,
      0x13, 0xb9, 0x25, 0xca, 0x2a, 0x06, 0xb1, 0xbc, 0x13, 0x83, 0xe8, 0x3a,
      0x65, 0x8c, 0x82, 0xba, 0x2e, 0x74, 0x27, 0x34, 0x23, 0x79, 0x08, 0x4d,
      0x5f, 0x66, 0xb5, 0x44, 0x57, 0x9f, 0x07, 0x66, 0x4c, 0xb2, 0x6e, 0xdd,
      0x4f, 0x10, 0xfd, 0x91, 0x3f, 0xdb, 0xc0, 0xde, 0x05, 0xef, 0x88, 0x7d,
      0x4d, 0x1e, 0xc1, 0xac, 0x95, 0x65, 0x23, 0x97, 0xea, 0x7f, 0xd4, 0xe4,
      0x75, 0x9f, 0xda, 0x8b},
     {},
     false,
     true},

    // ciphertext that starts with two null bytes, decrypts to real 11 byte
    // long plaintext
    {0,
     // lorem ipsum
     {0x6c, 0x6f, 0x72, 0x65, 0x6d, 0x20, 0x69, 0x70, 0x73, 0x75, 0x6d},
     {0x00, 0x00, 0x1f, 0x71, 0x87, 0x9b, 0x42, 0x61, 0x27, 0xf7, 0xde, 0xad,
      0x62, 0x1f, 0x73, 0x80, 0xa7, 0x09, 0x8c, 0xf7, 0xd2, 0x21, 0x73, 0xaa,
      0x27, 0x99, 0x1b, 0x14, 0x3c, 0x46, 0xd5, 0x33, 0x83, 0xc2, 0x09, 0xbd,
      0x0c, 0x9c, 0x00, 0xd8, 0x40, 0x78, 0x03, 0x7e, 0x71, 0x5f, 0x6b, 0x98,
      0xc6, 0x50, 0x05, 0xa7, 0x71, 0x20, 0x07, 0x05, 0x22, 0xed, 0xe5, 0x1d,
      0x47, 0x2c, 0x87, 0xef, 0x94, 0xb9, 0x4e, 0xad, 0x4c, 0x54, 0x28, 0xee,
      0x10, 0x8a, 0x34, 0x55, 0x61, 0x65, 0x83, 0x01, 0x91, 0x1e, 0xc5, 0xa8,
      0xf7, 0xdd, 0x43, 0xed, 0x4a, 0x39, 0x57, 0xfd, 0x29, 0xfb, 0x02, 0xa3,
      0x52, 0x9b, 0xf6, 0x3f, 0x80, 0x40, 0xd3, 0x95, 0x34, 0x90, 0x93, 0x9b,
      0xd8, 0xf7, 0x8b, 0x2a, 0x34, 0x04, 0xb6, 0xfb, 0x5f, 0xf7, 0x0a, 0x4b,
      0xfd, 0xaa, 0xc5, 0xc5, 0x41, 0xd6, 0xbc, 0xce, 0x49, 0xc9, 0x77, 0x8c,
      0xc3, 0x90, 0xbe, 0x24, 0xcb, 0xef, 0x1d, 0x1e, 0xca, 0x7e, 0x87, 0x04,
      0x57, 0x24, 0x1d, 0x3f, 0xf7, 0x2c, 0xa4, 0x4f, 0x9f, 0x56, 0xbd, 0xf3,
      0x1a, 0x89, 0x0f, 0xa5, 0xeb, 0x3a, 0x91, 0x07, 0xb6, 0x03, 0xcc, 0xc9,
      0xd0, 0x6a, 0x5d, 0xd9, 0x11, 0xa6, 0x64, 0xc8, 0x2b, 0x6a, 0xbd, 0x4f,
      0xe0, 0x36, 0xf8, 0xdb, 0x8d, 0x5a, 0x07, 0x0c, 0x2d, 0x86, 0x38, 0x6a,
      0xe1, 0x8d, 0x97, 0xad, 0xc1, 0x84, 0x76, 0x40, 0xc2, 0x11, 0xd9, 0x1f,
      0xf5, 0xc3, 0x38, 0x75, 0x74, 0xa2, 0x6f, 0x8e, 0xf2, 0x7c, 0xa7, 0xf4,
      0x8d, 0x2d, 0xd1, 0xf0, 0xc7, 0xf1, 0x4b, 0x81, 0xcc, 0x9d, 0x33, 0xee,
      0x68, 0x53, 0x03, 0x1d, 0x3e, 0xcf, 0x10, 0xa9, 0x14, 0xff, 0xd9, 0x09,
      0x47, 0x90, 0x9c, 0x80, 0x11, 0xfd, 0x30, 0x24, 0x92, 0x19, 0x34, 0x8e,
      0xbf, 0xf7, 0x6b, 0xfc},
     {},
     false,
     true},

    // valid ciphertext that generates a zero length fake plaintext
    {0,
     // lorem ipsum
     {0x6c, 0x6f, 0x72, 0x65, 0x6d, 0x20, 0x69, 0x70, 0x73, 0x75, 0x6d},
     {0xb5, 0xe4, 0x93, 0x08, 0xf6, 0xe9, 0x59, 0x00, 0x14, 0xff, 0xaf, 0xfc,
      0x5b, 0x85, 0x60, 0x75, 0x57, 0x39, 0xdd, 0x50, 0x1f, 0x1d, 0x4e, 0x92,
      0x27, 0xa7, 0xd2, 0x91, 0x40, 0x8c, 0xf4, 0xb7, 0x53, 0xf2, 0x92, 0x32,
      0x2f, 0xf8, 0xbe, 0xad, 0x61, 0x3b, 0xf2, 0xca, 0xa1, 0x81, 0xb2, 0x21,
      0xbc, 0x38, 0xca, 0xf6, 0x39, 0x2d, 0xea, 0xfb, 0x28, 0xeb, 0x21, 0xad,
      0x60, 0x93, 0x08, 0x41, 0xed, 0x02, 0xfd, 0x62, 0x25, 0xcc, 0x9c, 0x46,
      0x34, 0x09, 0xad, 0xbe, 0x7d, 0x8f, 0x32, 0x44, 0x02, 0x12, 0xfb, 0xe3,
      0x88, 0x1c, 0x51, 0x37, 0x5b, 0xb0, 0x95, 0x65, 0xef, 0xb2, 0x2e, 0x62,
      0xb0, 0x71, 0x47, 0x2f, 0xb3, 0x86, 0x76, 0xe5, 0xb4, 0xe2, 0x3a, 0x06,
      0x17, 0xdb, 0x5d, 0x14, 0xd9, 0x35, 0x19, 0xac, 0x00, 0x07, 0xa3, 0x0a,
      0x9c, 0x82, 0x2e, 0xb3, 0x1c, 0x38, 0xb5, 0x7f, 0xcb, 0x1b, 0xe2, 0x96,
      0x08, 0xfc, 0xf1, 0xca, 0x2a, 0xbd, 0xca, 0xf5, 0xd5, 0x75, 0x2b, 0xbc,
      0x2b, 0x5a, 0xc7, 0xdb, 0xa5, 0xaf, 0xcf, 0xf4, 0xa5, 0x64, 0x1d, 0xa3,
      0x60, 0xdd, 0x01, 0xf7, 0x11, 0x25, 0x39, 0xb1, 0xed, 0x46, 0xcd, 0xb5,
      0x50, 0xa3, 0xb1, 0x00, 0x65, 0x59, 0xb9, 0xfe, 0x18, 0x91, 0x03, 0x0e,
      0xc8, 0x0f, 0x07, 0x27, 0xc4, 0x24, 0x01, 0xdd, 0xd6, 0xcb, 0xb5, 0xe3,
      0xc8, 0x0f, 0x31, 0x2d, 0xf6, 0xec, 0x89, 0x39, 0x4c, 0x5a, 0x71, 0x18,
      0xf5, 0x73, 0x10, 0x5e, 0x7a, 0xb0, 0x0f, 0xe5, 0x78, 0x33, 0xc1, 0x26,
      0x14, 0x1b, 0x50, 0xa9, 0x35, 0x22, 0x48, 0x42, 0xad, 0xdf, 0xb4, 0x79,
      0xf7, 0x51, 0x60, 0x65, 0x9b, 0xa2, 0x88, 0x77, 0xb5, 0x12, 0xbb, 0x9a,
      0x93, 0x08, 0x4a, 0xd8, 0xbe, 0xc5, 0x40, 0xf9, 0x26, 0x40, 0xf6, 0x3a,
      0x11, 0xa0, 0x10, 0xe0},
     {},
     false,
     true},

    // valid ciphertext that generates a 245 byte long fake plaintext
    {0,
     // lorem ipsum
     {0x6c, 0x6f, 0x72, 0x65, 0x6d, 0x20, 0x69, 0x70, 0x73, 0x75, 0x6d},
     {0x1e, 0xa0, 0xb5, 0x0c, 0xa6, 0x52, 0x03, 0xd0, 0xa0, 0x92, 0x80, 0xd3,
      0x97, 0x04, 0xb2, 0x4f, 0xe6, 0xe4, 0x78, 0x00, 0x18, 0x9d, 0xb5, 0x03,
      0x3f, 0x20, 0x27, 0x61, 0xa7, 0x8b, 0xaf, 0xb2, 0x70, 0xc5, 0xe2, 0x5a,
      0xbd, 0x1f, 0x7e, 0xcc, 0x6e, 0x7a, 0xbc, 0x4f, 0x26, 0xd1, 0xb0, 0xcd,
      0x9b, 0x8c, 0x64, 0x8d, 0x52, 0x94, 0x16, 0xee, 0x64, 0xcc, 0xbd, 0xd7,
      0xaa, 0x72, 0xa7, 0x71, 0xd0, 0x35, 0x32, 0x62, 0xb5, 0x43, 0xf0, 0xe4,
      0x36, 0x07, 0x6f, 0x40, 0xa1, 0x09, 0x5f, 0x5c, 0x7d, 0xfd, 0x10, 0xdc,
      0xf0, 0x05, 0x9c, 0xcb, 0x30, 0xe9, 0x2d, 0xfa, 0x5e, 0x01, 0x56, 0x61,
      0x82, 0x15, 0xf1, 0xc3, 0xff, 0x3a, 0xa9, 0x97, 0xa9, 0xd9, 0x99, 0xe5,
      0x06, 0x92, 0x4f, 0x52, 0x89, 0xe3, 0xac, 0x72, 0xe5, 0xe2, 0x08, 0x6c,
      0xc7, 0xb4, 0x99, 0xd7, 0x15, 0x83, 0xed, 0x56, 0x10, 0x28, 0x67, 0x11,
      0x55, 0xdb, 0x40, 0x05, 0xbe, 0xe0, 0x18, 0x00, 0xa7, 0xcd, 0xbd, 0xae,
      0x78, 0x1d, 0xd3, 0x21, 0x99, 0xb8, 0x91, 0x4b, 0x5d, 0x40, 0x11, 0xdd,
      0x6f, 0xf1, 0x1c, 0xd2, 0x6d, 0x46, 0xaa, 0xd5, 0x49, 0x34, 0xd2, 0x93,
      0xb0, 0xbc, 0x40, 0x3d, 0xd2, 0x11, 0xbf, 0x13, 0xb5, 0xa5, 0xc6, 0x83,
      0x6a, 0x5e, 0x76, 0x99, 0x30, 0xf4, 0x37, 0xff, 0xd8, 0x63, 0x4f, 0xb7,
      0x37, 0x17, 0x76, 0xf4, 0xbc, 0x88, 0xfa, 0x6c, 0x27, 0x1d, 0x8a, 0xa6,
      0x01, 0x3d, 0xf8, 0x9a, 0xe6, 0x47, 0x01, 0x54, 0x49, 0x7c, 0x4a, 0xc8,
      0x61, 0xbe, 0x2a, 0x1c, 0x65, 0xeb, 0xff, 0xec, 0x13, 0x9b, 0xf7, 0xaa,
      0xba, 0x3a, 0x81, 0xc7, 0xc5, 0xcd, 0xd8, 0x4d, 0xa9, 0xaf, 0x5d, 0x3e,
      0xdf, 0xb9, 0x57, 0x84, 0x80, 0x74, 0x68, 0x6b, 0x58, 0x37, 0xec, 0xbc,
      0xb6, 0xa4, 0x1c, 0x50},
     {},
     false,
     true},

    // a random ciphertext that generates a fake 11 byte plaintext
    // and fails padding check
    {0,
     {0xaf, 0x9a, 0xc7, 0x01, 0x91, 0xc9, 0x24, 0x13, 0xcb, 0x9f, 0x2d},
     {0x5f, 0x02, 0xf4, 0xb1, 0xf4, 0x69, 0x35, 0xc7, 0x42, 0xeb, 0xe6, 0x2b,
      0x6f, 0x05, 0xaa, 0x0a, 0x32, 0x86, 0xaa, 0xb9, 0x1a, 0x49, 0xb3, 0x47,
      0x80, 0xad, 0xde, 0x64, 0x10, 0xab, 0x46, 0xf7, 0x38, 0x6e, 0x05, 0x74,
      0x83, 0x31, 0x86, 0x4a, 0xc9, 0x8e, 0x1d, 0xa6, 0x36, 0x86, 0xe4, 0xba,
      0xbe, 0x3a, 0x19, 0xed, 0x40, 0xa7, 0xf5, 0xce, 0xef, 0xb8, 0x91, 0x79,
      0x59, 0x6a, 0xab, 0x07, 0xab, 0x10, 0x15, 0xe0, 0x3b, 0x8f, 0x82, 0x50,
      0x84, 0xda, 0xb0, 0x28, 0xb6, 0x73, 0x12, 0x88, 0xf2, 0xe5, 0x11, 0xa4,
      0xb3, 0x14, 0xb6, 0xea, 0x39, 0x97, 0xd2, 0xe8, 0xfe, 0x28, 0x25, 0xce,
      0xf8, 0x89, 0x7c, 0xbb, 0xdf, 0xb6, 0xc9, 0x39, 0xd4, 0x41, 0xd6, 0xe0,
      0x49, 0x48, 0x41, 0x4b, 0xb6, 0x9e, 0x68, 0x29, 0x27, 0xef, 0x85, 0x76,
      0xc9, 0xa7, 0x09, 0x0d, 0x4a, 0xad, 0x0e, 0x74, 0xc5, 0x20, 0xd6, 0xd5,
      0xce, 0x63, 0xa1, 0x54, 0x72, 0x0f, 0x00, 0xb7, 0x6d, 0xe8, 0xcc, 0x55,
      0x0b, 0x1a, 0xa1, 0x4f, 0x01, 0x6d, 0x63, 0xa7, 0xb6, 0xd6, 0xea, 0xa1,
      0xf7, 0xdb, 0xe9, 0xe5, 0x02, 0x00, 0xd3, 0x15, 0x9b, 0x3d, 0x09, 0x9c,
      0x90, 0x01, 0x16, 0xbf, 0x4e, 0xba, 0x3b, 0x94, 0x20, 0x4f, 0x18, 0xb1,
      0x31, 0x7b, 0x07, 0x52, 0x97, 0x51, 0xab, 0xf6, 0x4a, 0x26, 0xb0, 0xa0,
      0xbf, 0x1c, 0x8c, 0xe7, 0x57, 0x33, 0x3b, 0x3d, 0x67, 0x32, 0x11, 0xb6,
      0x7c, 0xc0, 0x65, 0x3f, 0x2f, 0xe2, 0x62, 0x0d, 0x57, 0xc8, 0xb6, 0xee,
      0x57, 0x4a, 0x03, 0x23, 0xa1, 0x67, 0xea, 0xb1, 0x10, 0x6d, 0x9b, 0xc7,
      0xfd, 0x90, 0xd4, 0x15, 0xbe, 0x5f, 0x1e, 0x98, 0x91, 0xa0, 0xe6, 0xc7,
      0x09, 0xf4, 0xfc, 0x04, 0x04, 0xe8, 0x22, 0x6f, 0x84, 0x77, 0xb4, 0xe9,
      0x39, 0xb3, 0x6e, 0xb2},
     {},
     false,
     true},

    // an otherwise correct plaintext, but with wrong first byte
    // (0x01 instead of 0x00), generates a random 11 byte long plaintext
    {0,
     {0xa1, 0xf8, 0xc9, 0x25, 0x5c, 0x35, 0xcf, 0xba, 0x40, 0x3c, 0xcc},
     {0x9b, 0x2e, 0xc9, 0xc0, 0xc9, 0x17, 0xc9, 0x8f, 0x1a, 0xd3, 0xd0, 0x11,
      0x9a, 0xec, 0x6b, 0xe5, 0x1a, 0xe3, 0x10, 0x6e, 0x9a, 0xf1, 0x91, 0x4d,
      0x48, 0x60, 0x0a, 0xb6, 0xa2, 0xc0, 0xc0, 0xc8, 0xae, 0x02, 0xa2, 0xdc,
      0x30, 0x39, 0x90, 0x6f, 0xf3, 0xaa, 0xc9, 0x04, 0xaf, 0x32, 0xec, 0x79,
      0x8f, 0xd6, 0x5f, 0x3a, 0xd1, 0xaf, 0xa2, 0xe6, 0x94, 0x00, 0xe7, 0xc1,
      0xde, 0x81, 0xf5, 0x72, 0x8f, 0x3b, 0x32, 0x91, 0xf3, 0x82, 0x63, 0xbc,
      0x7a, 0x90, 0xa0, 0x56, 0x3e, 0x43, 0xce, 0x7a, 0x0d, 0x4e, 0xe9, 0xc0,
      0xd8, 0xa7, 0x16, 0x62, 0x1c, 0xa5, 0xd3, 0xd0, 0x81, 0x18, 0x87, 0x69,
      0xce, 0x1b, 0x13, 0x1a, 0xf7, 0xd3, 0x5b, 0x13, 0xde, 0xa9, 0x91, 0x53,
      0x57, 0x9c, 0x86, 0xdb, 0x31, 0xfe, 0x07, 0xd5, 0xa2, 0xc1, 0x4d, 0x62,
      0x1b, 0x77, 0x85, 0x4e, 0x48, 0xa8, 0xdf, 0x41, 0xb5, 0x79, 0x85, 0x63,
      0xaf, 0x48, 0x9a, 0x29, 0x1e, 0x41, 0x7b, 0x6a, 0x33, 0x4c, 0x63, 0x22,
      0x26, 0x27, 0x37, 0x61, 0x18, 0xc0, 0x2c, 0x53, 0xb6, 0xe8, 0x63, 0x10,
      0xf7, 0x28, 0x73, 0x4f, 0xfc, 0x86, 0xef, 0x9d, 0x7c, 0x8b, 0xf5, 0x6c,
      0x0c, 0x84, 0x1b, 0x24, 0xb8, 0x2b, 0x59, 0xf5, 0x1a, 0xee, 0x45, 0x26,
      0xba, 0x1c, 0x42, 0x68, 0x50, 0x6d, 0x30, 0x1e, 0x4e, 0xbc, 0x49, 0x8c,
      0x6a, 0xeb, 0xb6, 0xfd, 0x52, 0x58, 0xc8, 0x76, 0xbf, 0x90, 0x0b, 0xac,
      0x8c, 0xa4, 0xd3, 0x09, 0xdd, 0x52, 0x2f, 0x6a, 0x63, 0x43, 0x59, 0x9a,
      0x8b, 0xc3, 0x76, 0x0f, 0x42, 0x2c, 0x10, 0xc7, 0x2d, 0x0a, 0xd5, 0x27,
      0xce, 0x4a, 0xf1, 0x87, 0x41, 0x24, 0xac, 0xe3, 0xd9, 0x9b, 0xb7, 0x4d,
      0xb8, 0xd6, 0x9d, 0x25, 0x28, 0xdb, 0x22, 0xc3, 0xa3, 0x76, 0x44, 0x64,
      0x0f, 0x95, 0xc0, 0x5f},
     {},
     false,
     true},

    // an otherwise correct plaintext, but with wrong second byte
    // (0x01 instead of 0x02), generates a random 11 byte long plaintext
    {0,
     {0xe6, 0xd7, 0x00, 0x30, 0x9c, 0xa0, 0xed, 0x62, 0x45, 0x22, 0x54},
     {0x78, 0x2c, 0x2b, 0x59, 0xa2, 0x1a, 0x51, 0x12, 0x43, 0x82, 0x0a, 0xce,
      0xdd, 0x56, 0x7c, 0x13, 0x6f, 0x6d, 0x30, 0x90, 0xc1, 0x15, 0x23, 0x2a,
      0x82, 0xa5, 0xef, 0xb0, 0xb1, 0x78, 0x28, 0x5f, 0x55, 0xb5, 0xec, 0x2d,
      0x2b, 0xac, 0x96, 0xbf, 0x00, 0xd6, 0x59, 0x2e, 0xa7, 0xcd, 0xc3, 0x34,
      0x16, 0x10, 0xc8, 0xfb, 0x07, 0xe5, 0x27, 0xe5, 0xe2, 0xd2, 0x0c, 0xfa,
      0xf2, 0xc7, 0xf2, 0x3e, 0x37, 0x54, 0x31, 0xf4, 0x5e, 0x99, 0x89, 0x29,
      0xa0, 0x2f, 0x25, 0xfd, 0x95, 0x35, 0x4c, 0x33, 0x83, 0x80, 0x90, 0xbc,
      0xa8, 0x38, 0x50, 0x22, 0x59, 0xe9, 0x2d, 0x86, 0xd5, 0x68, 0xbc, 0x2c,
      0xdb, 0x13, 0x2f, 0xab, 0x2a, 0x39, 0x95, 0x93, 0xca, 0x60, 0xa0, 0x15,
      0xdc, 0x2b, 0xb1, 0xaf, 0xcd, 0x64, 0xfe, 0xf8, 0xa3, 0x83, 0x4e, 0x17,
      0xe5, 0x35, 0x8d, 0x82, 0x29, 0x80, 0xdc, 0x44, 0x6e, 0x84, 0x5b, 0x3a,
      0xb4, 0x70, 0x2b, 0x1e, 0xe4, 0x1f, 0xe5, 0xdb, 0x71, 0x6d, 0x92, 0x34,
      0x8d, 0x50, 0x91, 0xc1, 0x5d, 0x35, 0xa1, 0x10, 0x55, 0x5a, 0x35, 0xde,
      0xb4, 0x65, 0x0a, 0x5a, 0x1d, 0x2c, 0x98, 0x02, 0x5d, 0x42, 0xd4, 0x54,
      0x4f, 0x8b, 0x32, 0xaa, 0x6a, 0x5e, 0x02, 0xdc, 0x02, 0xde, 0xae, 0xd9,
      0xa7, 0x31, 0x3b, 0x73, 0xb4, 0x9b, 0x0d, 0x47, 0x72, 0xa3, 0x76, 0x8b,
      0x0e, 0xa0, 0xdb, 0x58, 0x46, 0xac, 0xe6, 0x56, 0x9c, 0xae, 0x67, 0x7b,
      0xf6, 0x7f, 0xb0, 0xac, 0xf3, 0xc2, 0x55, 0xdc, 0x01, 0xec, 0x84, 0x00,
      0xc9, 0x63, 0xb6, 0xe4, 0x9b, 0x10, 0x67, 0x72, 0x8b, 0x4e, 0x56, 0x3d,
      0x7e, 0x1e, 0x15, 0x15, 0x66, 0x43, 0x47, 0xb9, 0x2e, 0xe6, 0x4d, 0xb7,
      0xef, 0xb5, 0x45, 0x23, 0x57, 0xa0, 0x2f, 0xff, 0x7f, 0xcb, 0x74, 0x37,
      0xab, 0xc2, 0xe5, 0x79},
     {},
     false,
     true},

    // an otherwise correct plaintext, but with wrong second byte
    // (0x00 instead of 0x02), and a 0x02 on third position, generates a
    // random 11 byte long plaintext
    {0,
     {0x3d, 0x4a, 0x05, 0x4d, 0x93, 0x58, 0x20, 0x9e, 0x9c, 0xbb, 0xb9},
     {0x17, 0x86, 0x55, 0x0c, 0xe8, 0xd8, 0x43, 0x30, 0x52, 0xe0, 0x1e, 0xcb,
      0xa8, 0xb7, 0x6d, 0x30, 0x19, 0xf1, 0x35, 0x5b, 0x21, 0x2a, 0xc9, 0xd0,
      0xf5, 0x19, 0x1b, 0x02, 0x33, 0x25, 0xa7, 0xe7, 0x71, 0x4b, 0x78, 0x02,
      0xf8, 0xe9, 0xa1, 0x7c, 0x4c, 0xb3, 0xcd, 0x3a, 0x84, 0x04, 0x18, 0x91,
      0x47, 0x1b, 0x10, 0xca, 0x1f, 0xcf, 0xb5, 0xd0, 0x41, 0xd3, 0x4c, 0x82,
      0xe6, 0xd0, 0x01, 0x1c, 0xf4, 0xdc, 0x76, 0xb9, 0x0e, 0x9c, 0x2e, 0x07,
      0x43, 0x59, 0x05, 0x79, 0xd5, 0x5b, 0xcd, 0x78, 0x57, 0x05, 0x71, 0x52,
      0xc4, 0xa8, 0x04, 0x03, 0x61, 0x34, 0x3d, 0x1d, 0x22, 0xba, 0x67, 0x7d,
      0x62, 0xb0, 0x11, 0x40, 0x7c, 0x65, 0x2e, 0x23, 0x4b, 0x1d, 0x66, 0x3a,
      0xf2, 0x5e, 0x23, 0x86, 0x25, 0x1d, 0x74, 0x09, 0x19, 0x0f, 0x19, 0xfc,
      0x8e, 0xc3, 0xf9, 0x37, 0x4f, 0xdf, 0x12, 0x54, 0x63, 0x38, 0x74, 0xce,
      0x2e, 0xc2, 0xbf, 0xf4, 0x0a, 0xd0, 0xcb, 0x47, 0x3f, 0x97, 0x61, 0xec,
      0x7b, 0x68, 0xda, 0x45, 0xa4, 0xbd, 0x5e, 0x33, 0xf5, 0xd7, 0xda, 0xc9,
      0xb9, 0xa2, 0x08, 0x21, 0xdf, 0x94, 0x06, 0xb6, 0x53, 0xf7, 0x8a, 0x95,
      0xa6, 0xc0, 0xea, 0x0a, 0x4d, 0x57, 0xf8, 0x67, 0xe4, 0xdb, 0x22, 0xc1,
      0x7b, 0xf9, 0xa1, 0x2c, 0x15, 0x0f, 0x80, 0x9a, 0x7b, 0x72, 0xb6, 0xdb,
      0x86, 0xc2, 0x2a, 0x87, 0x32, 0x24, 0x1e, 0xbf, 0x3c, 0x6a, 0x4f, 0x2c,
      0xf8, 0x26, 0x71, 0xd9, 0x17, 0xab, 0xa8, 0xbc, 0x61, 0x05, 0x2b, 0x40,
      0xcc, 0xdd, 0xd7, 0x43, 0xa9, 0x4e, 0xa9, 0xb5, 0x38, 0x17, 0x51, 0x06,
      0x20, 0x19, 0x71, 0xcc, 0xa9, 0xd1, 0x36, 0xd2, 0x50, 0x81, 0x73, 0x9a,
      0xaf, 0x6c, 0xd1, 0x8b, 0x2a, 0xec, 0xf9, 0xad, 0x32, 0x0e, 0xa3, 0xf8,
      0x95, 0x02, 0xf9, 0x55},
     {},
     false,
     true},

    // an otherwise correct plaintext, but with a null byte on third
    // position (first byte of padding), generates a random 11 byte
    // long payload
    {0,
     {0x1f, 0x03, 0x7d, 0xd7, 0x17, 0xb0, 0x7d, 0x3e, 0x7f, 0x73, 0x59},
     {0x17, 0x95, 0x98, 0x82, 0x38, 0x12, 0xd2, 0xc5, 0x8a, 0x7e, 0xb5, 0x05,
      0x21, 0x15, 0x0a, 0x48, 0xbc, 0xca, 0x8b, 0x4e, 0xb5, 0x34, 0x14, 0x01,
      0x8b, 0x6b, 0xca, 0x19, 0xf4, 0x80, 0x14, 0x56, 0xc5, 0xe3, 0x6a, 0x94,
      0x00, 0x37, 0xac, 0x51, 0x6b, 0x0d, 0x64, 0x12, 0xba, 0x44, 0xec, 0x6b,
      0x4f, 0x26, 0x8a, 0x55, 0xef, 0x1c, 0x5f, 0xfb, 0xf1, 0x8a, 0x2f, 0x4e,
      0x35, 0x22, 0xbb, 0x7b, 0x6e, 0xd8, 0x97, 0x74, 0xb7, 0x9b, 0xff, 0xa2,
      0x2f, 0x7d, 0x31, 0x02, 0x16, 0x55, 0x65, 0x64, 0x2d, 0xe0, 0xd4, 0x3a,
      0x95, 0x5e, 0x96, 0xa1, 0xf2, 0xe8, 0x0e, 0x54, 0x30, 0x67, 0x1d, 0x72,
      0x66, 0xeb, 0x4f, 0x90, 0x5d, 0xc8, 0xff, 0x5e, 0x10, 0x6d, 0xc5, 0x58,
      0x8e, 0x5b, 0x02, 0x89, 0xe4, 0x9a, 0x49, 0x13, 0x94, 0x0e, 0x39, 0x2a,
      0x97, 0x06, 0x26, 0x16, 0xd2, 0xbd, 0xa3, 0x81, 0x55, 0x47, 0x1b, 0x7d,
      0x36, 0x0c, 0xfb, 0x94, 0x68, 0x1c, 0x70, 0x2f, 0x60, 0xed, 0x2d, 0x4d,
      0xe6, 0x14, 0xea, 0x72, 0xbf, 0x1c, 0x53, 0x16, 0x0e, 0x63, 0x17, 0x9f,
      0x6c, 0x5b, 0x89, 0x7b, 0x59, 0x49, 0x2b, 0xee, 0x21, 0x91, 0x08, 0x30,
      0x9f, 0x0b, 0x7b, 0x8c, 0xb2, 0xb1, 0x36, 0xc3, 0x46, 0xa5, 0xe9, 0x8b,
      0x8b, 0x4b, 0x84, 0x15, 0xfb, 0x1d, 0x71, 0x3b, 0xae, 0x06, 0x79, 0x11,
      0xe3, 0x05, 0x7f, 0x1c, 0x33, 0x5b, 0x4b, 0x7e, 0x39, 0x10, 0x1e, 0xaf,
      0xd5, 0xd2, 0x8f, 0x01, 0x89, 0x03, 0x7e, 0x43, 0x34, 0xf4, 0xfd, 0xb9,
      0x03, 0x84, 0x27, 0xb1, 0xd1, 0x19, 0xa6, 0x70, 0x2a, 0xa8, 0x23, 0x33,
      0x19, 0xcc, 0x97, 0xd4, 0x96, 0xcc, 0x28, 0x9a, 0xe8, 0xc9, 0x56, 0xdd,
      0xc8, 0x40, 0x42, 0x65, 0x9a, 0x2d, 0x43, 0xd6, 0xaa, 0x22, 0xf1, 0x2b,
      0x81, 0xab, 0x88, 0x4e},
     {},
     false,
     true},

    // an otherwise correct plaintext, but with a null byte on tenth
    // position (eight byte of padding), generates a random 11 byte long
    // plaintext
    {0,
     {0x63, 0xcb, 0x0b, 0xf6, 0x5f, 0xc8, 0x25, 0x5d, 0xd2, 0x9e, 0x17},
     {0xa7, 0xa3, 0x40, 0x67, 0x5a, 0x82, 0xc3, 0x0e, 0x22, 0x21, 0x9a, 0x55,
      0xbc, 0x07, 0xcd, 0xf3, 0x6d, 0x47, 0xd0, 0x18, 0x34, 0xc1, 0x83, 0x4f,
      0x91, 0x7f, 0x18, 0xb5, 0x17, 0x41, 0x9c, 0xe9, 0xde, 0x2a, 0x96, 0x46,
      0x0e, 0x74, 0x50, 0x24, 0x43, 0x64, 0x70, 0xed, 0x85, 0xe9, 0x42, 0x97,
      0xb2, 0x83, 0x53, 0x7d, 0x52, 0x18, 0x9c, 0x40, 0x6a, 0x3f, 0x53, 0x3c,
      0xb4, 0x05, 0xcc, 0x6a, 0x9d, 0xba, 0x46, 0xb4, 0x82, 0xce, 0x98, 0xb6,
      0xe3, 0xdd, 0x52, 0xd8, 0xfc, 0xe2, 0x23, 0x74, 0x25, 0x61, 0x7e, 0x38,
      0xc1, 0x1f, 0xbc, 0x46, 0xb6, 0x18, 0x97, 0xef, 0x20, 0x0d, 0x01, 0xe4,
      0xf2, 0x5f, 0x5f, 0x6c, 0x4c, 0x5b, 0x38, 0xcd, 0x0d, 0xe3, 0x8b, 0xa1,
      0x19, 0x08, 0xb8, 0x65, 0x95, 0xa8, 0x03, 0x6a, 0x08, 0xa4, 0x2a, 0x3d,
      0x05, 0xb7, 0x96, 0x00, 0xa9, 0x7a, 0xc1, 0x8b, 0xa3, 0x68, 0xa0, 0x8d,
      0x6c, 0xf6, 0xcc, 0xb6, 0x24, 0xf6, 0xe8, 0x00, 0x2a, 0xfc, 0x75, 0x59,
      0x9f, 0xba, 0x4d, 0xe3, 0xd4, 0xf3, 0xba, 0x7d, 0x20, 0x83, 0x91, 0xeb,
      0xe8, 0xd2, 0x1f, 0x82, 0x82, 0xb1, 0x8e, 0x2c, 0x10, 0x86, 0x9e, 0xb2,
      0x70, 0x2e, 0x68, 0xf9, 0x17, 0x6b, 0x42, 0xb0, 0xdd, 0xc9, 0xd7, 0x63,
      0xf0, 0xc8, 0x6b, 0xa0, 0xff, 0x92, 0xc9, 0x57, 0xaa, 0xea, 0xb7, 0x6d,
      0x9a, 0xb8, 0xda, 0x52, 0xea, 0x29, 0x7e, 0xc1, 0x1d, 0x92, 0xd7, 0x70,
      0x14, 0x6f, 0xaa, 0x1b, 0x30, 0x0e, 0x0f, 0x91, 0xef, 0x96, 0x9b, 0x53,
      0xe7, 0xd2, 0x90, 0x7f, 0xfc, 0x98, 0x4e, 0x9a, 0x9c, 0x9d, 0x11, 0xfb,
      0x7d, 0x6c, 0xba, 0x91, 0x97, 0x20, 0x59, 0xb4, 0x65, 0x06, 0xb0, 0x35,
      0xef, 0xec, 0x65, 0x75, 0xc4, 0x6d, 0x71, 0x14, 0xa6, 0xb9, 0x35, 0x86,
      0x48, 0x58, 0x44, 0x5f},
     {},
     false,
     true},

    // an otherwise correct plaintext, but with missing zero separator
    // decrypts to 11 byte random synthethic plaintext
    {0,
     {0x6f, 0x09, 0xa0, 0xb6, 0x26, 0x99, 0x33, 0x7c, 0x49, 0x7b, 0x0b},
     {0x3d, 0x1b, 0x97, 0xe7, 0xaa, 0x34, 0xea, 0xf1, 0xf4, 0xfc, 0x17, 0x1c,
      0xeb, 0x11, 0xdc, 0xff, 0xfd, 0x9a, 0x46, 0xa5, 0xb6, 0x96, 0x12, 0x05,
      0xb1, 0x0b, 0x30, 0x28, 0x18, 0xc1, 0xfc, 0xc9, 0xf4, 0xec, 0x78, 0xbf,
      0x18, 0xea, 0x0c, 0xee, 0x7e, 0x9f, 0xa5, 0xb1, 0x6f, 0xb4, 0xc6, 0x11,
      0x46, 0x3b, 0x36, 0x8b, 0x33, 0x12, 0xac, 0x11, 0xcf, 0x9c, 0x06, 0xb7,
      0xcf, 0x72, 0xb5, 0x4e, 0x28, 0x48, 0x48, 0xa5, 0x08, 0xd3, 0xf0, 0x23,
      0x28, 0xc6, 0x2c, 0x29, 0x99, 0xd0, 0xfb, 0x60, 0x92, 0x9f, 0x81, 0x78,
      0x3c, 0x7a, 0x25, 0x68, 0x91, 0xbc, 0x2f, 0xf4, 0xd9, 0x1d, 0xf2, 0xaf,
      0x96, 0xa2, 0x4f, 0xc5, 0x70, 0x1a, 0x18, 0x23, 0xaf, 0x93, 0x9c, 0xe6,
      0xdb, 0xdc, 0x51, 0x06, 0x08, 0xe3, 0xd4, 0x1e, 0xec, 0x17, 0x2a, 0xd2,
      0xd5, 0x1b, 0x9f, 0xc6, 0x1b, 0x42, 0x17, 0xc9, 0x23, 0xca, 0xdc, 0xf5,
      0xba, 0xc3, 0x21, 0x35, 0x5e, 0xf8, 0xbe, 0x5e, 0x5f, 0x09, 0x0c, 0xdc,
      0x2b, 0xd0, 0xc6, 0x97, 0xd9, 0x05, 0x82, 0x47, 0xdb, 0x3a, 0xd6, 0x13,
      0xfd, 0xce, 0x87, 0xd2, 0x95, 0x5a, 0x6d, 0x1c, 0x94, 0x8a, 0x51, 0x60,
      0xf9, 0x3d, 0xa2, 0x1f, 0x73, 0x1d, 0x74, 0x13, 0x7f, 0x5d, 0x1f, 0x53,
      0xa1, 0x92, 0x3a, 0xdb, 0x51, 0x3d, 0x2e, 0x6e, 0x15, 0x89, 0xd4, 0x4c,
      0xc0, 0x79, 0xf4, 0xc6, 0xdd, 0xd4, 0x71, 0xd3, 0x8a, 0xc8, 0x2d, 0x20,
      0xd8, 0xb1, 0xd2, 0x1f, 0x8d, 0x65, 0xf3, 0xb6, 0x90, 0x70, 0x86, 0x80,
      0x9f, 0x41, 0x23, 0xe0, 0x8d, 0x86, 0xfb, 0x38, 0x72, 0x95, 0x85, 0xde,
      0x02, 0x6a, 0x48, 0x5d, 0x8f, 0x0e, 0x70, 0x3f, 0xd4, 0x77, 0x2f, 0x66,
      0x68, 0xfe, 0xbf, 0x67, 0xdf, 0x94, 0x7b, 0x82, 0x19, 0x5f, 0xa3, 0x86,
      0x7e, 0x3a, 0x30, 0x65},
     {},
     false,
     true}};

/* 2049 bit key from Hubert's Bleichenbacher tests */
static const std::vector<uint8_t> kRsaBb2049{
    0x30, 0x82, 0x04, 0xbf, 0x02, 0x01, 0x01, 0x30, 0x0d, 0x06, 0x09, 0x2a,
    0x86, 0x48, 0x86, 0xf7, 0x0d, 0x01, 0x01, 0x01, 0x05, 0x00, 0x04, 0x82,
    0x04, 0xa9, 0x30, 0x82, 0x04, 0xa5, 0x02, 0x01, 0x00, 0x02, 0x82, 0x01,
    0x01, 0x01, 0x55, 0xf8, 0x89, 0x55, 0x6a, 0x17, 0x75, 0xf1, 0xc7, 0xa7,
    0x78, 0x6a, 0x50, 0xb1, 0x8b, 0xc2, 0x8c, 0x9e, 0x98, 0x6e, 0xde, 0x56,
    0x67, 0xca, 0xb3, 0x9b, 0x84, 0x12, 0x4e, 0x90, 0xeb, 0xa7, 0x5c, 0x1d,
    0xb0, 0x83, 0xac, 0x3e, 0x44, 0x3b, 0xba, 0x94, 0xdc, 0x23, 0x56, 0x0f,
    0x75, 0xe3, 0xa8, 0x16, 0x93, 0xa2, 0xa4, 0x3b, 0xdc, 0x74, 0x26, 0xd8,
    0xc4, 0xea, 0xfe, 0x68, 0xc8, 0x5d, 0xe0, 0xfe, 0x75, 0x7f, 0x6e, 0x49,
    0xbb, 0x9e, 0xd4, 0x47, 0xe6, 0x02, 0x43, 0x08, 0x00, 0xdb, 0xb0, 0x4c,
    0xeb, 0x22, 0xe7, 0xfa, 0x57, 0xa1, 0x8d, 0x33, 0x8f, 0xb6, 0x60, 0x26,
    0xcd, 0xb4, 0x67, 0xe7, 0x0c, 0xc0, 0x40, 0xe7, 0xd3, 0x67, 0xef, 0x40,
    0x3c, 0x7b, 0xf1, 0xe3, 0xdf, 0x62, 0x46, 0x50, 0x09, 0x46, 0x31, 0xf2,
    0x1e, 0xaf, 0xd2, 0xfb, 0x5b, 0xc9, 0x15, 0xff, 0x04, 0x37, 0x9a, 0xcd,
    0x11, 0x12, 0xf7, 0x32, 0xc0, 0xb4, 0x66, 0x07, 0xc1, 0x78, 0xd3, 0x8a,
    0x20, 0xf5, 0x2e, 0xda, 0x50, 0x9f, 0x2f, 0x9c, 0x04, 0x05, 0xd5, 0x10,
    0x69, 0xe8, 0x0c, 0xcf, 0x94, 0x15, 0x54, 0xd0, 0x47, 0x04, 0x67, 0x50,
    0x5c, 0x3c, 0xf5, 0x41, 0xea, 0x08, 0x97, 0xdf, 0xc9, 0xf4, 0x00, 0xce,
    0xcb, 0x29, 0x8f, 0xfc, 0x75, 0x33, 0x72, 0xd9, 0xf6, 0x93, 0x3a, 0xf1,
    0x74, 0xcc, 0x40, 0xed, 0x96, 0xd4, 0x67, 0x03, 0x17, 0x33, 0xb9, 0x7f,
    0x8c, 0xdd, 0xd3, 0xf9, 0x2b, 0xc3, 0xa0, 0x3e, 0xa8, 0x57, 0x6c, 0x41,
    0x7f, 0x24, 0x00, 0x7b, 0x5e, 0x4f, 0x75, 0x01, 0x10, 0x5b, 0x54, 0x4d,
    0xe9, 0xfa, 0xdc, 0xdf, 0xfa, 0xdf, 0x98, 0xdf, 0xb4, 0xbb, 0x05, 0xb8,
    0x19, 0x9f, 0x3f, 0x85, 0xac, 0xfd, 0x91, 0xf7, 0xa9, 0xa0, 0x94, 0xb9,
    0xa3, 0x83, 0xf5, 0x04, 0x90, 0x97, 0x02, 0x03, 0x01, 0x00, 0x01, 0x02,
    0x82, 0x01, 0x01, 0x01, 0x19, 0xc2, 0xb3, 0xf5, 0x0a, 0x7a, 0xd6, 0x15,
    0x26, 0x79, 0xd7, 0xff, 0x51, 0x09, 0x58, 0xac, 0x2d, 0x8c, 0xa6, 0xf0,
    0x02, 0x85, 0x92, 0xf3, 0x32, 0xd5, 0x5a, 0x16, 0x73, 0x61, 0x78, 0xa8,
    0xe6, 0x7f, 0x17, 0xe7, 0x05, 0xce, 0x30, 0x0e, 0x3e, 0x87, 0x54, 0x72,
    0x51, 0x00, 0x60, 0x13, 0xf9, 0x74, 0xd0, 0xa3, 0xdb, 0x49, 0xef, 0x34,
    0x4c, 0xa5, 0xa2, 0x6a, 0x34, 0xc0, 0x45, 0x07, 0x04, 0xd0, 0xe4, 0x22,
    0xe0, 0xce, 0x23, 0xa6, 0x94, 0x25, 0xc1, 0x5f, 0xef, 0xb6, 0xf2, 0x6e,
    0x10, 0x6e, 0xef, 0xf6, 0x4c, 0xc8, 0xb9, 0xd7, 0x44, 0x2e, 0x4d, 0xa4,
    0xe8, 0xc8, 0x50, 0x08, 0xea, 0xeb, 0x36, 0x58, 0x59, 0xa2, 0x29, 0x4f,
    0xa3, 0x93, 0x7b, 0xc2, 0x6b, 0xe5, 0x63, 0x32, 0xe7, 0xd8, 0x1e, 0x2c,
    0x16, 0x0e, 0xf6, 0x35, 0xcc, 0x52, 0x8a, 0xa7, 0xbe, 0x55, 0xe6, 0x33,
    0xa7, 0x23, 0xdb, 0xc1, 0xe1, 0x6b, 0xa2, 0x9e, 0x52, 0xb2, 0x9a, 0xef,
    0x2f, 0x9e, 0x56, 0x54, 0xfd, 0xc0, 0x66, 0x6b, 0xb0, 0xfc, 0x25, 0x4a,
    0xcb, 0xe8, 0x0e, 0x63, 0x87, 0x4f, 0x0f, 0x5f, 0x02, 0x07, 0x82, 0xe3,
    0xc9, 0xdc, 0xfc, 0x25, 0x20, 0xd0, 0xc9, 0xc4, 0xa7, 0xb6, 0x34, 0xe4,
    0x50, 0x3f, 0xbb, 0x49, 0x3e, 0x1a, 0xaf, 0xee, 0xb3, 0xf8, 0x8b, 0xd7,
    0xa1, 0x33, 0x98, 0x72, 0x5d, 0xae, 0x6f, 0xe3, 0x99, 0xe7, 0x75, 0xcd,
    0x5d, 0x4c, 0xf0, 0x9f, 0xc8, 0x38, 0x34, 0x7c, 0x4c, 0x98, 0xda, 0xb1,
    0xa4, 0x88, 0x3c, 0xce, 0x62, 0x05, 0x13, 0x61, 0x5a, 0xfa, 0xa1, 0x0a,
    0x63, 0x36, 0x8e, 0x6d, 0x7b, 0x79, 0xdf, 0x41, 0x66, 0xab, 0x16, 0x27,
    0x39, 0xef, 0x51, 0x5a, 0x44, 0x02, 0xee, 0x1e, 0x06, 0x01, 0xc5, 0xa5,
    0x5b, 0xc7, 0x1d, 0xf0, 0xe3, 0x0e, 0xdf, 0x81, 0x02, 0x81, 0x81, 0x01,
    0x88, 0xf6, 0x93, 0x60, 0xf0, 0x1e, 0x18, 0xd9, 0xa2, 0xde, 0x29, 0x52,
    0x53, 0xd2, 0x52, 0xc3, 0x1e, 0x44, 0x76, 0xce, 0xa5, 0xff, 0x7b, 0xf8,
    0x41, 0x3d, 0xf7, 0xfd, 0xe3, 0x56, 0x52, 0x3c, 0xdc, 0x97, 0x68, 0x05,
    0xf8, 0x4f, 0xc0, 0xdd, 0xec, 0x77, 0x0d, 0xf0, 0x6c, 0xed, 0x06, 0x5c,
    0x81, 0x13, 0x48, 0x75, 0x4b, 0x34, 0x6a, 0xf1, 0x69, 0x75, 0x68, 0x77,
    0xfd, 0x3b, 0x3d, 0x56, 0x86, 0x82, 0xc8, 0x78, 0x7d, 0x0b, 0x31, 0x4e,
    0xf6, 0xac, 0x67, 0xd6, 0x5e, 0x81, 0x33, 0x39, 0x8b, 0x62, 0xa0, 0x83,
    0xc0, 0xf8, 0x76, 0x5c, 0x5a, 0xd4, 0x0d, 0x5a, 0x81, 0xf9, 0xbb, 0xdc,
    0xe2, 0x52, 0x7e, 0xd7, 0xe9, 0x50, 0x08, 0xcb, 0x10, 0x29, 0xcb, 0x4c,
    0xab, 0xd1, 0xf9, 0xe9, 0xbe, 0xdf, 0xc2, 0x86, 0xc9, 0x65, 0x52, 0x25,
    0x5d, 0xa7, 0xea, 0xb1, 0x92, 0x17, 0x8e, 0xf7, 0x02, 0x81, 0x81, 0x00,
    0xde, 0xc7, 0xcf, 0x11, 0xda, 0xde, 0x83, 0xa4, 0xc4, 0x3d, 0x2f, 0x80,
    0x19, 0x7f, 0x21, 0xfd, 0x5d, 0x46, 0xfd, 0x57, 0xb4, 0x31, 0xf4, 0x4f,
    0xe8, 0x1a, 0x1d, 0xe3, 0x7f, 0x6a, 0x09, 0x1f, 0xfc, 0x04, 0x64, 0xed,
    0x97, 0x1d, 0xc8, 0x50, 0x88, 0x35, 0xad, 0xe6, 0xcc, 0x5f, 0x56, 0x6f,
    0x39, 0x65, 0x61, 0x3a, 0x8b, 0x36, 0x79, 0x8c, 0x92, 0xe6, 0xe2, 0x3f,
    0x52, 0xef, 0x90, 0x7e, 0x95, 0x67, 0xe3, 0x41, 0xbe, 0xbc, 0x53, 0x37,
    0x18, 0x96, 0x25, 0xfb, 0xbe, 0xab, 0x1f, 0x3b, 0x7b, 0x3f, 0x92, 0xff,
    0xb2, 0x68, 0x1e, 0x6e, 0xf5, 0xa7, 0x84, 0xa8, 0xc2, 0xd7, 0x8f, 0x7c,
    0x2d, 0x89, 0xaa, 0xaa, 0x24, 0xd2, 0xce, 0xdb, 0xd0, 0x66, 0x81, 0xcf,
    0xe6, 0x5c, 0x36, 0xc7, 0xbf, 0xa3, 0xc5, 0xba, 0x13, 0x51, 0x62, 0x22,
    0x2e, 0xf5, 0xc2, 0xe9, 0x14, 0xc9, 0x83, 0x61, 0x02, 0x81, 0x80, 0x12,
    0x09, 0x3f, 0x3a, 0x73, 0xca, 0xed, 0xd9, 0x0f, 0x60, 0xa3, 0x04, 0xe4,
    0x54, 0x02, 0xf8, 0x71, 0xab, 0x32, 0xc8, 0xc9, 0x55, 0xb0, 0x9a, 0xf4,
    0x63, 0xa3, 0xbe, 0x43, 0x70, 0xf2, 0xd5, 0x58, 0x4a, 0x9a, 0xbb, 0xab,
    0x69, 0xfd, 0xb0, 0x31, 0xea, 0x44, 0xf9, 0x84, 0x06, 0x5d, 0x04, 0x61,
    0xe8, 0x40, 0xab, 0x21, 0x88, 0x86, 0x60, 0x0e, 0x37, 0x15, 0x54, 0x6c,
    0x8b, 0x0b, 0x85, 0xad, 0x26, 0xd3, 0x8c, 0xb4, 0x30, 0x8f, 0x52, 0xd0,
    0x7f, 0x99, 0x44, 0x7d, 0x91, 0xf0, 0x87, 0xf3, 0x9d, 0xd3, 0x40, 0x38,
    0xdb, 0x2e, 0x93, 0x8e, 0x97, 0xad, 0x05, 0x3a, 0x71, 0xfb, 0xed, 0x67,
    0x75, 0xe1, 0xdc, 0x87, 0x18, 0xe5, 0x4e, 0x6c, 0xaf, 0x7e, 0x65, 0x46,
    0x7d, 0x9c, 0xba, 0xdd, 0xc7, 0xe7, 0x65, 0xc8, 0x58, 0x9e, 0x2c, 0x98,
    0xdf, 0xdc, 0x25, 0xca, 0x4e, 0xca, 0x81, 0x02, 0x81, 0x81, 0x00, 0x8c,
    0xce, 0x61, 0x34, 0x79, 0xcf, 0x96, 0x08, 0xf7, 0xf7, 0x6c, 0x24, 0x5c,
    0xf9, 0x1b, 0xb4, 0x95, 0xd6, 0x1e, 0x9d, 0xe6, 0x48, 0x84, 0x90, 0x54,
    0xb4, 0xdd, 0x1b, 0x43, 0x16, 0xf3, 0xf9, 0x81, 0x42, 0x0d, 0xc0, 0x95,
    0x78, 0xbf, 0x79, 0x16, 0xfe, 0x46, 0x91, 0xcf, 0xae, 0x9a, 0x64, 0xe6,
    0x34, 0x0b, 0x86, 0x03, 0x23, 0x45, 0x23, 0xf2, 0x5d, 0x77, 0xb6, 0x6a,
    0x66, 0xfc, 0x3e, 0xe5, 0x93, 0xa9, 0xf1, 0x8d, 0xea, 0x5d, 0xf6, 0x3e,
    0xd5, 0xf7, 0xdf, 0xeb, 0x9d, 0x20, 0xba, 0x69, 0xa5, 0xbe, 0xf5, 0x59,
    0xff, 0xb0, 0xec, 0x94, 0xdb, 0x72, 0x5f, 0x6f, 0xf6, 0xea, 0xbb, 0xa3,
    0xd4, 0x95, 0x47, 0xc0, 0xca, 0x74, 0xf0, 0x3e, 0x01, 0xec, 0x1e, 0x49,
    0x0d, 0x13, 0x9a, 0xa0, 0xa7, 0x94, 0x7b, 0x8d, 0x66, 0x2c, 0xce, 0x4a,
    0x3c, 0x0f, 0x1b, 0x5e, 0x86, 0x17, 0x41, 0x02, 0x81, 0x81, 0x00, 0xf9,
    0x95, 0x9b, 0x34, 0xc4, 0xbc, 0xa8, 0xce, 0x48, 0x88, 0x78, 0x1b, 0x31,
    0xb1, 0xe9, 0xb5, 0xd8, 0xad, 0xf5, 0xd0, 0xd3, 0xe3, 0xed, 0x54, 0x5e,
    0x83, 0x67, 0xd3, 0xf8, 0x54, 0x5b, 0xa6, 0x44, 0x32, 0xb8, 0x87, 0x30,
    0x35, 0xef, 0x88, 0x1c, 0x2b, 0xcd, 0xe0, 0x0d, 0x18, 0x09, 0xf9, 0x2c,
    0x40, 0xd8, 0x78, 0x37, 0xb5, 0xc4, 0xf9, 0xac, 0xf4, 0x8b, 0x36, 0xb8,
    0xdc, 0x53, 0xa5, 0x95, 0x61, 0xa4, 0x56, 0x52, 0x34, 0x02, 0xd1, 0xe8,
    0xfa, 0x3a, 0xf3, 0x00, 0xe5, 0x4c, 0x91, 0xb6, 0x3e, 0x6c, 0xee, 0x06,
    0xfe, 0x6d, 0xe6, 0x66, 0xf3, 0x92, 0x95, 0x82, 0xa0, 0x3e, 0x1f, 0x45,
    0x4e, 0x77, 0x89, 0xfb, 0x07, 0x81, 0xa4, 0xd6, 0xfb, 0xb5, 0x26, 0xef,
    0x88, 0x16, 0x21, 0xfd, 0x1e, 0xac, 0xd2, 0x14, 0x66, 0xe4, 0xcd, 0xd9,
    0x8a, 0xed, 0x10, 0xf4, 0xe7, 0x6f, 0x79};

// Bleichenbacher 2049 tests
const std::vector<RsaDecryptTestVector> kRsaBb2049Vectors = {
    // malformed plaintext that generates a fake plaintext of length
    // specified by 3rd length from the end of PRF output
    {0,
     {0x42},
     {0x00, 0xb2, 0x6f, 0x64, 0x04, 0xb8, 0x26, 0x49, 0x62, 0x9f, 0x27, 0x04,
      0x49, 0x42, 0x82, 0x44, 0x37, 0x76, 0x92, 0x91, 0x22, 0xe2, 0x79, 0xa9,
      0xcf, 0x30, 0xb0, 0xc6, 0xfe, 0x81, 0x22, 0xa0, 0xa9, 0x04, 0x28, 0x70,
      0xd9, 0x7c, 0xc8, 0xef, 0x65, 0x49, 0x0f, 0xe5, 0x8f, 0x03, 0x1e, 0xb2,
      0x44, 0x23, 0x52, 0x19, 0x1f, 0x5f, 0xbc, 0x31, 0x10, 0x26, 0xb5, 0x14,
      0x7d, 0x32, 0xdf, 0x91, 0x45, 0x99, 0xf3, 0x8b, 0x82, 0x5e, 0xbb, 0x82,
      0x4a, 0xf0, 0xd6, 0x3f, 0x2d, 0x54, 0x1a, 0x24, 0x5c, 0x57, 0x75, 0xd1,
      0xc4, 0xb7, 0x86, 0x30, 0xe4, 0x99, 0x6c, 0xc5, 0xfe, 0x41, 0x3d, 0x38,
      0x45, 0x5a, 0x77, 0x6c, 0xf4, 0xed, 0xcc, 0x0a, 0xa7, 0xfc, 0xcb, 0x31,
      0xc5, 0x84, 0xd6, 0x05, 0x02, 0xed, 0x2b, 0x77, 0x39, 0x8f, 0x53, 0x6e,
      0x13, 0x7f, 0xf7, 0xba, 0x64, 0x30, 0xe9, 0x25, 0x8e, 0x21, 0xc2, 0xdb,
      0x5b, 0x82, 0xf5, 0x38, 0x0f, 0x56, 0x68, 0x76, 0x11, 0x0a, 0xc4, 0xc7,
      0x59, 0x17, 0x89, 0x00, 0xfb, 0xad, 0x7a, 0xb7, 0x0e, 0xa0, 0x7b, 0x1d,
      0xaf, 0x7a, 0x16, 0x39, 0xcb, 0xb4, 0x19, 0x65, 0x43, 0xa6, 0xcb, 0xe8,
      0x27, 0x1f, 0x35, 0xdd, 0xdb, 0x81, 0x20, 0x30, 0x4f, 0x6e, 0xef, 0x83,
      0x05, 0x9e, 0x1c, 0x5c, 0x56, 0x78, 0x71, 0x0f, 0x90, 0x4a, 0x6d, 0x76,
      0x0c, 0x4d, 0x1d, 0x8a, 0xd0, 0x76, 0xbe, 0x17, 0x90, 0x4b, 0x9e, 0x69,
      0x91, 0x00, 0x40, 0xb4, 0x79, 0x14, 0xa0, 0x17, 0x6f, 0xb7, 0xee, 0xa0,
      0xc0, 0x64, 0x44, 0xa6, 0xc4, 0xb8, 0x6d, 0x67, 0x4d, 0x19, 0xa5, 0x56,
      0xa1, 0xde, 0x54, 0x90, 0x37, 0x3c, 0xb0, 0x1c, 0xe3, 0x1b, 0xbd, 0x15,
      0xa5, 0x63, 0x33, 0x62, 0xd3, 0xd2, 0xcd, 0x7d, 0x4a, 0xf1, 0xb4, 0xc5,
      0x12, 0x12, 0x88, 0xb8, 0x94},
     {},
     false,
     true},

    // a valid ciphertext that starts with a null byte, decrypts to 11 byte
    // long value
    {0,
     // lorem ipsum
     {0x6c, 0x6f, 0x72, 0x65, 0x6d, 0x20, 0x69, 0x70, 0x73, 0x75, 0x6d},
     {0x01, 0x33, 0x00, 0xed, 0xbf, 0x0b, 0xb3, 0x57, 0x1e, 0x59, 0x88, 0x9f,
      0x7e, 0xd7, 0x69, 0x70, 0xbf, 0x6d, 0x57, 0xe1, 0xc8, 0x9b, 0xbb, 0x6d,
      0x1c, 0x39, 0x91, 0xd9, 0xdf, 0x8e, 0x65, 0xed, 0x54, 0xb5, 0x56, 0xd9,
      0x28, 0xda, 0x7d, 0x76, 0x8f, 0xac, 0xb3, 0x95, 0xbb, 0xcc, 0x81, 0xe9,
      0xf8, 0x57, 0x3b, 0x45, 0xcf, 0x81, 0x95, 0xdb, 0xd8, 0x5d, 0x83, 0xa5,
      0x92, 0x81, 0xcd, 0xdf, 0x41, 0x63, 0xae, 0xc1, 0x1b, 0x53, 0xb4, 0x14,
      0x00, 0x53, 0xe3, 0xbd, 0x10, 0x9f, 0x78, 0x7a, 0x7c, 0x3c, 0xec, 0x31,
      0xd5, 0x35, 0xaf, 0x1f, 0x50, 0xe0, 0x59, 0x8d, 0x85, 0xd9, 0x6d, 0x91,
      0xea, 0x01, 0x91, 0x3d, 0x07, 0x09, 0x7d, 0x25, 0xaf, 0x99, 0xc6, 0x74,
      0x64, 0xeb, 0xf2, 0xbb, 0x39, 0x6f, 0xb2, 0x8a, 0x92, 0x33, 0xe5, 0x6f,
      0x31, 0xf7, 0xe1, 0x05, 0xd7, 0x1a, 0x23, 0xe9, 0xef, 0x3b, 0x73, 0x6d,
      0x1e, 0x80, 0xe7, 0x13, 0xd1, 0x69, 0x17, 0x13, 0xdf, 0x97, 0x33, 0x47,
      0x79, 0x55, 0x2f, 0xc9, 0x4b, 0x40, 0xdd, 0x73, 0x3c, 0x72, 0x51, 0xbc,
      0x52, 0x2b, 0x67, 0x3d, 0x3e, 0xc9, 0x35, 0x4a, 0xf3, 0xdd, 0x4a, 0xd4,
      0x4f, 0xa7, 0x1c, 0x06, 0x62, 0x21, 0x3a, 0x57, 0xad, 0xa1, 0xd7, 0x51,
      0x49, 0x69, 0x7d, 0x0e, 0xb5, 0x5c, 0x05, 0x3a, 0xae, 0xd5, 0xff, 0xd0,
      0xb8, 0x15, 0x83, 0x2f, 0x45, 0x41, 0x79, 0x51, 0x9d, 0x37, 0x36, 0xfb,
      0x4f, 0xaf, 0x80, 0x84, 0x16, 0x07, 0x1d, 0xb0, 0xd0, 0xf8, 0x01, 0xac,
      0xa8, 0x54, 0x83, 0x11, 0xee, 0x70, 0x8c, 0x13, 0x1f, 0x4b, 0xe6, 0x58,
      0xb1, 0x5f, 0x6b, 0x54, 0x25, 0x68, 0x72, 0xc2, 0x90, 0x3a, 0xc7, 0x08,
      0xbd, 0x43, 0xb0, 0x17, 0xb0, 0x73, 0xb5, 0x70, 0x7b, 0xc8, 0x4c, 0x2c,
      0xd9, 0xda, 0x70, 0xe9, 0x67},
     {},
     false,
     true},

    // a valid ciphertext that starts with a null byte, decrypts to 11 byte
    // long value
    {0,
     // lorem ipsum
     {0x6c, 0x6f, 0x72, 0x65, 0x6d, 0x20, 0x69, 0x70, 0x73, 0x75, 0x6d},
     {0x00, 0x02, 0xaa, 0xdf, 0x84, 0x6a, 0x32, 0x9f, 0xad, 0xc6, 0x76, 0x09,
      0x80, 0x30, 0x3d, 0xbd, 0x87, 0xbf, 0xad, 0xfa, 0x78, 0xc2, 0x01, 0x5c,
      0xe4, 0xd6, 0xc5, 0x78, 0x2f, 0xd9, 0xd3, 0xf1, 0x07, 0x8b, 0xd3, 0xc0,
      0xa2, 0xc5, 0xbf, 0xbd, 0xd1, 0xc0, 0x24, 0x55, 0x2e, 0x50, 0x54, 0xd9,
      0x8b, 0x5b, 0xcd, 0xc9, 0x4e, 0x47, 0x6d, 0xd2, 0x80, 0xe6, 0x4d, 0x65,
      0x00, 0x89, 0x32, 0x65, 0x42, 0xce, 0x7c, 0x61, 0xd4, 0xf1, 0xab, 0x40,
      0x00, 0x4c, 0x2e, 0x6a, 0x88, 0xa8, 0x83, 0x61, 0x35, 0x68, 0x55, 0x6a,
      0x10, 0xf3, 0xf9, 0xed, 0xea, 0xb6, 0x7a, 0xe8, 0xdd, 0xdc, 0x1e, 0x6b,
      0x08, 0x31, 0xc2, 0x79, 0x3d, 0x27, 0x15, 0xde, 0x94, 0x3f, 0x7c, 0xe3,
      0x4c, 0x5c, 0x05, 0xd1, 0xb0, 0x9f, 0x14, 0x43, 0x1f, 0xde, 0x56, 0x6d,
      0x17, 0xe7, 0x6c, 0x9f, 0xee, 0xe9, 0x0d, 0x86, 0xa2, 0xc1, 0x58, 0x61,
      0x6e, 0xc8, 0x1d, 0xda, 0x0c, 0x64, 0x2f, 0x58, 0xc0, 0xba, 0x8f, 0xa4,
      0x49, 0x58, 0x43, 0x12, 0x4a, 0x72, 0x35, 0xd4, 0x6f, 0xb4, 0x06, 0x97,
      0x15, 0xa5, 0x1b, 0xf7, 0x10, 0xfd, 0x02, 0x42, 0x59, 0x13, 0x1b, 0xa9,
      0x4d, 0xa7, 0x35, 0x97, 0xac, 0xe4, 0x94, 0x85, 0x6c, 0x94, 0xe7, 0xa3,
      0xec, 0x26, 0x15, 0x45, 0x79, 0x3b, 0x09, 0x90, 0x27, 0x9b, 0x15, 0xfa,
      0x91, 0xc7, 0xfd, 0x13, 0xdb, 0xfb, 0x1d, 0xf2, 0xf2, 0x21, 0xda, 0xb9,
      0xfa, 0x9f, 0x7c, 0x1d, 0x21, 0xe4, 0x8a, 0xa4, 0x9f, 0x6a, 0xae, 0xcb,
      0xab, 0xf5, 0xee, 0x76, 0xdc, 0x6c, 0x2a, 0xf2, 0x31, 0x7f, 0xfb, 0x4e,
      0x30, 0x31, 0x15, 0x38, 0x6a, 0x97, 0xf8, 0x72, 0x9a, 0xfc, 0x3d, 0x0c,
      0x89, 0x41, 0x96, 0x69, 0x23, 0x5f, 0x1a, 0x3a, 0x69, 0x57, 0x0e, 0x08,
      0x36, 0xc7, 0x9f, 0xc1, 0x62},
     {},
     false,
     true},

    // a valid ciphertext that starts with two null bytes, decrypts to
    // 11 byte long value
    {0,
     // lorem ipsum
     {0x6c, 0x6f, 0x72, 0x65, 0x6d, 0x20, 0x69, 0x70, 0x73, 0x75, 0x6d},
     {0x00, 0x00, 0xf3, 0x6d, 0xa3, 0xb7, 0x2d, 0x8f, 0xf6, 0xde, 0xd7, 0x4e,
      0x7e, 0xfd, 0x08, 0xc0, 0x19, 0x08, 0xf3, 0xf5, 0xf0, 0xde, 0x7b, 0x55,
      0xea, 0xb9, 0x2b, 0x5f, 0x87, 0x51, 0x90, 0x80, 0x9c, 0x39, 0xd4, 0x16,
      0x2e, 0x1e, 0x66, 0x49, 0x61, 0x8f, 0x85, 0x4f, 0xd8, 0x4a, 0xea, 0xb0,
      0x39, 0x70, 0xd1, 0x6b, 0xb8, 0x14, 0xe9, 0x99, 0x85, 0x2c, 0x06, 0xde,
      0x38, 0xd8, 0x2b, 0x95, 0xc0, 0xf3, 0x2e, 0x2a, 0x7b, 0x57, 0x14, 0x02,
      0x1f, 0xe3, 0x03, 0x38, 0x9b, 0xe9, 0xc0, 0xea, 0xc2, 0x4c, 0x90, 0xa6,
      0xb7, 0x21, 0x0f, 0x92, 0x9d, 0x39, 0x0f, 0xab, 0xf9, 0x03, 0xd4, 0x4e,
      0x04, 0x11, 0x0b, 0xb7, 0xa7, 0xfd, 0x6c, 0x38, 0x3c, 0x27, 0x58, 0x04,
      0x72, 0x1e, 0xfa, 0x6d, 0x7c, 0x93, 0xaa, 0x64, 0xc0, 0xbb, 0x2b, 0x18,
      0xd9, 0x7c, 0x52, 0x20, 0xa8, 0x46, 0xc6, 0x6a, 0x48, 0x95, 0xae, 0x52,
      0xad, 0xdd, 0xbe, 0x2a, 0x99, 0x96, 0x82, 0x5e, 0x01, 0x35, 0x85, 0xad,
      0xce, 0xc4, 0xb3, 0x2b, 0xa6, 0x1d, 0x78, 0x27, 0x37, 0xbd, 0x34, 0x3e,
      0x5f, 0xab, 0xd6, 0x8e, 0x8a, 0x95, 0xb8, 0xb1, 0x34, 0x03, 0x18, 0x55,
      0x98, 0x60, 0x79, 0x2d, 0xd7, 0x0d, 0xff, 0xbe, 0x05, 0xa1, 0x05, 0x2b,
      0x54, 0xcb, 0xfb, 0x48, 0xcf, 0xa7, 0xbb, 0x3c, 0x19, 0xce, 0xa5, 0x20,
      0x76, 0xbd, 0xda, 0xc5, 0xc2, 0x5e, 0xe2, 0x76, 0xf1, 0x53, 0xa6, 0x10,
      0xf6, 0xd0, 0x6e, 0xd6, 0x96, 0xd1, 0x92, 0xd8, 0xae, 0x45, 0x07, 0xff,
      0xae, 0x4e, 0x5b, 0xdd, 0xa1, 0x0a, 0x62, 0x5d, 0x6b, 0x67, 0xf3, 0x2f,
      0x7c, 0xff, 0xcd, 0x48, 0xde, 0xe2, 0x43, 0x1f, 0xe6, 0x6f, 0x61, 0x05,
      0xf9, 0xd1, 0x7e, 0x61, 0x1c, 0xdc, 0xc6, 0x74, 0x86, 0x8e, 0x81, 0x69,
      0x2a, 0x36, 0x0f, 0x40, 0x52},
     {},
     false,
     true},

    // a random ciphertext that generates a fake 11 byte plaintext
    // and fails the padding check
    {0,
     {0x11, 0x89, 0xb6, 0xf5, 0x49, 0x8f, 0xd6, 0xdf, 0x53, 0x2b, 0x00},
     {0x00, 0xf9, 0x10, 0x20, 0x08, 0x30, 0xfc, 0x8f, 0xff, 0x47, 0x8e, 0x99,
      0xe1, 0x45, 0xf1, 0x47, 0x4b, 0x31, 0x2e, 0x25, 0x12, 0xd0, 0xf9, 0x0b,
      0x8c, 0xef, 0x77, 0xf8, 0x00, 0x1d, 0x09, 0x86, 0x16, 0x88, 0xc1, 0x56,
      0xd1, 0xcb, 0xaf, 0x8a, 0x89, 0x57, 0xf7, 0xeb, 0xf3, 0x5f, 0x72, 0x44,
      0x66, 0x95, 0x2d, 0x05, 0x24, 0xca, 0xd4, 0x8a, 0xad, 0x4f, 0xba, 0x1e,
      0x45, 0xce, 0x8e, 0xa2, 0x7e, 0x8f, 0x3b, 0xa4, 0x41, 0x31, 0xb7, 0x83,
      0x1b, 0x62, 0xd6, 0x0c, 0x07, 0x62, 0x66, 0x1f, 0x4c, 0x1d, 0x1a, 0x88,
      0xcd, 0x06, 0x26, 0x3a, 0x25, 0x9a, 0xbf, 0x1b, 0xa9, 0xe6, 0xb0, 0xb1,
      0x72, 0x06, 0x9a, 0xfb, 0x86, 0xa7, 0xe8, 0x83, 0x87, 0x72, 0x6f, 0x8a,
      0xb3, 0xad, 0xb3, 0x0b, 0xfd, 0x6b, 0x3f, 0x6b, 0xe6, 0xd8, 0x5d, 0x5d,
      0xfd, 0x04, 0x4e, 0x7e, 0xf0, 0x52, 0x39, 0x54, 0x74, 0xa9, 0xcb, 0xb1,
      0xc3, 0x66, 0x7a, 0x92, 0x78, 0x0b, 0x43, 0xa2, 0x26, 0x93, 0x01, 0x5a,
      0xf6, 0xc5, 0x13, 0x04, 0x1b, 0xda, 0xf8, 0x7d, 0x43, 0xb2, 0x4d, 0xdd,
      0x24, 0x4e, 0x79, 0x1e, 0xea, 0xea, 0x10, 0x66, 0xe1, 0xf4, 0x91, 0x71,
      0x17, 0xb3, 0xa4, 0x68, 0xe2, 0x2e, 0x0f, 0x73, 0x58, 0x85, 0x2b, 0xb9,
      0x81, 0x24, 0x8d, 0xe4, 0xd7, 0x20, 0xad, 0xd2, 0xd1, 0x5d, 0xcc, 0xba,
      0x62, 0x80, 0x35, 0x59, 0x35, 0xb6, 0x7c, 0x96, 0xf9, 0xdc, 0xb6, 0xc4,
      0x19, 0xcc, 0x38, 0xab, 0x9f, 0x6f, 0xba, 0x2d, 0x64, 0x9e, 0xf2, 0x06,
      0x6e, 0x0c, 0x34, 0xc9, 0xf7, 0x88, 0xae, 0x49, 0xba, 0xbd, 0x90, 0x25,
      0xfa, 0x85, 0xb2, 0x11, 0x13, 0xe5, 0x6c, 0xe4, 0xf4, 0x3a, 0xa1, 0x34,
      0xc5, 0x12, 0xb0, 0x30, 0xdd, 0x7a, 0xc7, 0xce, 0x82, 0xe7, 0x6f, 0x0b,
      0xe9, 0xce, 0x09, 0xeb, 0xca},
     {},
     false,
     true},

    // an otherwise correct plaintext, but with wrong first byte
    // (0x01 instead of 0x00), generates a random 11 byte long plaintext
    {0,
     {0xf6, 0xd0, 0xf5, 0xb7, 0x80, 0x82, 0xfe, 0x61, 0xc0, 0x46, 0x74},
     {
         0x00, 0x2c, 0x9d, 0xdc, 0x36, 0xba, 0x4c, 0xf0, 0x03, 0x86, 0x92, 0xb2,
         0xd3, 0xa1, 0xc6, 0x1a, 0x4b, 0xb3, 0x78, 0x6a, 0x97, 0xce, 0x2e, 0x46,
         0xa3, 0xba, 0x74, 0xd0, 0x31, 0x58, 0xae, 0xef, 0x45, 0x6c, 0xe0, 0xf4,
         0xdb, 0x04, 0xdd, 0xa3, 0xfe, 0x06, 0x22, 0x68, 0xa1, 0x71, 0x12, 0x50,
         0xa1, 0x8c, 0x69, 0x77, 0x8a, 0x62, 0x80, 0xd8, 0x8e, 0x13, 0x3a, 0x16,
         0x25, 0x4e, 0x1f, 0x0e, 0x30, 0xce, 0x8d, 0xac, 0x9b, 0x57, 0xd2, 0xe3,
         0x9a, 0x2f, 0x7d, 0x7b, 0xe3, 0xee, 0x4e, 0x08, 0xae, 0xc2, 0xfd, 0xbe,
         0x8d, 0xad, 0xad, 0x7f, 0xdb, 0xf4, 0x42, 0xa2, 0x9a, 0x8f, 0xb4, 0x08,
         0x57, 0x40, 0x7b, 0xf6, 0xbe, 0x35, 0x59, 0x6b, 0x8e, 0xef, 0xb5, 0xc2,
         0xb3, 0xf5, 0x8b, 0x89, 0x44, 0x52, 0xc2, 0xdc, 0x54, 0xa6, 0x12, 0x3a,
         0x1a, 0x38, 0xd6, 0x42, 0xe2, 0x37, 0x51, 0x74, 0x65, 0x97, 0xe0, 0x8d,
         0x71, 0xac, 0x92, 0x70, 0x4a, 0xdc, 0x17, 0x80, 0x3b, 0x19, 0xe1, 0x31,
         0xb4, 0xd1, 0x92, 0x78, 0x81, 0xf4, 0x3b, 0x02, 0x00, 0xe6, 0xf9, 0x56,
         0x58, 0xf5, 0x59, 0xf9, 0x12, 0xc8, 0x89, 0xb4, 0xcd, 0x51, 0x86, 0x27,
         0x84, 0x36, 0x48, 0x96, 0xcd, 0x6e, 0x86, 0x18, 0xf4, 0x85, 0xa9, 0x92,
         0xf8, 0x29, 0x97, 0xad, 0x6a, 0x09, 0x17, 0xe3, 0x2a, 0xe5, 0x87, 0x2e,
         0xaf, 0x85, 0x00, 0x92, 0xb2, 0xd6, 0xc7, 0x82, 0xad, 0x35, 0xf4, 0x87,
         0xb7, 0x96, 0x82, 0x33, 0x3c, 0x17, 0x50, 0xc6, 0x85, 0xd7, 0xd3, 0x2a,
         0xb3, 0xe1, 0x53, 0x8f, 0x31, 0xdc, 0xaa, 0x5e, 0x7d, 0x5d, 0x28, 0x25,
         0x87, 0x52, 0x42, 0xc8, 0x39, 0x47, 0x30, 0x8d, 0xcf, 0x63, 0xba, 0x4b,
         0xff, 0xf2, 0x03, 0x34, 0xc9, 0xc1, 0x40, 0xc8, 0x37, 0xdb, 0xdb, 0xae,
         0x7a, 0x8d, 0xee, 0x72, 0xff,
     },
     {},
     false,
     true},

    // an otherwise correct plaintext, but with wrong second byte
    // (0x01 instead of 0x02), generates a random 11 byte long plaintext
    {0,
     {0x1a, 0xb2, 0x87, 0xfc, 0xef, 0x3f, 0xf1, 0x70, 0x67, 0x91, 0x4d},
     {0x00, 0xc5, 0xd7, 0x78, 0x26, 0xc1, 0xab, 0x7a, 0x34, 0xd6, 0x39, 0x0f,
      0x9d, 0x34, 0x2d, 0x5d, 0xbe, 0x84, 0x89, 0x42, 0xe2, 0x61, 0x82, 0x87,
      0x95, 0x2b, 0xa0, 0x35, 0x0d, 0x7d, 0xe6, 0x72, 0x61, 0x12, 0xe9, 0xce,
      0xbc, 0x39, 0x1a, 0x0f, 0xae, 0x18, 0x39, 0xe2, 0xbf, 0x16, 0x82, 0x29,
      0xe3, 0xe0, 0xd7, 0x1d, 0x41, 0x61, 0x80, 0x15, 0x09, 0xf1, 0xf2, 0x8f,
      0x6e, 0x14, 0x87, 0xca, 0x52, 0xdf, 0x05, 0xc4, 0x66, 0xb6, 0xb0, 0xa6,
      0xfb, 0xbe, 0x57, 0xa3, 0x26, 0x8a, 0x97, 0x06, 0x10, 0xec, 0x0b, 0xea,
      0xc3, 0x9e, 0xc0, 0xfa, 0x67, 0xba, 0xbc, 0xe1, 0xef, 0x2a, 0x86, 0xbf,
      0x77, 0x46, 0x6d, 0xc1, 0x27, 0xd7, 0xd0, 0xd2, 0x96, 0x2c, 0x20, 0xe6,
      0x65, 0x93, 0x12, 0x6f, 0x27, 0x68, 0x63, 0xcd, 0x38, 0xdc, 0x63, 0x51,
      0x42, 0x8f, 0x88, 0x4c, 0x13, 0x84, 0xf6, 0x7c, 0xad, 0x0a, 0x0f, 0xfd,
      0xbc, 0x2a, 0xf1, 0x67, 0x11, 0xfb, 0x68, 0xdc, 0x55, 0x9b, 0x96, 0xb3,
      0x7b, 0x4f, 0x04, 0xcd, 0x13, 0x3f, 0xfc, 0x7d, 0x79, 0xc4, 0x3c, 0x42,
      0xca, 0x49, 0x48, 0xfa, 0x89, 0x5b, 0x9d, 0xae, 0xb8, 0x53, 0x15, 0x0c,
      0x8a, 0x51, 0x69, 0x84, 0x9b, 0x73, 0x0c, 0xc7, 0x7d, 0x68, 0xb0, 0x21,
      0x7d, 0x6c, 0x0e, 0x3d, 0xbf, 0x38, 0xd7, 0x51, 0xa1, 0x99, 0x81, 0x86,
      0x63, 0x34, 0x18, 0x36, 0x7e, 0x75, 0x76, 0x53, 0x05, 0x66, 0xc2, 0x3d,
      0x6d, 0x4e, 0x0d, 0xa9, 0xb0, 0x38, 0xd0, 0xbb, 0x51, 0x69, 0xce, 0x40,
      0x13, 0x3e, 0xa0, 0x76, 0x47, 0x2d, 0x05, 0x50, 0x01, 0xf0, 0x13, 0x56,
      0x45, 0x94, 0x0f, 0xd0, 0x8e, 0xa4, 0x42, 0x69, 0xaf, 0x26, 0x04, 0xc8,
      0xb1, 0xba, 0x22, 0x50, 0x53, 0xd6, 0xdb, 0x9a, 0xb4, 0x35, 0x77, 0x68,
      0x94, 0x01, 0xbd, 0xc0, 0xf3},
     {},
     false,
     true}};

#endif  // rsaencrypt_bb2048_vectors_h__
