/* -*- Mode: IDL; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this file,
 * You can obtain one at http://mozilla.org/MPL/2.0/.
 */

interface nsIContentChild;

/**
 * An optimized QueryInterface method, generated by generateQI.
 *
 * For JS callers, this behaves like a normal QueryInterface function. When
 * called with a supported interface, it returns its `this` object. When
 * called with an unsupported interface, it throws NS_ERROR_NO_INTERFACE.
 *
 * C++ callers use a fast path, and never call the JSAPI or WebIDL methods of
 * this object.
 */
[ChromeOnly, Exposed=Window]
interface MozQueryInterface {
  [Throws]
  legacycaller any (any aIID);
};

/**
 * A collection of static utility methods that are only exposed to system code.
 * This is exposed in all the system globals where we can expose stuff by
 * default, so should only include methods that are **thread-safe**.
 */
[ChromeOnly, Exposed=(Window,Worker)]
namespace ChromeUtils {
  /**
   * Get the |NodeId| for the given JS Object.
   * |NodeId| is the identifier of |JS::ubi::Node|.
   */
  NodeId getObjectNodeId(object obj);

  /**
   * Serialize a snapshot of the heap graph, as seen by |JS::ubi::Node| and
   * restricted by |boundaries|, and write it to the provided file path.
   *
   * @param boundaries        The portion of the heap graph to write.
   *
   * @returns                 The path to the file the heap snapshot was written
   *                          to. This is guaranteed to be within the temp
   *                          directory and its file name will match the regexp
   *                          `\d+(\-\d+)?\.fxsnapshot`.
   */
  [Throws]
  DOMString saveHeapSnapshot(optional HeapSnapshotBoundaries boundaries = {});

  /**
   * This is the same as saveHeapSnapshot, but with a different return value.
   *
   * @returns                 The snapshot ID of the file. This is the file name
   *                          without the temp directory or the trailing
   *                          `.fxsnapshot`.
   */
  [Throws]
  DOMString saveHeapSnapshotGetId(optional HeapSnapshotBoundaries boundaries = {});

  /**
   * Deserialize a core dump into a HeapSnapshot.
   *
   * @param filePath          The file path to read the heap snapshot from.
   */
  [Throws, NewObject]
  HeapSnapshot readHeapSnapshot(DOMString filePath);

  /**
   * Return the keys in a weak map.  This operation is
   * non-deterministic because it is affected by the scheduling of the
   * garbage collector and the cycle collector.
   *
   * @param aMap weak map or other JavaScript value
   * @returns If aMap is a weak map object, return the keys of the weak
   *          map as an array.  Otherwise, return undefined.
   */
  [Throws, NewObject]
  any nondeterministicGetWeakMapKeys(any map);

  /**
   * Return the keys in a weak set.  This operation is
   * non-deterministic because it is affected by the scheduling of the
   * garbage collector and the cycle collector.
   *
   * @param aSet weak set or other JavaScript value
   * @returns If aSet is a weak set object, return the keys of the weak
   *          set as an array.  Otherwise, return undefined.
   */
  [Throws, NewObject]
  any nondeterministicGetWeakSetKeys(any aSet);

  /**
   * Converts a buffer to a Base64 URL-encoded string per RFC 4648.
   *
   * @param source The buffer to encode.
   * @param options Additional encoding options.
   * @returns The encoded string.
   */
  [Throws]
  ByteString base64URLEncode(BufferSource source,
                             Base64URLEncodeOptions options);

  /**
   * Decodes a Base64 URL-encoded string per RFC 4648.
   *
   * @param string The string to decode.
   * @param options Additional decoding options.
   * @returns The decoded buffer.
   */
  [Throws, NewObject]
  ArrayBuffer base64URLDecode(ByteString string,
                              Base64URLDecodeOptions options);

  /**
   * Cause the current process to fatally crash unless the given condition is
   * true. This is similar to MOZ_RELEASE_ASSERT in C++ code.
   *
   * WARNING: This message is included publicly in the crash report, and must
   * not contain private information.
   *
   * Crash report will be augmented with the current JS stack information.
   */
  void releaseAssert(boolean condition,
                     optional DOMString message = "<no message>");

#ifdef NIGHTLY_BUILD

  /**
   * If the chrome code has thrown a JavaScript Dev Error
   * in the current JSRuntime. the first such error, or `undefined`
   * otherwise.
   *
   * A JavaScript Dev Error is an exception thrown by JavaScript
   * code that matches both conditions:
   * - it was thrown by chrome code;
   * - it is either a `ReferenceError`, a `TypeError` or a `SyntaxError`.
   *
   * Such errors are stored regardless of whether they have been
   * caught.
   *
   * This mechanism is designed to help ensure that the code of
   * Firefox is free from Dev Errors, even if they are accidentally
   * caught by clients.
   *
   * The object returned is not an exception. It has fields:
   * - DOMString stack
   * - DOMString filename
   * - DOMString lineNumber
   * - DOMString message
   */
  [Throws]
  readonly attribute any recentJSDevError;

  /**
   * Reset `recentJSDevError` to `undefined` for the current JSRuntime.
   */
  void clearRecentJSDevError();
#endif // NIGHTLY_BUILD

  /**
   * If the profiler is currently running and recording the current thread,
   * add a marker for the current thread. No-op otherwise.
   *
   * @param name              The name of the marker.
   * @param startTime         The timestamp to use as the start of the marker.
   *                          If omitted, the marker will have no duration.
   *                          In window and ChromeWorker contexts, use a
   *                          timestamp from `performance.now()`.
   *                          In JS modules, use `Cu.now()` to get a timestamp.
   * @param text              Text to associate with the marker.
   */
  void addProfilerMarker(UTF8String name,
                         optional DOMHighResTimeStamp startTime,
                         optional UTF8String text);

  /**
   * IF YOU ADD NEW METHODS HERE, MAKE SURE THEY ARE THREAD-SAFE.
   */
};

/**
 * Additional ChromeUtils methods that are _not_ thread-safe, and hence not
 * exposed in workers.
 */
[Exposed=Window]
partial namespace ChromeUtils {
  /**
   * A helper that converts OriginAttributesDictionary to a opaque suffix string.
   *
   * @param originAttrs       The originAttributes from the caller.
   */
  ByteString
  originAttributesToSuffix(optional OriginAttributesDictionary originAttrs = {});

  /**
   * Returns true if the members of |originAttrs| match the provided members
   * of |pattern|.
   *
   * @param originAttrs       The originAttributes under consideration.
   * @param pattern           The pattern to use for matching.
   */
  boolean
  originAttributesMatchPattern(optional OriginAttributesDictionary originAttrs = {},
                               optional OriginAttributesPatternDictionary pattern = {});

  /**
   * Returns an OriginAttributesDictionary with values from the |origin| suffix
   * and unspecified attributes added and assigned default values.
   *
   * @param origin            The origin URI to create from.
   * @returns                 An OriginAttributesDictionary with values from
   *                          the origin suffix and unspecified attributes
   *                          added and assigned default values.
   */
  [Throws]
  OriginAttributesDictionary
  createOriginAttributesFromOrigin(DOMString origin);

  /**
   * Returns an OriginAttributesDictionary that is a copy of |originAttrs| with
   * unspecified attributes added and assigned default values.
   *
   * @param originAttrs       The origin attributes to copy.
   * @returns                 An OriginAttributesDictionary copy of |originAttrs|
   *                          with unspecified attributes added and assigned
   *                          default values.
   */
  OriginAttributesDictionary
  fillNonDefaultOriginAttributes(optional OriginAttributesDictionary originAttrs = {});

  /**
   * Returns true if the 2 OriginAttributes are equal.
   */
  boolean
  isOriginAttributesEqual(optional OriginAttributesDictionary aA = {},
                          optional OriginAttributesDictionary aB = {});

  /**
   * Loads and compiles the script at the given URL and returns an object
   * which may be used to execute it repeatedly, in different globals, without
   * re-parsing.
   */
  [NewObject]
  Promise<PrecompiledScript>
  compileScript(DOMString url, optional CompileScriptOptionsDictionary options = {});

  /**
   * Returns an optimized QueryInterface method which, when called from
   * JavaScript, acts as an ordinary QueryInterface function call, and when
   * called from XPConnect, circumvents JSAPI entirely.
   *
   * The list of interfaces may include a mix of JS ID objects and interface
   * name strings.
   *
   * nsISupports is implicitly supported, and must not be included in the
   * interface list.
   */
  [Affects=Nothing, NewObject, Throws]
  MozQueryInterface generateQI(sequence<any> interfaces);

  /**
   * Waive Xray on a given value. Identity op for primitives.
   */
  [Throws]
  any waiveXrays(any val);

  /**
   * Strip off Xray waivers on a given value. Identity op for primitives.
   */
  [Throws]
  any unwaiveXrays(any val);

  /**
   * Gets the name of the JSClass of the object.
   *
   * if |unwrap| is true, all wrappers are unwrapped first. Unless you're
   * specifically trying to detect whether the object is a proxy, this is
   * probably what you want.
   */
  DOMString getClassName(object obj, optional boolean unwrap = true);

  /**
   * Clones the properties of the given object into a new object in the given
   * target compartment (or the caller compartment if no target is provided).
   * Property values themeselves are not cloned.
   *
   * Ignores non-enumerable properties, properties on prototypes, and properties
   * with getters or setters.
   */
  [Throws]
  object shallowClone(object obj, optional object? target = null);

  /**
   * Dispatches the given callback to the main thread when it would be
   * otherwise idle. Similar to Window.requestIdleCallback, but not bound to a
   * particular DOM windw.
   */
  [Throws]
  void idleDispatch(IdleRequestCallback callback,
                    optional IdleRequestOptions options = {});

  /**
   * Synchronously loads and evaluates the js file located at
   * 'aResourceURI' with a new, fully privileged global object.
   *
   * If `aTargetObj` is specified, and non-null, all properties exported by
   * the module are copied to that object.
   *
   * If `aTargetObj` is not specified, or is non-null, an object is returned
   * containing all of the module's exported properties. The same object is
   * returned for every call.
   *
   * If `aTargetObj` is specified and null, the module's global object is
   * returned, rather than its explicit exports. This behavior is deprecated,
   * and will removed in the near future, since it is incompatible with the
   * ES6 module semanitcs we intend to migrate to. It should not be used in
   * new code.
   *
   * @param aResourceURI A resource:// URI string to load the module from.
   * @param aTargetObj the object to install the exported properties on or null.
   * @returns the module code's global object.
   *
   * The implementation maintains a hash of aResourceURI->global obj.
   * Subsequent invocations of import with 'aResourceURI' pointing to
   * the same file will not cause the module to be re-evaluated, but
   * the symbols in EXPORTED_SYMBOLS will be exported into the
   * specified target object and the global object returned as above.
   */
  [Throws]
  object import(DOMString aResourceURI, optional object? aTargetObj);

  /**
   * Defines a property on the given target which lazily imports a JavaScript
   * module when accessed.
   *
   * The first time the property is accessed, the module at the given URL is
   * imported, and the property is replaced with the module's exported symbol
   * of the same name.
   *
   * Some points to note when using this utility:
   *
   * - The cached module export is always stored on the `this` object that was
   *   used to access the getter. This means that if you define the lazy
   *   getter on a prototype, the module will be re-imported every time the
   *   property is accessed on a new instance.
   *
   * - The getter property may be overwritten by simple assignment, but as
   *   with imports, the new property value is always defined on the `this`
   *   object that the setter is called with.
   *
   * - If the module import fails, the getter will throw an error, and the
   *   property will not be replaced. Each subsequent attempt to access the
   *   getter will attempt to re-import the object, which will likely continue
   *   to result in errors.
   *
   * @param target The target object on which to define the property.
   * @param id The name of the property to define, and of the symbol to
   *           import.
   * @param resourceURI The resource URI of the module, as passed to
   *                    ChromeUtils.import.
   */
  [Throws]
  void defineModuleGetter(object target, DOMString id, DOMString resourceURI);

  /**
   * Returns the scripted location of the first ancestor stack frame with a
   * principal which is subsumed by the given principal. If no such frame
   * exists on the call stack, returns null.
   */
  object? getCallerLocation(Principal principal);

  /**
   * Creates a JS Error object with the given message and stack.
   *
   * If a stack object is provided, the error object is created in the global
   * that it belongs to.
   */
  [Throws]
  object createError(DOMString message, optional object? stack = null);

  /**
   * Request performance metrics to the current process & all content processes.
   */
  [Throws]
  Promise<sequence<PerformanceInfoDictionary>> requestPerformanceMetrics();

  /**
   * Set the collection of specific detailed performance timing information.
   * Selecting 0 for the mask will end existing collection. All metrics that
   * are chosen will be cleared after updating the mask.
   *
   * @param aCollectionMask A bitmask where each bit corresponds to a metric
   *        to be collected as listed in PerfStats::Metric.
   */
  void setPerfStatsCollectionMask(unsigned long long aCollectionMask);

  /**
   * Collect results of detailed performance timing information.
   * The output is a JSON string containing performance timings.
   */
  [Throws]
  Promise<DOMString> collectPerfStats();

  /**
  * Returns a Promise containing a sequence of I/O activities
  */
  [Throws]
  Promise<sequence<IOActivityDataDictionary>> requestIOActivity();

  /**
  * Returns a Promise containing all processes info
  */
  [Throws]
  Promise<ParentProcInfoDictionary> requestProcInfo();

  [ChromeOnly, Throws]
  boolean hasReportingHeaderForOrigin(DOMString aOrigin);

  [ChromeOnly]
  PopupBlockerState getPopupControlState();

  [ChromeOnly]
  boolean isPopupTokenUnused();

  /**
   * Milliseconds from the last iframe loading an external protocol.
   */
  [ChromeOnly]
  double lastExternalProtocolIframeAllowed();

  /**
   * For testing purpose we need to reset this value.
   */
  [ChromeOnly]
  void resetLastExternalProtocolIframeAllowed();

  /**
   * Register a new toplevel window global actor. This method may only be
   * called in the parent process. |name| must be globally unique.
   *
   * See JSWindowActor.webidl for WindowActorOptions fields documentation.
   */
  [ChromeOnly, Throws]
  void registerWindowActor(UTF8String aName, optional WindowActorOptions aOptions = {});

  [ChromeOnly]
  void unregisterWindowActor(UTF8String aName);

  /**
   * Register a new toplevel content global actor. This method may only be
   * called in the parent process. |name| must be globally unique.
   *
   * See JSProcessActor.webidl for ProcessActorOptions fields documentation.
   */
  [ChromeOnly, Throws]
  void registerProcessActor(UTF8String aName, optional ProcessActorOptions aOptions = {});

  [ChromeOnly]
  void unregisterProcessActor(UTF8String aName);

  [ChromeOnly]
  // aError should a nsresult.
  boolean isClassifierBlockingErrorCode(unsigned long aError);

  /**
   * If leak detection is enabled, print a note to the leak log that this
   * process will intentionally crash. This should be called only on child
   * processes for testing purpose.
   */
  [ChromeOnly, Throws]
  void privateNoteIntentionalCrash();

  // This is used to generate fake media control keys event in testing.
  [ChromeOnly]
  void generateMediaControlKeysTestEvent(MediaControlKeysTestEvent aEvent);

  /**
   * ContentChild of the current process.
   */
  [ChromeOnly]
  readonly attribute nsIContentChild? contentChild;

  // This is used to get the media metadata from the current main controller in
  // testing.
  [ChromeOnly]
  MediaMetadataInit getCurrentActiveMediaMetadata();

  // This is used to get the actual media playback state from the current main
  // controller in testing.
  [ChromeOnly]
  MediaSessionPlaybackTestState getCurrentMediaSessionPlaybackState();
};

/*
 * This type is a WebIDL representation of mozilla::ProcType.
 */
enum WebIDLProcType {
 "web",
 "webIsolated",
 "file",
 "extension",
 "privilegedabout",
 "privilegedmozilla",
 "webLargeAllocation",
 "withCoopCoep",
 "browser",
 "plugin",
 "ipdlUnitTest",
 "gmpPlugin",
 "gpu",
 "vr",
 "rdd",
 "socket",
 "remoteSandboxBroker",
#ifdef MOZ_ENABLE_FORKSERVER
 "forkServer",
#endif
 "unknown",
};

/**
 * These dictionaries hold information about Firefox running processes and
 * threads.
 *
 * See widget/ProcInfo.h for fields documentation.
 */
dictionary ThreadInfoDictionary {
  long long tid = 0;
  DOMString name = "";
  unsigned long long cpuUser = 0;
  unsigned long long cpuKernel = 0;
};

dictionary ChildProcInfoDictionary {
  // System info
  long long pid = 0;
  DOMString filename = "";
  unsigned long long virtualMemorySize = 0;
  long long residentSetSize = 0;
  unsigned long long cpuUser = 0;
  unsigned long long cpuKernel = 0;
  sequence<ThreadInfoDictionary> threads = [];
  // Firefox info
  unsigned long long ChildID = 0;
  DOMString origin = "";
  WebIDLProcType type = "web";
};

dictionary ParentProcInfoDictionary {
  // System info
  long long pid = 0;
  DOMString filename = "";
  unsigned long long virtualMemorySize = 0;
  long long residentSetSize = 0;
  unsigned long long cpuUser = 0;
  unsigned long long cpuKernel = 0;
  sequence<ThreadInfoDictionary> threads = [];
  sequence<ChildProcInfoDictionary> children = [];
  // Firefox info
  WebIDLProcType type = "browser";
};

/**
 * Dictionaries duplicating IPDL types in dom/ipc/DOMTypes.ipdlh
 * Used by requestPerformanceMetrics
 */
dictionary MediaMemoryInfoDictionary {
  unsigned long long audioSize = 0;
  unsigned long long videoSize = 0;
  unsigned long long resourcesSize = 0;
};

dictionary MemoryInfoDictionary {
  unsigned long long domDom = 0;
  unsigned long long domStyle = 0;
  unsigned long long domOther = 0;
  unsigned long long GCHeapUsage = 0;
  required MediaMemoryInfoDictionary media;
};

dictionary CategoryDispatchDictionary
{
  unsigned short category = 0;
  unsigned short count = 0;
};

dictionary PerformanceInfoDictionary {
  ByteString host = "";
  unsigned long pid = 0;
  unsigned long long windowId = 0;
  unsigned long long duration = 0;
  unsigned long long counterId = 0;
  boolean isWorker = false;
  boolean isTopLevel = false;
  required MemoryInfoDictionary memoryInfo;
  sequence<CategoryDispatchDictionary> items = [];
};

/**
 * Used by requestIOActivity() to return the number of bytes
 * that were read (rx) and/or written (tx) for a given location.
 *
 * Locations can be sockets or files.
 */
dictionary IOActivityDataDictionary {
  ByteString location = "";
  unsigned long long rx = 0;
  unsigned long long tx = 0;
};

/**
 * Used by principals and the script security manager to represent origin
 * attributes. The first dictionary is designed to contain the full set of
 * OriginAttributes, the second is used for pattern-matching (i.e. does this
 * OriginAttributesDictionary match the non-empty attributes in this pattern).
 *
 * IMPORTANT: If you add any members here, you need to do the following:
 * (1) Add them to both dictionaries.
 * (2) Update the methods on mozilla::OriginAttributes, including equality,
 *     serialization, deserialization, and inheritance.
 * (3) Update the methods on mozilla::OriginAttributesPattern, including matching.
 */
[GenerateInitFromJSON]
dictionary OriginAttributesDictionary {
  unsigned long userContextId = 0;
  boolean inIsolatedMozBrowser = false;
  unsigned long privateBrowsingId = 0;
  DOMString firstPartyDomain = "";
  DOMString geckoViewSessionContextId = "";
};

[GenerateInitFromJSON, GenerateToJSON]
dictionary OriginAttributesPatternDictionary {
  unsigned long userContextId;
  boolean inIsolatedMozBrowser;
  unsigned long privateBrowsingId;
  DOMString firstPartyDomain;
  DOMString geckoViewSessionContextId;
};

dictionary CompileScriptOptionsDictionary {
  /**
   * The character set from which to decode the script.
   */
  DOMString charset = "utf-8";

  /**
   * If true, certain parts of the script may be parsed lazily, the first time
   * they are used, rather than eagerly parsed at load time.
   */
  boolean lazilyParse = false;

  /**
   * If true, the script will be compiled so that its last expression will be
   * returned as the value of its execution. This makes certain types of
   * optimization impossible, and disables the JIT in many circumstances, so
   * should not be used when not absolutely necessary.
   */
  boolean hasReturnValue = false;
};

/**
 * A JS object whose properties specify what portion of the heap graph to
 * write. The recognized properties are:
 *
 * * globals: [ global, ... ]
 *   Dump only nodes that either:
 *   - belong in the compartment of one of the given globals;
 *   - belong to no compartment, but do belong to a Zone that contains one of
 *     the given globals;
 *   - are referred to directly by one of the last two kinds of nodes; or
 *   - is the fictional root node, described below.
 *
 * * debugger: Debugger object
 *   Like "globals", but use the Debugger's debuggees as the globals.
 *
 * * runtime: true
 *   Dump the entire heap graph, starting with the JSRuntime's roots.
 *
 * One, and only one, of these properties must exist on the boundaries object.
 *
 * The root of the dumped graph is a fictional node whose ubi::Node type name is
 * "CoreDumpRoot". If we are dumping the entire ubi::Node graph, this root node
 * has an edge for each of the JSRuntime's roots. If we are dumping a selected
 * set of globals, the root has an edge to each global, and an edge for each
 * incoming JS reference to the selected Zones.
 */
dictionary HeapSnapshotBoundaries {
  sequence<object> globals;
  object           debugger;
  boolean          runtime;
};

dictionary Base64URLEncodeOptions {
  /** Specifies whether the output should be padded with "=" characters. */
  required boolean pad;
};

enum Base64URLDecodePadding {
  /**
   * Fails decoding if the input is unpadded. RFC 4648, section 3.2 requires
   * padding, unless the referring specification prohibits it.
   */
  "require",

  /** Tolerates padded and unpadded input. */
  "ignore",

  /**
   * Fails decoding if the input is padded. This follows the strict base64url
   * variant used in JWS (RFC 7515, Appendix C) and HTTP Encrypted
   * Content-Encoding (draft-ietf-httpbis-encryption-encoding-01).
   */
  "reject"
};

dictionary Base64URLDecodeOptions {
  /** Specifies the padding mode for decoding the input. */
  required Base64URLDecodePadding padding;
};

// Keep this in sync with PopupBlocker::PopupControlState!
enum PopupBlockerState {
  "openAllowed",
  "openControlled",
  "openBlocked",
  "openAbused",
  "openOverridden",
};

// Keep this in sync with MediaControlKeysEvent in MediaControlKeysEvent.h!
enum MediaControlKeysTestEvent {
  "focus",
  "play",
  "pause",
  "playPause",
  "previoustrack",
  "nexttrack",
  "seekbackward",
  "seekforward",
  "stop",
};

// Keep this in sync with MediaSessionPlaybackState in MediaSession.webidl!
// The reason we create an new enum MediaSessionPlaybackTestState is because of
// building issue. If we use MediaSessionPlaybackState directly, then the
// codegen would automatically add an header include of MediaSessionBinding.h
// in the ChromeUtilsBinding.h, which would cause the complier complaining about
// lots of undeclared identifier. In addition, there is an universal definition
// for 'None' on X11, and it conflicts with the 'None' in the enum of
// MediaSessionPlaybackState, which causes a build fail. As this state is
// defined in the ChromeUtilsBinding.h, which is automatically generated by the
// codegen, we can't include 'X11UndefineNone.h' to workaround this problem, so
// we have to change 'None' to 'Stopped'.
enum MediaSessionPlaybackTestState {
  "stopped",
  "paused",
  "playing",
};
